"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mode = exports.StaticSiteAuthorization = exports.SpaAuthorization = exports.Authorization = void 0;
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const core_1 = require("@aws-cdk/core");
const cdk_lambda_at_edge_pattern_1 = require("@cloudcomponents/cdk-lambda-at-edge-pattern");
const auth_flow_1 = require("./auth-flow");
const retrieve_user_pool_client_secret_1 = require("./retrieve-user-pool-client-secret");
const secret_generator_1 = require("./secret-generator");
const user_pool_client_redirects_1 = require("./user-pool-client-redirects");
const user_pool_domain_1 = require("./user-pool-domain");
class Authorization extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.userPool = props.userPool;
        this.redirectPaths = (_a = props.redirectPaths) !== null && _a !== void 0 ? _a : {
            signIn: '/parseauth',
            authRefresh: '/refreshauth',
            signOut: '/',
        };
        this.signOutUrlPath = (_b = props.signOutUrl) !== null && _b !== void 0 ? _b : '/signout';
        this.httpHeaders = (_c = props.httpHeaders) !== null && _c !== void 0 ? _c : {
            'Content-Security-Policy': "default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self' 'unsafe-inline'; object-src 'none'; connect-src 'self'",
            'Strict-Transport-Security': 'max-age=31536000; includeSubdomains; preload',
            'Referrer-Policy': 'same-origin',
            'X-XSS-Protection': '1; mode=block',
            'X-Frame-Options': 'DENY',
            'X-Content-Type-Options': 'nosniff',
            'Cache-Control': 'no-cache',
        };
        this.oauthScopes = (_d = props.oauthScopes) !== null && _d !== void 0 ? _d : [aws_cognito_1.OAuthScope.PHONE, aws_cognito_1.OAuthScope.EMAIL, aws_cognito_1.OAuthScope.PROFILE, aws_cognito_1.OAuthScope.OPENID, aws_cognito_1.OAuthScope.COGNITO_ADMIN];
        this.cookieSettings = props.cookieSettings;
        this.userPoolClient = this.createUserPoolClient();
        this.nonceSigningSecret = this.generateNonceSigningSecret();
        this.cognitoAuthDomain = this.retrieveCognitoAuthDomain();
        this.authFlow = this.createAuthFlow((_e = props.logLevel) !== null && _e !== void 0 ? _e : cdk_lambda_at_edge_pattern_1.LogLevel.WARN);
    }
    updateUserPoolClientCallbacks(redirects) {
        const { callbackUrls, logoutUrls } = redirects;
        new user_pool_client_redirects_1.UserPoolClientRedirects(this, 'UserPoolClientRedirects', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            callbackUrls,
            logoutUrls,
        });
    }
    createDefaultBehavior(origin) {
        return {
            origin,
            forwardQueryString: true,
            edgeLambdas: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
        };
    }
    createLegacyDefaultBehavior() {
        return {
            isDefaultBehavior: true,
            lambdaFunctionAssociations: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
        };
    }
    createAdditionalBehaviors(origin) {
        return {
            [this.redirectPaths.signIn]: {
                origin,
                forwardQueryString: true,
                edgeLambdas: [this.authFlow.parseAuth],
            },
            [this.redirectPaths.authRefresh]: {
                origin,
                forwardQueryString: true,
                edgeLambdas: [this.authFlow.refreshAuth],
            },
            [this.signOutUrlPath]: {
                origin,
                forwardQueryString: true,
                edgeLambdas: [this.authFlow.signOut],
            },
        };
    }
    createLegacyAdditionalBehaviors() {
        return [
            {
                pathPattern: this.redirectPaths.signIn,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.parseAuth],
            },
            {
                pathPattern: this.redirectPaths.authRefresh,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.refreshAuth],
            },
            {
                pathPattern: this.signOutUrlPath,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.signOut],
            },
        ];
    }
    generateNonceSigningSecret() {
        const { secret } = new secret_generator_1.SecretGenerator(this, 'SecretGenerator');
        return secret;
    }
    retrieveCognitoAuthDomain() {
        const userPoolDomain = new user_pool_domain_1.UserPoolDomain(this, 'UserPoolDomain', {
            userPool: this.userPool,
        });
        return userPoolDomain.cognitoAuthDomain;
    }
}
exports.Authorization = Authorization;
class SpaAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.SPA;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: false,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO],
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; SameSite=Lax',
                accessToken: 'Path=/; Secure; SameSite=Lax',
                refreshToken: 'Path=/; Secure; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
}
exports.SpaAuthorization = SpaAuthorization;
class StaticSiteAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.STATIC_SITE;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: true,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO],
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        const clientSecret = this.retrieveUserPoolClientSecret();
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            clientSecret,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                accessToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                refreshToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
    retrieveUserPoolClientSecret() {
        const { clientSecret } = new retrieve_user_pool_client_secret_1.RetrieveUserPoolClientSecret(this, 'RetrieveUserPoolClientSecret', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
        });
        return clientSecret;
    }
}
exports.StaticSiteAuthorization = StaticSiteAuthorization;
var Mode;
(function (Mode) {
    Mode["SPA"] = "SPA";
    Mode["STATIC_SITE"] = "STATIC_SITE";
})(Mode = exports.Mode || (exports.Mode = {}));
//# sourceMappingURL=data:application/json;base64,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