import unittest
import string
import random

from noscrypto import Client, Server, Utils

def getPseudoRandomString(length=20):
    return "".join([random.choice(string.printable) for i in range(length)])

class TestClientLogin(unittest.TestCase):
    def test_auto_endline(self):
        self.assertEqual(Client.LoginEncrypt(b"Test"), Client.LoginEncrypt(b"Test\n"))
        
    def test_real_loginpacket_encryption(self):
        self.assertEqual(
        Client.LoginEncrypt("NoS0575 571739 admin C7AD44CBAD762A5DA0A452F9E854FDC1E0E7A52A38015F23F3EAB1D80B931DD472634DFAC71CD34EBC35D16AB7FB8A90C81F975113D6C7538DC69DD8DE9077EC 6c38fd45-ef91-4054-82b8-9c96b4b0b209 00727371 0\x0B0.9.3.3134 0 9825E5FC051F9A17AEFE047A48F4C6F9\n".encode("ascii")),
        b"\x9C\xBB\x9F\x02\x05\x03\x05\xF2\x05\x03\x01\x03\xFF\x09\xF2\xB1\xB6\xBD\xB9\xBC\xF2\x8F\x03\x91\x96\x06\x06\x8F\x90\x91\x96\x03\x04\x00\x91\x05\x96\x91\x02\x91\x06\x05\x00\x94\x09\x95\x0A\x05\x06\x94\x96\x8F\x01\x95\x02\x95\x03\x91\x05\x00\x91\xFF\x0A\x02\x01\x05\x94\x00\xFF\x94\xFF\x95\x91\x90\x01\x96\x0A\x02\x90\x09\xFF\x01\x96\x96\x06\x03\x00\x04\xFF\x06\x96\x94\x91\x8F\x03\x01\x8F\x96\xFF\x06\x95\x90\x8F\xFF\x05\x96\x01\x04\x91\x90\x03\x94\x90\x0A\x91\x09\x02\x8F\x0A\x01\x94\x09\x03\x05\x01\x01\xFF\x96\x04\x8F\x03\x05\xFF\x0A\x96\x8F\x04\x09\x96\x96\x0A\x96\x95\x09\x02\x03\x03\x95\x8F\xF2\x04\xAF\xFF\x0A\xB4\xB6\x06\x05\xFD\xB5\xB4\x09\x01\xFD\x06\x02\x05\x06\xFD\x0A\x00\xB0\x0A\xFD\x09\xAF\x09\x04\xB0\x06\xB0\x02\xB0\x00\x02\x09\xF2\x02\x02\x03\x00\x03\xFF\x03\x01\xF2\x02\xD7\x02\xFC\x09\xFC\xFF\xFC\xFF\x01\xFF\x06\xF2\x02\xF2\x09\x0A\x00\x05\x95\x05\x94\x8F\x02\x05\x01\x94\x09\x91\x01\x03\x91\x95\x94\x95\x02\x06\x03\x91\x06\x0A\x94\x06\x8F\x04\x94\x09\xD8")
        
    def test_real_loginpacket_decryption(self):
        self.assertEqual(Client.LoginDecrypt(b"\x75\x70\x78\x7B\x72\x2F\x44\x19"), b"failc 5\n")
        
        
class TestServerLogin(unittest.TestCase):
    def test_auto_endline(self):
        self.assertEqual(Server.LoginEncrypt(b"Test"), Server.LoginEncrypt(b"Test\n"))
        
    def test_real_loginpacket_decryption(self):
        self.assertEqual(Server.LoginDecrypt(b"\x9C\xBB\x9F\x02\x05\x03\x05\xF2\x05\x03\x01\x03\xFF\x09\xF2\xB1\xB6\xBD\xB9\xBC\xF2\x8F\x03\x91\x96\x06\x06\x8F\x90\x91\x96\x03\x04\x00\x91\x05\x96\x91\x02\x91\x06\x05\x00\x94\x09\x95\x0A\x05\x06\x94\x96\x8F\x01\x95\x02\x95\x03\x91\x05\x00\x91\xFF\x0A\x02\x01\x05\x94\x00\xFF\x94\xFF\x95\x91\x90\x01\x96\x0A\x02\x90\x09\xFF\x01\x96\x96\x06\x03\x00\x04\xFF\x06\x96\x94\x91\x8F\x03\x01\x8F\x96\xFF\x06\x95\x90\x8F\xFF\x05\x96\x01\x04\x91\x90\x03\x94\x90\x0A\x91\x09\x02\x8F\x0A\x01\x94\x09\x03\x05\x01\x01\xFF\x96\x04\x8F\x03\x05\xFF\x0A\x96\x8F\x04\x09\x96\x96\x0A\x96\x95\x09\x02\x03\x03\x95\x8F\xF2\x04\xAF\xFF\x0A\xB4\xB6\x06\x05\xFD\xB5\xB4\x09\x01\xFD\x06\x02\x05\x06\xFD\x0A\x00\xB0\x0A\xFD\x09\xAF\x09\x04\xB0\x06\xB0\x02\xB0\x00\x02\x09\xF2\x02\x02\x03\x00\x03\xFF\x03\x01\xF2\x02\xD7\x02\xFC\x09\xFC\xFF\xFC\xFF\x01\xFF\x06\xF2\x02\xF2\x09\x0A\x00\x05\x95\x05\x94\x8F\x02\x05\x01\x94\x09\x91\x01\x03\x91\x95\x94\x95\x02\x06\x03\x91\x06\x0A\x94\x06\x8F\x04\x94\x09\xD8"),
        "NoS0575 571739 admin C7AD44CBAD762A5DA0A452F9E854FDC1E0E7A52A38015F23F3EAB1D80B931DD472634DFAC71CD34EBC35D16AB7FB8A90C81F975113D6C7538DC69DD8DE9077EC 6c38fd45-ef91-4054-82b8-9c96b4b0b209 00727371 0\x0B0.9.3.3134 0 9825E5FC051F9A17AEFE047A48F4C6F9\n".encode("ascii"))
        
    def test_real_loginpacket_encryption(self):
        self.assertEqual(Server.LoginEncrypt("failc 5\n".encode("ascii")), b"\x75\x70\x78\x7B\x72\x2F\x44\x19")
        
        
class TestClientServerLoginFlow(unittest.TestCase):
    def test_server_send_client_recv(self):
        s = getPseudoRandomString() + "\n"
        self.assertEqual(Client.LoginDecrypt(Server.LoginEncrypt(s.encode("ascii"))), s.encode("ascii"))
        
    def test_client_send_server_recv(self):
        s = getPseudoRandomString() + "\n"
        self.assertEqual(Server.LoginDecrypt(Client.LoginEncrypt(s.encode("ascii"))), s.encode("ascii"))
        
class TestClientPackUnpack(unittest.TestCase):
    def test_client_mask_generator(self):
        packet = "14326 say dfskjda12312ąśąźżźżććżąąąśąąą2137dadaęóąśłżźćń;1122".encode("cp1250")
        generated_mask = "".join([str(int(i)) for i in Utils._GetMask(packet, Client._ENCRYPTION_TABLE)])
        
        self.assertEqual(generated_mask, "1111110001000000011111000000000000000001111000000000000001111")

    def test_server_pack_client_unpack(self):
        self.assertEqual(Utils._Unpack(Utils._Pack("2137 say ąźć123pd".encode("cp1250"), Client._ENCRYPTION_TABLE), Client._DECRYPTION_TABLE), "2137 say ąźć123pd".encode("cp1250"))
        
    def test_real_client_pack(self):
        self.assertEqual(Utils._Pack("17535 walk 20 26 1 11".encode("ascii"), Client._ENCRYPTION_TABLE), b"\x86\x5B\x97\x91\x04\x88\x9E\x93\x94\x8B\x16\x41\x6A\x15\x15\x50\xFF")
        
    def test_real_client_pack2(self):
        self.assertEqual(Utils._Pack("48967 c_blist  0 0 0 0 0 0 0 0 17 185 302 882 942 999 1591 1592 4083 5065 5068 5069 5070 5206 5307 5361 5982 5991".encode("ascii"), Client._ENCRYPTION_TABLE), b"\x86\x8C\xDA\xB1\x07\x9C\xA0\x9D\x93\x96\x8C\x8B\xE4\x11\x41\x41\x41\x41\x41\x41\x41\x41\x5B\x15\xC9\x17\x46\x1C\xC6\x1D\x86\x1D\xDD\x15\x9D\x51\x59\xD6\x18\x4C\x71\x94\xA9\x19\x4A\xC1\x94\xAD\x19\x4B\x41\x96\x4A\x19\x74\xB1\x97\xA5\x19\xDC\x61\x9D\xD5\xFF")
   
    def test_real_client_pack_special_characters(self):
        self.assertEqual(Utils._Pack("14326 say dfskjda12312ąśąźżźżććżąąąśąąą2137dadaęóąśłżźćń;1122".encode("cp1250"), Client._ENCRYPTION_TABLE),
        b"\x86\x58\x76\xA1\x03\x8C\x9E\x86\x81\x10\x07\x9B\x99\x8C\x94\x95\x9B\x9E\x85\x56\x75\x60\x11\x46\x63\x46\x60\x40\x60\x40\x19\x19\x40\x46\x46\x46\x63\x46\x46\x46\x84\x65\x7B\x0E\x9B\x9E\x9B\x9E\x15\x0C\x46\x63\x4C\x40\x60\x19\x0E\xC4\x84\x55\x66\xFF")
        
   
class TestClientWorld(unittest.TestCase):
    def test_real_client_encrypt(self):
        session = 53836
        packet = "17535 walk 20 26 1 11".encode("ascii")
        
        self.assertEqual(Client.WorldEncrypt(packet, session), b"\xFA\xCF\x0B\x05\x78\xFC\x12\x07\x08\xFF\x8A\xB5\xDE\x89\x89\xC4\x73")
        
    def test_real_client_encrypt2(self):
        session = 10685
        packet = "48967 c_blist  0 0 0 0 0 0 0 0 17 185 302 882 942 999 1591 1592 4083 5065 5068 5069 5070 5206 5307 5361 5982 5991".encode("ascii")
        
        self.assertEqual(Client.WorldEncrypt(packet, session), b"\x42\x4C\x16\x6F\xC1\x5C\x60\x5B\x4D\x52\x4C\x45\x24\xCF\x7F\x7F\x7F\x7F\x7F\x7F\x7F\x7F\x95\xD3\x07\xD1\x82\xDC\x02\xDB\x42\xDB\x1B\xD3\x5B\x8F\x97\x12\xD8\x8C\xAF\x54\x67\xD7\x86\xFF\x54\x6B\xD7\x85\x7F\x52\x86\xD7\xB4\x6F\x51\x63\xD7\x1C\x9F\x5B\x13\x39")
        
    def test_real_client_encypt_special_characters(self):
        session = 34353
        packet = "14326 say dfskjda12312ąśąźżźżććżąąąśąąą2137dadaęóąśłżźćń;1122".encode("cp1250")
        
        self.assertEqual(Client.WorldEncrypt(packet, session), b"\xF7\xC9\xE7\x12\x74\xFD\x0F\xF7\xF2\x81\x78\x0C\x0A\xFD\x05\x06\x0C\x0F\xF6\xC7\xE6\xD1\x82\xB7\xD4\xB7\xD1\xB1\xD1\xB1\x8A\x8A\xB1\xB7\xB7\xB7\xD4\xB7\xB7\xB7\xF5\xD6\xEC\x7F\x0C\x0F\x0C\x0F\x86\x7D\xB7\xD4\xBD\xB1\xD1\x8A\x7F\x35\xF5\xC6\xD7\x70")
    
    def test_real_client_decrypt(self):
        self.assertEqual(Client.WorldDecrypt(b"\x04\x8C\x8B\x9E\x8B\x96\x16\x65\x16\x65\x1A\x41\xA4\x14\x15\x46\x8E\xFF"), "stat 221 221 60 60 0 1024\n".encode("ascii"))
        
class TestServerWorld(unittest.TestCase):
    def test_real_server_decrypt(self):
        session = 53836
        packet = b"\xFA\xCF\x0B\x05\x78\xFC\x12\x07\x08\xFF\x8A\xB5\xDE\x89\x89\xC4\x73"
        
        self.assertEqual(Server.WorldDecrypt(packet, session), "17535 walk 20 26 1 11".encode("ascii"))
        
    def test_real_server_encrypt2(self):
        session = 10685
        packet = b"\x42\x4C\x16\x6F\xC1\x5C\x60\x5B\x4D\x52\x4C\x45\x24\xCF\x7F\x7F\x7F\x7F\x7F\x7F\x7F\x7F\x95\xD3\x07\xD1\x82\xDC\x02\xDB\x42\xDB\x1B\xD3\x5B\x8F\x97\x12\xD8\x8C\xAF\x54\x67\xD7\x86\xFF\x54\x6B\xD7\x85\x7F\x52\x86\xD7\xB4\x6F\x51\x63\xD7\x1C\x9F\x5B\x13\x39"
        
        self.assertEqual(Server.WorldDecrypt(packet, session), "48967 c_blist  0 0 0 0 0 0 0 0 17 185 302 882 942 999 1591 1592 4083 5065 5068 5069 5070 5206 5307 5361 5982 5991".encode("ascii"))
        
    def test_real_server_decrypt_special_characters(self):
        session = 34353
        packet = b"\xF7\xC9\xE7\x12\x74\xFD\x0F\xF7\xF2\x81\x78\x0C\x0A\xFD\x05\x06\x0C\x0F\xF6\xC7\xE6\xD1\x82\xB7\xD4\xB7\xD1\xB1\xD1\xB1\x8A\x8A\xB1\xB7\xB7\xB7\xD4\xB7\xB7\xB7\xF5\xD6\xEC\x7F\x0C\x0F\x0C\x0F\x86\x7D\xB7\xD4\xBD\xB1\xD1\x8A\x7F\x35\xF5\xC6\xD7\x70"
        
        self.assertEqual(Server.WorldDecrypt(packet, session), "14326 say dfskjda12312ąśąźżźżććżąąąśąąą2137dadaęóąśłżźćń;1122".encode("cp1250"))
        
    def test_real_server_encrypt(self):
        self.assertEqual(Server.WorldEncrypt("stat 221 221 60 60 0 1024\n".encode("ascii")), b"\x04\x8C\x8B\x9E\x8B\x96\x16\x65\x16\x65\x1A\x41\xA4\x14\x15\x46\x8E\xFF")
        
if __name__ == "__main__":
    unittest.main()