import os
import shutil
import sys
import unittest
from unittest import mock

import click
from test.cwl_classes import Tool


class CWLTestCase(unittest.TestCase):
    maxDiff = None
    test_dir = "test_dir/"


    def setUp(self):
        if os.path.exists(self.test_dir):
            shutil.rmtree(self.test_dir)
        os.mkdir(self.test_dir)

    def tearDown(self):
        shutil.rmtree(self.test_dir)

    def standard_testargs(self, parser_name):
        return [parser_name, "--generate_cwl_tool", "-d", self.test_dir]

    def open_tool(self, parser_name):
        return Tool(self.test_dir + parser_name.replace('.py', '.cwl').strip('./'))

    def generate_and_open_test_tool(self, testargs, function, parser_name):
        with mock.patch.object(sys, 'argv', testargs):
            function()
        return self.open_tool(parser_name)

    @staticmethod
    def prepare_argument_parser():
        @click.command()
        @click.argument('keyword', type=click.STRING)
        @click.option('--choices', type=click.Choice(['md5', 'sha1']))
        @click.option('--double-type', type=(str, int), default=(None, None))
        def function():
            pass

        return function

    def get_simple_tool(self, parser_name, testargs=None, add_help=True):
        parser = self.prepare_argument_parser()
        if not testargs:
            testargs = self.standard_testargs(parser_name)
        return parser, self.generate_and_open_test_tool(testargs, parser, parser_name)

    def test_subparser(self):
        """
        Tests if tools with subparsers run without errors
        """
        @click.group()
        @click.pass_context
        @click.option('--debug', default=False)
        def cli(debug):
            pass

        @cli.command()
        @click.option('-s', '--some-option')
        def sync(some_option):
            pass

        @cli.command()
        @click.option('-o', '--some-other-option')
        def async(some_option):
            pass

        parser_name = 'test-group.py'
        with mock.patch.object(sys, 'argv', self.standard_testargs(parser_name)):
            cli()
        sync_tool = self.open_tool('test-group-sync.cwl')
        async_tool = self.open_tool('test-group-async.cwl')
        self.assertEqual(sync_tool.inputs['some_option'].input_binding.prefix, '--some-option')
        self.assertEqual(async_tool.inputs['some_other_option'].input_binding.prefix, '--some-other-option')

    def test_basecommand(self):
        """
        if `basecommand` is provided in a program which is run with shebang,
        it replaces autogenerated shebang basecommand
        """
        parser_name = './test-shebang.py'
        file_name = parser_name.strip('./')
        testargs = [parser_name, "--generate_cwl_tool", "-d", self.test_dir, "-b", "python3"]
        parser, tool = self.get_simple_tool(file_name, testargs)
        self.assertEqual(['python3'], tool.basecommand)
        self.assertEqual(file_name, tool.inputs[file_name].id)

    def test_choices(self):
        choices = ['rock', 'scissors', 'paper']

        @click.command()
        @click.option('--item', type=click.Choice(choices))
        def function(item):
            pass

        parser_name = 'test-choices.py'
        tool = self.generate_and_open_test_tool(self.standard_testargs(parser_name), function, parser_name)
        self.assertEqual(tool.inputs['item'].type[1]['symbols'], choices)

    def test_tuples(self):
        @click.command()
        @click.option('--item', type=(str, int), default=(None, None))
        def function(item):
            pass

        parser_name = 'test-type.py'
        tool = self.generate_and_open_test_tool(self.standard_testargs(parser_name), function, parser_name)
        self.assertEqual(tool.inputs['item'].type, ['null', 'string', 'int'])

    def test_flags(self):
        @click.command()
        @click.option('--shout/--no-shout', default=False)
        @click.option('--simple-shout', is_flag=True)
        def function(shout):
            pass

        parser_name = 'test-flags.py'
        tool = self.generate_and_open_test_tool(self.standard_testargs(parser_name), function, parser_name)
        self.assertEqual(tool.inputs['shout'].type, ['null', 'boolean'])
        self.assertEqual(tool.inputs['simple_shout'].type, ['null', 'boolean'])
        self.assertEqual(tool.inputs['shout'].default, False)
        self.assertEqual(tool.inputs['simple_shout'].default, False)

    def test_prefix_chars(self):
        @click.command()
        @click.option('+w/-w')
        def function(w):
            pass

        parser_name = 'test-prefix-chars.py'
        tool = self.generate_and_open_test_tool(self.standard_testargs(parser_name), function, parser_name)
        self.assertEqual(tool.inputs['w'].type, ['null', 'boolean'])

    def test_docstring(self):
        @click.command()
        @click.argument('argument')
        def function(w):
            """This script prints hello NAME COUNT times."""
            pass

        parser_name = 'test-docstring.py'
        tool = self.generate_and_open_test_tool(self.standard_testargs(parser_name), function, parser_name)
        self.assertEqual(tool.description.strip('\n'), function.__doc__)

if __name__ == '__main__':
    unittest.main()