/*
 * Decompiled with CFR 0.152.
 */
package ibis.smartsockets.util.ssh;

import com.google.common.net.HostAndPort;
import ibis.smartsockets.util.Forwarder;
import ibis.smartsockets.util.SmartSocketsException;
import ibis.smartsockets.util.ssh.CertificateCredential;
import ibis.smartsockets.util.ssh.CertificateNotFoundException;
import ibis.smartsockets.util.ssh.Credential;
import ibis.smartsockets.util.ssh.CredentialMap;
import ibis.smartsockets.util.ssh.CredentialNotFoundException;
import ibis.smartsockets.util.ssh.DefaultCredential;
import ibis.smartsockets.util.ssh.InvalidCredentialException;
import ibis.smartsockets.util.ssh.InvalidLocationException;
import ibis.smartsockets.util.ssh.PasswordCredential;
import ibis.smartsockets.util.ssh.SSHConnection;
import ibis.smartsockets.util.ssh.UserCredential;
import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.security.KeyPair;
import java.util.ArrayList;
import org.apache.sshd.agent.local.ProxyAgentFactory;
import org.apache.sshd.client.SshClient;
import org.apache.sshd.client.channel.ChannelDirectTcpip;
import org.apache.sshd.client.config.hosts.DefaultConfigFileHostEntryResolver;
import org.apache.sshd.client.future.ConnectFuture;
import org.apache.sshd.client.keyverifier.AcceptAllServerKeyVerifier;
import org.apache.sshd.client.keyverifier.DefaultKnownHostsServerKeyVerifier;
import org.apache.sshd.client.keyverifier.RejectAllServerKeyVerifier;
import org.apache.sshd.client.keyverifier.ServerKeyVerifier;
import org.apache.sshd.client.session.ClientSession;
import org.apache.sshd.common.NamedResource;
import org.apache.sshd.common.config.keys.FilePasswordProvider;
import org.apache.sshd.common.session.SessionContext;
import org.apache.sshd.common.util.io.resource.PathResource;
import org.apache.sshd.common.util.net.SshdSocketAddress;
import org.apache.sshd.common.util.security.SecurityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSHUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(SSHUtil.class);
    private static final String VIA_TAG = " via:";
    public static final int DEFAULT_SSH_PORT = 22;

    public static SshClient createSSHClient() {
        return SSHUtil.createSSHClient(true, true, true, false, false);
    }

    public static SshClient createSSHClient(boolean useKnownHosts, boolean loadSSHConfig, boolean stricHostCheck, boolean useSSHAgent, boolean useAgentForwarding) {
        SshClient client = SshClient.setUpDefaultClient();
        if (useKnownHosts) {
            DefaultKnownHostsServerKeyVerifier tmp;
            if (stricHostCheck) {
                tmp = new DefaultKnownHostsServerKeyVerifier((ServerKeyVerifier)RejectAllServerKeyVerifier.INSTANCE, true);
            } else {
                tmp = new DefaultKnownHostsServerKeyVerifier((ServerKeyVerifier)AcceptAllServerKeyVerifier.INSTANCE, true);
                tmp.setModifiedServerKeyAcceptor((clientSession, remoteAddress, entry, expected, actual) -> true);
            }
            client.setServerKeyVerifier(tmp);
        } else {
            client.setServerKeyVerifier(AcceptAllServerKeyVerifier.INSTANCE);
        }
        if (loadSSHConfig) {
            client.setHostConfigEntryResolver(DefaultConfigFileHostEntryResolver.INSTANCE);
        }
        if (useSSHAgent) {
            client.setAgentFactory(new ProxyAgentFactory());
        }
        if (useAgentForwarding) {
            LOGGER.debug("(UNIMPLEMENTED) Enabling ssh-agent-forwarding");
        }
        client.start();
        return client;
    }

    public static String validateHost(String adaptorName, String host) throws InvalidLocationException {
        if (host == null || host.isEmpty()) {
            throw new InvalidLocationException(adaptorName, "Failed to parse host: " + host);
        }
        return host;
    }

    public static String getHost(String adaptorName, String location) throws InvalidLocationException {
        try {
            return SSHUtil.validateHost(adaptorName, HostAndPort.fromString(location).getHostText().trim());
        }
        catch (Exception e) {
            throw new InvalidLocationException(adaptorName, "Failed to parse location: " + location);
        }
    }

    public static int getPort(String adaptorName, String location) throws InvalidLocationException {
        try {
            return HostAndPort.fromString(location).getPortOrDefault(22);
        }
        catch (Exception e) {
            throw new InvalidLocationException(adaptorName, "Failed to parse location: " + location);
        }
    }

    public static UserCredential extractCredential(SshdSocketAddress location, Credential credential) {
        if (credential instanceof CredentialMap) {
            CredentialMap map = (CredentialMap)credential;
            String key = location.toString();
            if (map.containsCredential(key)) {
                return map.get(key);
            }
            return map.get(location.getHostName());
        }
        return (UserCredential)credential;
    }

    public static UserCredential[] extractCredentials(String adaptorName, SshdSocketAddress[] locations, Credential credentials) throws CredentialNotFoundException {
        UserCredential[] result = new UserCredential[locations.length];
        for (int i = 0; i < locations.length; ++i) {
            result[i] = SSHUtil.extractCredential(locations[i], credentials);
            if (result[i] != null) continue;
            throw new CredentialNotFoundException(adaptorName, "No credential provided for location: " + locations[i]);
        }
        return result;
    }

    public static SshdSocketAddress[] extractLocations(String adaptorName, String location) throws InvalidLocationException {
        if (location == null) {
            throw new IllegalArgumentException("Location may nor be null");
        }
        ArrayList<SshdSocketAddress> result = new ArrayList<SshdSocketAddress>();
        String tmp = location;
        int index = tmp.lastIndexOf(VIA_TAG);
        while (index != -1) {
            if (index == 0) {
                throw new InvalidLocationException(adaptorName, "Could not parse location: " + location);
            }
            result.add(SSHUtil.extractSocketAddress(adaptorName, tmp.substring(index + VIA_TAG.length()).trim()));
            tmp = tmp.substring(0, index);
            index = tmp.lastIndexOf(VIA_TAG);
        }
        if (tmp.isEmpty()) {
            throw new InvalidLocationException(adaptorName, "Could not parse location: " + location);
        }
        result.add(SSHUtil.extractSocketAddress(adaptorName, tmp.trim()));
        return result.toArray(new SshdSocketAddress[result.size()]);
    }

    private static SshdSocketAddress extractSocketAddress(String adaptorName, String location) throws InvalidLocationException {
        URI uri;
        try {
            uri = new URI("sftp://" + location);
        }
        catch (Exception e) {
            throw new InvalidLocationException(adaptorName, "Failed to parse location: " + location, e);
        }
        String host = uri.getHost();
        int port = uri.getPort();
        if (port == -1) {
            port = 22;
        }
        return new SshdSocketAddress(host, port);
    }

    private static ClientSession connectAndAuthenticate(String adaptorName, SshClient client, String host, int port, UserCredential credential, long timeout) throws SmartSocketsException {
        if (host == null) {
            throw new IllegalArgumentException("Target host may not be null");
        }
        String username = credential.getUsername();
        if (username == null) {
            throw new InvalidCredentialException(adaptorName, "Failed to retrieve username from credential");
        }
        ClientSession session = null;
        try {
            session = ((ConnectFuture)client.connect(username, host, port).verify(timeout)).getSession();
        }
        catch (IOException e) {
            throw new SmartSocketsException(adaptorName, "Connection setup to " + host + ":" + port + " failed!", e);
        }
        if (!(credential instanceof DefaultCredential)) {
            PasswordCredential c;
            if (credential instanceof CertificateCredential) {
                c = (CertificateCredential)credential;
                String certfile = ((CertificateCredential)c).getCertificateFile();
                Path path = Paths.get(certfile, new String[0]).toAbsolutePath().normalize();
                if (!path.toFile().exists()) {
                    throw new CertificateNotFoundException(adaptorName, "Certificate not found: " + path);
                }
                KeyPair pair = null;
                try (InputStream inputStream = Files.newInputStream(path, StandardOpenOption.READ);){
                    char[] password = c.getPassword();
                    pair = password.length == 0 ? SecurityUtils.loadKeyPairIdentities(session, new PathResource(path), inputStream, null).iterator().next() : SecurityUtils.loadKeyPairIdentities(session, new PathResource(path), inputStream, new PasswordProvider(password)).iterator().next();
                }
                catch (Exception e) {
                    throw new SmartSocketsException(adaptorName, "Failed to load certificate: " + path, e);
                }
                session.addPublicKeyIdentity(pair);
            } else if (credential instanceof PasswordCredential) {
                c = (PasswordCredential)credential;
                session.addPasswordIdentity(new String(c.getPassword()));
            } else {
                throw new InvalidCredentialException(adaptorName, "Unsupported credential type: " + credential.getClass().getName());
            }
        }
        try {
            session.auth().verify(timeout);
        }
        catch (IOException e) {
            throw new SmartSocketsException(adaptorName, "Connection authentication failed", e);
        }
        return session;
    }

    public static SSHConnection connect(String adaptorName, SshClient client, String location, Credential credential, int bufferSize, long timeout) throws SmartSocketsException {
        if (credential == null) {
            throw new IllegalArgumentException("Credential may not be null");
        }
        if (timeout <= 0L) {
            throw new IllegalArgumentException("Invalid timeout: " + timeout);
        }
        if (location == null) {
            throw new IllegalArgumentException("Location may not be null");
        }
        SshdSocketAddress[] locations = SSHUtil.extractLocations(adaptorName, location);
        UserCredential[] creds = SSHUtil.extractCredentials(adaptorName, locations, credential);
        SSHConnection connection = new SSHConnection(locations.length - 1);
        ClientSession session = SSHUtil.connectAndAuthenticate(adaptorName, client, locations[0].getHostName(), locations[0].getPort(), creds[0], timeout);
        try {
            for (int i = 1; i < locations.length; ++i) {
                ChannelDirectTcpip channel = session.createDirectTcpipChannel(null, locations[i]);
                channel.open().await(timeout);
                ServerSocket server = new ServerSocket(0);
                int port = server.getLocalPort();
                Tunnel tunnel = new Tunnel(server, channel, bufferSize);
                tunnel.start();
                connection.addHop(i - 1, session, tunnel);
                session = SSHUtil.connectAndAuthenticate(adaptorName, client, "localhost", port, creds[i], timeout);
            }
        }
        catch (IOException e) {
            connection.close();
            try {
                session.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            throw new SmartSocketsException(adaptorName, "Failed to set up SSH forwarding", e);
        }
        catch (SmartSocketsException xe) {
            connection.close();
            throw xe;
        }
        connection.setSession(session);
        return connection;
    }

    static class Tunnel
    extends Thread {
        private final ServerSocket server;
        private final ChannelDirectTcpip channel;
        private final int bufferSize;
        private Socket socket;
        private Exception ex;

        Tunnel(ServerSocket ss, ChannelDirectTcpip tmp, int bufferSize) {
            this.server = ss;
            this.channel = tmp;
            this.bufferSize = bufferSize;
        }

        public synchronized Exception getException() {
            return this.ex;
        }

        public synchronized void close() {
            try {
                this.socket.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }

        private void closeServer() {
            try {
                this.server.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                Socket s = this.server.accept();
                s.setTcpNoDelay(true);
                this.closeServer();
                Tunnel tunnel = this;
                synchronized (tunnel) {
                    this.socket = s;
                }
                new Forwarder(s.getInputStream(), this.channel.getInvertedIn(), null, "LOCAL TO REMOTE");
                new Forwarder(this.channel.getInvertedOut(), s.getOutputStream(), null, "REMOTE TO LOCAL");
            }
            catch (IOException e) {
                Tunnel tunnel = this;
                synchronized (tunnel) {
                    this.ex = e;
                }
            }
        }
    }

    static class PasswordProvider
    implements FilePasswordProvider {
        private final char[] password;

        public PasswordProvider(char[] password) {
            this.password = (char[])password.clone();
        }

        @Override
        public String getPassword(SessionContext session, NamedResource resourceKey, int retryIndex) throws IOException {
            return new String(this.password);
        }
    }
}

