#!python3

"""
    Configuration for project documentation using Sphinx.
"""

# standard
import sys
from datetime import date, datetime
from importlib.metadata import version
from os import environ, path

import toml

__version__ = version("qgis-plugin-ci")
pyproject = toml.load("../pyproject.toml")

# -- Build environment -----------------------------------------------------
on_rtd = environ.get("READTHEDOCS", None) == "True"

# -- Project information -----------------------------------------------------
author = ", ".join([a["name"] for a in pyproject["project"]["authors"]])
__copyright__ = f"2019 - {date.today().year}, {author}"
project = "QGIS Plugin CI"
version = release = __version__


# -- General configuration ---------------------------------------------------

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    # Sphinx included
    "sphinx.ext.autodoc",
    "sphinx.ext.autosectionlabel",
    "sphinx.ext.githubpages",
    "sphinx.ext.intersphinx",
    "sphinx.ext.extlinks",
    # 3rd party
    "myst_parser",
    "sphinx_copybutton",
    "sphinx_design",
    "sphinx_sitemap",
]


# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
#
source_suffix = {".md": "markdown", ".rst": "restructuredtext"}
autosectionlabel_prefix_document = True
# The master toctree document.
master_doc = "index"

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = "en"

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path .
exclude_patterns = ["_build", ".venv", "Thumbs.db", ".DS_Store", "_output", "ext_libs"]

# The name of the Pygments (syntax highlighting) style to use.
pygments_style = "sphinx"


# -- Options for HTML output -------------------------------------------------

# -- Theme

# final URL
html_baseurl = pyproject["project"]["urls"]["homepage"]

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
# html_static_path = ["static"]
html_extra_path = ["robots.txt"]

# theme
html_theme = "furo"
html_theme_options = {
    "navigation_with_keys": True,
    "source_repository": pyproject["project"]["urls"]["repository"],
    "source_branch": "main",
    "source_directory": "docs/",
}


# -- EXTENSIONS --------------------------------------------------------

myst_enable_extensions = [
    "colon_fence",
    "deflist",
    "html_image",
    "linkify",
    "replacements",
    "smartquotes",
    "substitution",
]

myst_heading_anchors = 3

# replacement variables
myst_substitutions = {
    "author": author,
    "date_update": datetime.now().strftime("%d %B %Y"),
    "description": pyproject["project"]["description"],
    "repo_url": pyproject["project"]["urls"]["homepage"],
    "title": project,
    "version": version,
}

myst_url_schemes = ("http", "https", "mailto")

# Configuration for intersphinx (refer to others docs).
intersphinx_mapping = {"python": ("https://docs.python.org/3/", None)}

# sitemap
sitemap_url_scheme = "{link}"

# -- Options for Sphinx API doc ----------------------------------------------


# run api doc
def run_apidoc(_):
    from sphinx.ext.apidoc import main

    cur_dir = path.normpath(path.dirname(__file__))
    output_path = path.join(cur_dir, "_apidoc")
    modules = path.normpath(path.join(cur_dir, "../qgispluginci"))
    exclusions = ["../.venv", "../scripts", "../server", "../tests"]
    main(["-e", "-f", "-M", "-o", output_path, modules] + exclusions)


# launch setup
def setup(app):
    app.connect("builder-inited", run_apidoc)
