"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Peer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
/**
 * Peer object factories (to be used in Security Group management).
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 *
 * @stability stable
 */
class Peer {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * Create an IPv4 peer from a CIDR.
     *
     * @stability stable
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * Any IPv4 address.
     *
     * @stability stable
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * Create an IPv6 peer from a CIDR.
     *
     * @stability stable
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * Any IPv6 address.
     *
     * @stability stable
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * A prefix list.
     *
     * @stability stable
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
}
exports.Peer = Peer;
_a = JSII_RTTI_SYMBOL_1;
Peer[_a] = { fqn: "@aws-cdk/aws-ec2.Peer", version: "1.103.0" };
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super('0.0.0.0/0');
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super('::/0');
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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