# -*- coding: utf-8 -*-

import functools

from fast_tracker.common.async_wrapper import async_wrapper
from fast_tracker.api.cat_header_mixin import CatHeaderMixin
from fast_tracker.api.time_trace import TimeTrace, current_trace
from fast_tracker.core.external_node import ExternalNode
from fast_tracker.common.object_wrapper import FunctionWrapper, wrap_object


class ExternalTrace(CatHeaderMixin, TimeTrace):

    def __init__(self, library, url, method=None, span_type=None, span_layer=None, **kwargs):
        parent = None
        if kwargs:
            if len(kwargs) > 1:
                raise TypeError("无效参数:", kwargs)
            parent = kwargs['parent']
        super(ExternalTrace, self).__init__(parent)

        self.library = library
        self.url = url
        self.method = method
        self.params = {}
        self.span_type = span_type
        self.span_layer = span_layer

    def __repr__(self):
        return '<%s %s>' % (self.__class__.__name__, dict(
                library=self.library, url=self.url, method=self.method))

    def terminal_node(self):
        return True

    def create_node(self):
        return ExternalNode(
                library=self.library,
                url=self.url,
                method=self.method,
                children=self.children,
                start_time=self.start_time,
                end_time=self.end_time,
                duration=self.duration,
                exclusive=self.exclusive,
                params=self.params,
                is_async=self.is_async,
                guid=self.guid,
                agent_attributes=self.agent_attributes,
                user_attributes=self.user_attributes,
                span_type=self.span_type,
                span_layer=self.span_layer
        )


def ExternalTraceWrapper(wrapped, library, url, method=None):

    def dynamic_wrapper(wrapped, instance, args, kwargs):
        wrapper = async_wrapper(wrapped)
        if not wrapper:
            parent = current_trace()
            if not parent:
                return wrapped(*args, **kwargs)
        else:
            parent = None

        if callable(url):
            if instance is not None:
                _url = url(instance, *args, **kwargs)
            else:
                _url = url(*args, **kwargs)

        else:
            _url = url

        if callable(method):
            if instance is not None:
                _method = method(instance, *args, **kwargs)
            else:
                _method = method(*args, **kwargs)

        else:
            _method = method

        trace = ExternalTrace(library, _url, _method, parent=parent)

        if wrapper:
            return wrapper(wrapped, trace)(*args, **kwargs)

        with trace:
            return wrapped(*args, **kwargs)

    def literal_wrapper(wrapped, instance, args, kwargs):
        wrapper = async_wrapper(wrapped)
        if not wrapper:
            parent = current_trace()
            if not parent:
                return wrapped(*args, **kwargs)
        else:
            parent = None

        trace = ExternalTrace(library, url, method, parent=parent)

        wrapper = async_wrapper(wrapped)
        if wrapper:
            return wrapper(wrapped, trace)(*args, **kwargs)

        with trace:
            return wrapped(*args, **kwargs)

    if callable(url) or callable(method):
        return FunctionWrapper(wrapped, dynamic_wrapper)

    return FunctionWrapper(wrapped, literal_wrapper)


def external_trace(library, url, method=None):
    return functools.partial(ExternalTraceWrapper, library=library,
            url=url, method=method)


def wrap_external_trace(module, object_path, library, url, method=None):
    wrap_object(module, object_path, ExternalTraceWrapper,
            (library, url, method))
