#!/usr/bin/env python
# ******************************************************************************
# Copyright 2022 Brainchip Holdings Ltd.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ******************************************************************************
import tensorflow as tf
import keras

from quantizeml.tensors import FixedPoint, MAX_BUFFER_BITWIDTH
from .layers import deserialize_quant_object, Calibrable


__all__ = ["softmax2", "QuantizedSoftmax2"]


def softmax2(logits, axis=-1):
    """Computes softmax-like activations, but using base 2 for the exponential.

    Used as approximation of the softmax activation. The sum of all outputs
    generated by softmax2 is 1.
    This function performs the equivalent of
    ```python
    softmax = 2**logits / tf.reduce_sum(tf.exp(logits), axis, keepdims=True)
    ```

    Args:
        logits (:obj:`tf.Tensor`): a non-empty `Tensor`.
        axis (int, list, optional): the dimension softmax2 would be performed
            on. The default is -1 which indicates the last dimension.

    Returns:
        :obj:`tf.Tensor` value of softmax2 function with the same type and
            shape as `logits`.

    Raises:
        InvalidArgumentError: if `logits` is empty or `axis` is beyond the last
            dimension of `logits`.
    """

    exp = tf.cast(2**logits, dtype=logits.dtype)
    sum_exp = tf.reduce_sum(exp, axis=axis, keepdims=True)
    return exp / sum_exp


@tf.keras.utils.register_keras_serializable()
class QuantizedSoftmax2(Calibrable, keras.layers.Layer):
    """A quantized layer to do a quantized function similar to the softmax, but
    using base 2 instead of e. So we replace

        softmax(x_i) = e^x_i / sum(e^x_k)

    With this:

        softmax2(x_i) = 2^x_i / sum(2^x_k)

    As in the case of the softmax function, the summation of the pseudo-softmax
    outputs is always equal to one. Consequently, the values softmax2(x_i) can
    be interpreted as probabilities.

    Implementation is inspired from this paper:

    Cardarilli, G.C., Di Nunzio, L., Fazzolari, R. et al.
    A pseudo-softmax function for hardware-based high speed image classification.
    Sci Rep 11, 15307 (2021). https://doi.org/10.1038/s41598-021-94691-7
    """

    def __init__(self, *args, quant_config={}, **kwargs):
        super().__init__(*args, **kwargs)
        self.quant_config = quant_config
        self.out_quantizer = deserialize_quant_object(
            self.quant_config, "output_quantizer", False)
        self.sum_exp_bitwidth = self.quant_config.get("sum_exp_bitwidth", 16)
        self.exp_bitwidth = self.quant_config.get(
            "exp_bitwidth", self.sum_exp_bitwidth // 2)
        self.exp_reciprocal_bitwidth = self.quant_config.get(
            "exp_reciprocal_bitwidth", MAX_BUFFER_BITWIDTH) - 1

    def call(self, x, training=None):
        # raise an error if the inputs are not FixedPoint or tf.Tensor
        if not isinstance(x, FixedPoint):
            raise TypeError(f"QuantizedSoftmax2 only accepts FixedPoint\
                              inputs. Receives {type(x)} inputs.")

        # To avoid overflowing, some modifications are made to the input.
        # First remove the fractional part of the input (floor(x)). We can do
        # this because the exponential function will return very big numbers,
        # so fractional ones can be ignored in the ratio with the sum.
        x = x >> x.frac_bits

        # The pseudo-softmax is defined as:
        #
        # p = 2^x/sum(2^x)
        #
        # but we do this instead:
        #
        # p' = 2^y/sum(2^y)
        #
        # where
        #
        # y = x - x0
        #
        # This will give approximately the same result as the original.
        # We choose x0 to be the maximum of x, minus a positive integer
        # constant "exp_bitwidth". So now
        #
        # y = x - (max(x) - exp_bitwidth)
        #
        # If we remove the negative values of y, the 2^y will only return
        # positive values in the range [0, 2^exp_bitwidth]. This approximation
        # is good enough, because the negative values of y would result only
        # in fractional values of 2^y, that can be ignored in the ratio between
        # 2^y and sum(2^y).

        # Find max(x)
        x_max = tf.reduce_max(x, axis=-1, keepdims=True)
        x0 = x_max - self.exp_bitwidth
        # And y can be calculated as
        y = x - x0

        # This will calculate 2^y. Note that hardware will only need to store
        # the sum, there is no need to store the 2^y.
        # To calculate 2^y, hardware can just do a bitshift, but here we will
        # multiply 1 by the 2^y
        one = FixedPoint(1.0, frac_bits=0, value_bits=self.sum_exp_bitwidth)
        # To multiply by a power of two, the mul_pow_2 function can be used.
        # This uses an int a parameter, but y_values = y since frac_bits = 0.
        exp = one.mul_pow_2(y.values)

        # Calculate the sum of the exponential now. Promote exp to avoid
        # saturating on bitwidth.
        sum_exp = tf.reduce_sum(exp, axis=-1, keepdims=True)

        # Instead of dividing by the sum, the reciprocal is estimated.
        sum_exp = sum_exp.promote(self.exp_reciprocal_bitwidth)
        sum_exp_rec = tf.math.reciprocal(sum_exp)

        # The output is the multiplication of the 1/sum(2^y) and the 2^y.
        # Since y has 0 frac_bits, the multiplication in hardware can be done
        # with a simple bitshift.
        outputs = sum_exp_rec * exp

        # Finally, the output is quantized to allow reducing the bitwidth
        if self.out_quantizer is not None:
            outputs = self.out_quantizer(outputs, training=training)
        return outputs

    def get_config(self):
        config = super().get_config()
        config["quant_config"] = self.quant_config
        return config
