#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright by: P.J. Grochowski

import time

from PyQt5.QtCore import QTimer
from PyQt5.QtWidgets import QStyle, QWidget

from kast.interface.qt5.UiServices import UiServices
from kast.interface.qt5.service.UiEvent import UiEvent, UiState
from kast.interface.qt5.view.MediaControlView import Ui_MediaControlView
from kast.interface.qt5.viewModel.VideoPreviewViewModel import VideoPreviewViewModel
from kast.interface.qt5.viewModel.ViewModelBase import ViewBase, ViewModelBase
from kast.media.casting.CastState import CastPlayerState, CastState, Seconds


class View(ViewBase, QWidget, Ui_MediaControlView):
    pass


class MediaControlViewModel(ViewModelBase):

    SEEK_DELTA = Seconds(10)

    def __init__(
        self,
        parent: QWidget,
        uiServices: UiServices,
        videoPreviewViewModel: VideoPreviewViewModel
    ) -> None:
        self._view = View(parent=parent)
        super().__init__(uiServices=uiServices, view=self._view)
        self.uiServices.uiEventObserver.register(self, self._onUiEvent)

        self._videoPreviewViewModel = videoPreviewViewModel
        self._videoPreviewViewModel.setMuteCallback(self._onLocalMute)
        self._videoPreviewViewModel.setVolumeCallback(self._onLocalVolume)

        self._lastMediaState = CastPlayerState.Unknown

        self._view.buttonPlayPause.setIcon(self._view.style().standardIcon(QStyle.SP_MediaPlay))
        self._view.buttonStop.setIcon(self._view.style().standardIcon(QStyle.SP_MediaStop))
        self._view.buttonSeekFront.setIcon(self._view.style().standardIcon(QStyle.SP_MediaSeekForward))
        self._view.buttonSeekBack.setIcon(self._view.style().standardIcon(QStyle.SP_MediaSeekBackward))
        self._view.buttonMuteRemote.setIcon(self._view.style().standardIcon(QStyle.SP_MediaVolume))
        self._view.buttonMuteLocal.setIcon(self._view.style().standardIcon(QStyle.SP_MediaVolume))

        self._videoPositionUpdater = QTimer()
        self._videoPositionUpdater.setInterval(1_000)
        self._videoPositionUpdater.timeout.connect(self._signalVideoPositionUpdate)
        self._videoPositionUpdater.start()

        self._view.buttonPlayPause.clicked.connect(self._signalClickedPlayPause)
        self._view.buttonStop.clicked.connect(self._signalClickedStop)
        self._view.buttonSeekFront.clicked.connect(self._signalClickedSeekForward)
        self._view.buttonSeekBack.clicked.connect(self._signalClickedSeekBackward)
        self._view.buttonMuteRemote.clicked.connect(self._signalClickedMuteRemote)
        self._view.buttonMuteLocal.clicked.connect(self._signalClickedMuteLocal)

        self._view.sliderSeek.sliderReleased.connect(self._signalSeekPosition)
        self._view.sliderVolumeRemote.sliderReleased.connect(self._signalSetVolumeRemote)
        self._view.sliderVolumeLocal.sliderReleased.connect(self._signalSetVolumeLocal)

        self._view.setVisible(False)

    def _onUiEvent(self, uiEvent: UiEvent, castState: CastState) -> None:
        if(
                castState.mediaState.playerState == CastPlayerState.Unknown and
                self._lastMediaState != CastPlayerState.Unknown
        ):
            def interfaceCallback() -> None:
                self.services.castController.quit()
                self.services.castController.disconnect()
                self.uiServices.uiEventObserver.notify(uiEvent=UiEvent(state=UiState.Idle))
            self.uiServices.interfaceScheduler.schedule(callback=interfaceCallback)
        self._lastMediaState = castState.mediaState.playerState

        isStreaming = uiEvent.state == UiState.Streaming
        self._view.setVisible(isStreaming)
        self._videoPreviewViewModel.enable(state=isStreaming)
        self._videoPreviewViewModel.setPlayerState(state=castState.mediaState.playerState)

        if uiEvent.state == UiState.Closing:
            self._videoPreviewViewModel.cleanup()

        iconPlayPause = QStyle.SP_MediaPause if castState.mediaState.isPlaying() else QStyle.SP_MediaPlay
        self._view.buttonPlayPause.setIcon(self._view.style().standardIcon(iconPlayPause))

        self._updateVideoPosition(position=castState.mediaState.adjustedPosition, duration=castState.mediaState.duration)

        volumeLevel = int(round(castState.mediaState.volumeLevel * 100))
        self._view.sliderVolumeRemote.setSliderPosition(volumeLevel)

        iconMuted = QStyle.SP_MediaVolumeMuted if castState.mediaState.volumeMuted or volumeLevel == 0 else QStyle.SP_MediaVolume
        self._view.buttonMuteRemote.setIcon(self._view.style().standardIcon(iconMuted))

    def _onLocalMute(self, muted: bool) -> None:
        iconMuted = QStyle.SP_MediaVolumeMuted if muted else QStyle.SP_MediaVolume
        self._view.buttonMuteLocal.setIcon(self._view.style().standardIcon(iconMuted))

    def _onLocalVolume(self, volume: int) -> None:
        self._view.sliderVolumeLocal.setValue(volume)

    def _updateVideoPosition(self, position: Seconds, duration: Seconds) -> None:
        def formatTime(seconds: Seconds) -> str:
            return time.strftime('%H:%M:%S', time.gmtime(seconds.value))

        self._view.labelTime.setText(f"{formatTime(position)} / {formatTime(duration)}")

        self._view.sliderSeek.setRange(0, duration.value)
        self._view.sliderSeek.setSliderPosition(position.value)

        self._videoPreviewViewModel.updateVideoPosition(position=position)

    def _signalVideoPositionUpdate(self) -> None:
        mediaState = self.uiServices.uiEventObserver.castState.mediaState
        if mediaState.playerState == CastPlayerState.Playing:
            self._updateVideoPosition(
                position=mediaState.adjustedPosition,
                duration=mediaState.duration
            )

    def _signalClickedPlayPause(self) -> None:
        if self.uiServices.uiEventObserver.castState.mediaState.isPlaying():
            self.services.castController.pause()
        else:
            self.services.castController.play()

    def _signalClickedStop(self) -> None:
        self.services.castController.stop()

    def _signalClickedSeekForward(self) -> None:
        self.services.castController.seek(timePos=Seconds(self._view.sliderSeek.value()) + self.SEEK_DELTA)

    def _signalClickedSeekBackward(self) -> None:
        self.services.castController.seek(timePos=Seconds(self._view.sliderSeek.value()) - self.SEEK_DELTA)

    def _signalSeekPosition(self) -> None:
        self.services.castController.seek(timePos=Seconds(self._view.sliderSeek.value()))

    def _signalClickedMuteRemote(self) -> None:
        self.services.castController.setMute(not self.uiServices.uiEventObserver.castState.mediaState.volumeMuted)

    def _signalSetVolumeRemote(self) -> None:
        self.services.castController.setVolume(val=self._view.sliderVolumeRemote.value()/100)

    def _signalClickedMuteLocal(self) -> None:
        self._videoPreviewViewModel.muted = not self._videoPreviewViewModel.muted

    def _signalSetVolumeLocal(self) -> None:
        self._videoPreviewViewModel.volume = self._view.sliderVolumeLocal.value()
