#!/usr/bin/env python3

# Nga: a Virtual Machine
# Copyright (c) 2010 - 2020, Charles Childers
# Floating Point I/O by Arland Childers, (c) 2020
# Optimizations and process() rewrite by Greg Copeland
# -------------------------------------------------------------
# This implementation of the VM differs from the reference
# model in a number of important ways.
#
# To aid in performance, it does the following:
# - caching the Retro dictionary in a Python dict()
# - replaces some Retro words with implementations in Python
#   - s:eq?
#   - s:length
#   - s:to-number
#   - d:lookup
#
# Each major component is managed as a separate class. We have
# a class for each I/O device, for each stack, and for the
# memory pool. The main VM is also in a separate class.
#
# It's intended that an amalgamation tool will be developed to
# combine the separate files into a single one for deployment.
# -------------------------------------------------------------

import os, sys, math, time, struct, random, datetime

class Clock:
    def __getitem__(self, id):
        import datetime
        import time

        now = datetime.datetime.now()
        ids = {
            "time": time.time,
            "year": now.year,
            "month": now.month,
            "day": now.day,
            "hour": now.hour,
            "minute": now.minute,
            "second": now.second,
            # No time_utc?
            "year_utc": now.utcnow().year,
            "month_utc": now.utcnow().month,
            "day_utc": now.utcnow().day,
            "hour_utc": now.utcnow().hour,
            "minute_utc": now.utcnow().minute,
            "second_utc": now.utcnow().second,
        }
        return ids[id]
import random

class RNG:
    def __call__(self):
        return random.randint(-2147483647, 2147483646)
import os


class FileSystem:
    def __init__(self):
        self.files = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0]

    def open(self, params):
        name, mode = params
        slot = 0
        i = 1
        while i < 8:
            if self.files[i] == 0:
                slot = i
            i += 1
        if slot > 0:
            if mode == 0:
                if os.path.exists(name):
                    self.files[slot] = open(name, "r")
                else:
                    slot = 0
            elif mode == 1:
                self.files[slot] = open(name, "w")
            elif mode == 2:
                self.files[slot] = open(name, "a")
            elif mode == 3:
                if os.path.exists(name):
                    self.files[slot] = open(name, "r+")
                else:
                    slot = 0
        return slot

    def read(self, slot):
        return ord(self.files[slot].read(1))

    def write(self, params):
        slot, char = params
        self.files[slot].write(chr(stack.pop()))
        return 1

    def close(self, slot):
        self.files[slot].close()
        self.files[slot] = 0
        return 0

    def pos(self, slot):
        return self.files[slot].tell()

    def seek(slot, pos):
        return self.files[slot].seek(pos, 0)

    def size(self, slot):
        at = self.files[slot].tell()
        self.files[slot].seek(0, 2)  # SEEK_END
        end = self.files[slot].tell()
        self.files[slot].seek(at, 0)  # SEEK_SET
        return end

    def delete(self, name):
        name = extract_string(stack.pop())
        i = 0
        if os.path.exists(name):
            os.remove(name)
            i = 1
        return i

class FloatStack(object):
    def __init__(self, *d):
        self.data = list(d)

    def __getitem__(self, id):
        return self.data[id]

    def add(self):
        self.data.append(self.data.pop() + self.data.pop())

    def sub(self):
        self.data.append(0 - (self.data.pop() - self.data.pop()))

    def mul(self):
        self.data.append(self.data.pop() * self.data.pop())

    def div(self):
        a, b = self.data.pop(), self.data.pop()
        self.data.append(b / a)

    def ceil(self):
        self.data.append(math.ceil(self.data.pop()))

    def floor(self):
        self.data.append(math.floor(self.data.pop()))

    def eq(self):
        return 0 - (self.data.pop() == self.data.pop())

    def neq(self):
        return 0 - (self.data.pop() != self.data.pop())

    def gt(self):
        a, b = self.data.pop(), self.data.pop()
        return 0 - (b > a)

    def lt(self):
        a, b = self.data.pop(), self.data.pop()
        return 0 - (b < a)

    def depth(self):
        return len(self.data)

    def drop(self):
        self.data.pop()

    def pop(self):
        return self.data.pop()

    def swap(self):
        a, b = self.data.pop(), self.data.pop()
        self.data += [a, b]

    def push(self, n):
        self.data.append(n)

    def log(self):
        a, b = self.data.pop(), self.data.pop()
        self.data.append(math.log(b, a))

    def power(self):
        a, b = self.data.pop(), self.data.pop()
        self.data.append(math.pow(a, b))

    def sin(self):
        self.data.append(math.sin(self.data.pop()))

    def cos(self):
        self.data.append(math.cos(self.data.pop()))

    def tan(self):
        self.data.append(math.tan(self.data.pop()))

    def asin(self):
        self.data.append(math.asin(self.data.pop()))

    def acos(self):
        self.data.append(math.acos(self.data.pop()))

    def atan(self):
        self.data.append(math.atan(self.data.pop()))
class IntegerStack(list):
    def __init__(self):
        stack = [] * 128
        self.extend(stack)

    def depth(self):
        return len(self)

    def tos(self):
        return self[-1]

    def push(self, v):
        self.append(v)

    def dup(self):
        self.append(self[-1])

    def drop(self):
        self.pop()

    def swap(self):
        a = self[-2]
        self[-2] = self[-1]
        self[-1] = a
import os
import struct


class Memory(list):
    def __init__(self, source, initial, size):
        m = [0] * size
        self.extend(m)
        if len(initial) == 0:
            cells = int(os.path.getsize(source) / 4)
            f = open(source, "rb")
            i = 0
            for cell in list(struct.unpack(cells * "i", f.read())):
                self[i] = cell
                i = i + 1
            f.close()
        else:
            i = 0
            for cell in initial:
                self[i] = cell
                i = i + 1

    def load_image(self, name):
        cells = int(os.path.getsize(name) / 4)
        f = open(name, "rb")
        i = 0
        for cell in list(struct.unpack(cells * "i", f.read())):
            self[i] = cell
            i = i + 1
        f.close()

    def size(self):
        return len(self)
InitialImage = [
1793, -1, 12716, 12758, 202101, 0, 10, 1, 10, 2, 10, 
3, 10, 4, 10, 5, 10, 6, 10, 7, 10, 8, 
10, 9, 10, 10, 11, 10, 12, 10, 13, 10, 14, 
10, 15, 10, 16, 10, 17, 10, 18, 10, 19, 10, 
20, 10, 21, 10, 22, 10, 23, 10, 24, 10, 25, 
10, 68223234, 1, 2575, 85000450, 1, 656912, 3211, 3220, 268505089, 63, 
62, 285281281, 0, 63, 2063, 10, 101384453, 0, 9, 10, 2049, 
56, 25, 459011, 76, 524546, 76, 302256641, 1, 10, 16974595, 0, 
50529798, 10, 25, 524547, 95, 50529798, 10, 17108738, 1, 251790353, 101777669, 
1, 17565186, 86, 524545, 90, 64, 167838467, -1, 134287105, 3, 59, 
659457, 3, 459023, 107, 2049, 56, 25, 2049, 107, 1793, 114, 
2049, 114, 117506307, 0, 107, 0, 524545, 25, 112, 168820993, 0, 
126, 1642241, 126, 134283523, 7, 112, 1793, 107, 7, 524545, 2049, 
107, 1793, 107, 16846593, 126, 141, 140, 1793, 64, 16846593, 126, 
112, 140, 1793, 64, 7, 10, 659713, 1, 659713, 2, 659713, 
3, 1793, 168, 17108737, 3, 2, 524559, 107, 2049, 107, 2049, 
107, 2049, 121, 168820998, 2, 3196, 1025, 167841793, 181, 5, 17826049, 
0, 181, 2, 15, 25, 524546, 164, 134287105, 182, 95, 2305, 
183, 459023, 191, 134287361, 182, 186, 659201, 181, 2049, 56, 25, 
84152833, 48, 286458116, 10, 459014, 206, 184618754, 45, 25, 16974851, -1, 
168886532, 1, 134284289, 1, 215, 134284289, 0, 206, 660227, 32, 0, 
0, 112, 114, 101, 102, 105, 120, 58, 125, 0, 285278479, 
232, 7, 2576, 524546, 81, 1641217, 1, 167838467, 229, 2049, 245, 
2049, 241, 524545, 232, 201, 17826050, 231, 0, 2572, 2563, 2049, 
222, 1793, 133, 459023, 133, 17760513, 146, 3, 166, 8, 251727617, 
3, 2, 2049, 160, 16, 168820993, -1, 126, 2049, 201, 2049, 
160, 459023, 133, 285282049, 3, 2, 134287105, 126, 280, 524545, 1793, 
107, 16846593, 3, 0, 107, 8, 659201, 3, 524545, 25, 112, 
17043201, 3, 7, 2049, 112, 2049, 107, 268505092, 126, 1642241, 126, 
656131, 659201, 3, 524545, 7, 112, 2049, 107, 459009, 19, 112, 
459009, 54, 112, 459009, 15, 112, 459009, 17, 112, 1793, 5, 
10, 524546, 160, 134284303, 162, 1807, 1024, 0, 0, 1642241, 231, 
285282049, 347, 1, 459012, 342, 117509889, 181, 342, 134287105, 347, 201, 
16845825, 0, 357, 339, 1793, 64, 1793, 371, 17826050, 347, 251, 
8, 117506305, 348, 360, 64, 2116, 11340, 11700, 11400, 13685, 13104, 
12432, 12402, 9603, 9801, 11514, 11413, 11110, 12528, 11948, 10302, 13340, 
9700, 13455, 12753, 10500, 10670, 12654, 13320, 11960, 13908, 10088, 10605, 
11865, 11025, 0, 2049, 201, 987393, 1, 1793, 107, 524546, 447, 
2049, 445, 2049, 445, 17891588, 2, 447, 8, 17045505, -24, -16, 
17043736, -8, 1118488, 1793, 107, 17043202, 1, 169021201, 2049, 56, 25, 
33883396, 101450758, 6404, 459011, 437, 34668804, 2, 2049, 434, 524545, 379, 
437, 302056196, 379, 659969, 1, 0, 9, 152, 100, 117, 112, 
0, 456, 11, 152, 100, 114, 111, 112, 0, 463, 13, 
152, 115, 119, 97, 112, 0, 471, 21, 152, 99, 97, 
108, 108, 0, 479, 26, 152, 101, 113, 63, 0, 487, 
28, 152, 45, 101, 113, 63, 0, 494, 30, 152, 108, 
116, 63, 0, 502, 32, 152, 103, 116, 63, 0, 509, 
34, 152, 102, 101, 116, 99, 104, 0, 516, 36, 152, 
115, 116, 111, 114, 101, 0, 525, 38, 152, 43, 0, 
534, 40, 152, 45, 0, 539, 42, 152, 42, 0, 544, 
44, 152, 47, 109, 111, 100, 0, 549, 46, 152, 97, 
110, 100, 0, 557, 48, 152, 111, 114, 0, 564, 50, 
152, 120, 111, 114, 0, 570, 52, 152, 115, 104, 105, 
102, 116, 0, 577, 333, 158, 112, 117, 115, 104, 0, 
586, 336, 158, 112, 111, 112, 0, 594, 330, 158, 48, 
59, 0, 601, 56, 146, 102, 101, 116, 99, 104, 45, 
110, 101, 120, 116, 0, 607, 59, 146, 115, 116, 111, 
114, 101, 45, 110, 101, 120, 116, 0, 621, 222, 146, 
115, 58, 116, 111, 45, 110, 117, 109, 98, 101, 114, 
0, 635, 95, 146, 115, 58, 101, 113, 63, 0, 650, 
81, 146, 115, 58, 108, 101, 110, 103, 116, 104, 0, 
659, 64, 146, 99, 104, 111, 111, 115, 101, 0, 671, 
74, 152, 105, 102, 0, 681, 72, 146, 45, 105, 102, 
0, 687, 262, 158, 112, 114, 101, 102, 105, 120, 58, 
40, 0, 694, 126, 133, 67, 111, 109, 112, 105, 108, 
101, 114, 0, 706, 3, 133, 72, 101, 97, 112, 0, 
718, 107, 146, 44, 0, 726, 121, 146, 115, 44, 0, 
731, 127, 158, 59, 0, 737, 289, 158, 91, 0, 742, 
305, 158, 93, 0, 747, 2, 133, 68, 105, 99, 116, 
105, 111, 110, 97, 114, 121, 0, 752, 159, 146, 100, 
58, 108, 105, 110, 107, 0, 766, 160, 146, 100, 58, 
120, 116, 0, 776, 162, 146, 100, 58, 99, 108, 97, 
115, 115, 0, 784, 164, 146, 100, 58, 110, 97, 109, 
101, 0, 795, 146, 146, 99, 108, 97, 115, 115, 58, 
119, 111, 114, 100, 0, 805, 158, 146, 99, 108, 97, 
115, 115, 58, 109, 97, 99, 114, 111, 0, 819, 133, 
146, 99, 108, 97, 115, 115, 58, 100, 97, 116, 97, 
0, 834, 166, 146, 100, 58, 97, 100, 100, 45, 104, 
101, 97, 100, 101, 114, 0, 848, 263, 158, 112, 114, 
101, 102, 105, 120, 58, 35, 0, 864, 269, 158, 112, 
114, 101, 102, 105, 120, 58, 58, 0, 876, 283, 158, 
112, 114, 101, 102, 105, 120, 58, 38, 0, 888, 267, 
158, 112, 114, 101, 102, 105, 120, 58, 36, 0, 900, 
320, 158, 114, 101, 112, 101, 97, 116, 0, 912, 322, 
158, 97, 103, 97, 105, 110, 0, 922, 369, 146, 105, 
110, 116, 101, 114, 112, 114, 101, 116, 0, 931, 201, 
146, 100, 58, 108, 111, 111, 107, 117, 112, 0, 944, 
152, 146, 99, 108, 97, 115, 115, 58, 112, 114, 105, 
109, 105, 116, 105, 118, 101, 0, 956, 4, 133, 86, 
101, 114, 115, 105, 111, 110, 0, 975, 416, 146, 105, 
0, 986, 107, 146, 100, 0, 991, 410, 146, 114, 0, 
996, 339, 146, 101, 114, 114, 58, 110, 111, 116, 102, 
111, 117, 110, 100, 0, 0, 0, 0, 0, 0, 0, 
0, 125, 125, 0, 111, 114, 45, 101, 97, 99, 104, 
0, 108, 101, 102, 116, 0, 110, 103, 45, 119, 105, 
116, 104, 0, 95, 70, 79, 85, 78, 68, 0, 124, 
125, 126, 0, 72, 73, 74, 75, 76, 77, 78, 79, 
80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 1001, 1543, 146, 69, 
79, 77, 0, 1, -3, 15, 10, 1536, 1556, 146, 100, 
101, 112, 116, 104, 0, 1, -1, 15, 10, 1547, 1570, 
146, 100, 58, 108, 97, 115, 116, 0, 1, 2, 15, 
10, 1560, 1587, 146, 100, 58, 108, 97, 115, 116, 46, 
120, 116, 0, 2049, 1570, 2049, 160, 15, 10, 1574, 1609, 
146, 100, 58, 108, 97, 115, 116, 46, 99, 108, 97, 
115, 115, 0, 2049, 1570, 2049, 162, 15, 10, 1593, 1630, 
146, 100, 58, 108, 97, 115, 116, 46, 110, 97, 109, 
101, 0, 2049, 1570, 2049, 164, 10, 1615, 1646, 146, 114, 
101, 99, 108, 97, 115, 115, 0, 2049, 1570, 2049, 162, 
16, 10, 1635, 1665, 146, 105, 109, 109, 101, 100, 105, 
97, 116, 101, 0, 1, 158, 2049, 1646, 10, 1652, 1678, 
146, 100, 97, 116, 97, 0, 1, 133, 2049, 1646, 10, 
1670, 1696, 146, 112, 114, 105, 109, 105, 116, 105, 118, 
101, 0, 1, 152, 2049, 1646, 10, 1683, 1706, 158, 40, 
0, 10, 1701, 1712, 158, 41, 0, 10, 1707, 1728, 146, 
99, 111, 109, 112, 105, 108, 101, 58, 108, 105, 116, 
0, 1, 1, 2049, 107, 2049, 107, 10, 1713, 1751, 146, 
99, 111, 109, 112, 105, 108, 101, 58, 106, 117, 109, 
112, 0, 1, 1793, 2049, 107, 2049, 107, 10, 1735, 1774, 
146, 99, 111, 109, 112, 105, 108, 101, 58, 99, 97, 
108, 108, 0, 1, 2049, 2049, 107, 2049, 107, 10, 1758, 
1796, 146, 99, 111, 109, 112, 105, 108, 101, 58, 114, 
101, 116, 0, 1, 10, 2049, 107, 10, 1781, 1815, 146, 
99, 111, 109, 112, 105, 108, 105, 110, 103, 63, 0, 
1, 126, 15, 10, 1801, 1831, 158, 112, 114, 101, 102, 
105, 120, 58, 96, 0, 2049, 222, 2049, 107, 10, 1819, 
1848, 158, 112, 114, 101, 102, 105, 120, 58, 92, 0, 
2049, 416, 10, 1836, 1863, 158, 112, 114, 101, 102, 105, 
120, 58, 94, 0, 2049, 410, 10, 1851, 1874, 146, 104, 
101, 114, 101, 0, 1, 3, 15, 10, 1866, 1890, 158, 
112, 114, 101, 102, 105, 120, 58, 64, 0, 2049, 201, 
2049, 160, 15, 2049, 1815, 1793, 1906, 1, 3841, 2049, 107, 
2049, 107, 10, 1, 1899, 1793, 1912, 15, 10, 1, 1910, 
2049, 64, 10, 1878, 1929, 158, 112, 114, 101, 102, 105, 
120, 58, 33, 0, 2049, 201, 2049, 160, 15, 2049, 1815, 
1793, 1945, 1, 4097, 2049, 107, 2049, 107, 10, 1, 1938, 
1793, 1951, 16, 10, 1, 1949, 2049, 64, 10, 1917, 1968, 
146, 100, 58, 99, 114, 101, 97, 116, 101, 0, 1, 
133, 1, 0, 2049, 166, 2049, 1874, 2049, 1570, 2049, 160, 
16, 10, 1956, 1991, 146, 118, 97, 114, 45, 110, 0, 
2049, 1968, 2049, 107, 10, 1982, 2003, 146, 118, 97, 114, 
0, 134284289, 0, 1991, 10, 1996, 2016, 146, 99, 111, 110, 
115, 116, 0, 2049, 1968, 2049, 1570, 2049, 160, 16, 10, 
2007, 2032, 152, 116, 117, 99, 107, 0, 100926722, 10, 2024, 
2042, 152, 111, 118, 101, 114, 0, 67502597, 10, 2034, 2051, 
152, 110, 105, 112, 0, 772, 10, 2044, 2066, 152, 100, 
114, 111, 112, 45, 112, 97, 105, 114, 0, 771, 10, 
2053, 2076, 152, 63, 100, 117, 112, 0, 6402, 10, 2068, 
2090, 146, 100, 117, 112, 45, 112, 97, 105, 114, 0, 
67502597, 67502597, 10, 2078, 2100, 146, 100, 105, 112, 0, 525572, 
6, 10, 2093, 2110, 146, 115, 105, 112, 0, 67502597, 1, 
21, 2049, 2100, 10, 2103, 2122, 146, 98, 105, 0, 1, 
2110, 2049, 2100, 8, 10, 2116, 2135, 146, 98, 105, 42, 
0, 1, 2100, 2049, 2100, 8, 10, 2128, 2148, 146, 98, 
105, 64, 0, 2, 2049, 2135, 10, 2141, 2159, 146, 116, 
114, 105, 0, 1793, 2168, 1, 2110, 2049, 2100, 2049, 2110, 
10, 1, 2161, 2049, 2100, 8, 10, 2152, 2182, 146, 116, 
114, 105, 42, 0, 1793, 2199, 1793, 2192, 4, 1, 2100, 
2049, 2100, 10, 1, 2186, 2049, 2100, 2049, 2100, 10, 1, 
2184, 2049, 2100, 8, 10, 2174, 2213, 146, 116, 114, 105, 
64, 0, 2, 2, 2049, 2182, 10, 2205, 2227, 146, 119, 
104, 105, 108, 101, 0, 1793, 2236, 525570, 1639430, 3, 1, 
2229, 7, 10, 1, 2229, 8, 3, 10, 2218, 2250, 146, 
117, 110, 116, 105, 108, 0, 1793, 2261, 525570, 385942534, -1, 
25, 3, 1, 2252, 7, 10, 1, 2252, 8, 3, 10, 
2241, 2275, 146, 116, 105, 109, 101, 115, 0, 1793, 2287, 
4, 25, 33886721, 1, 2053, 1542, 1, 2278, 7, 10, 1, 
2277, 8, 3, 10, 2266, 2304, 158, 112, 114, 101, 102, 
105, 120, 58, 124, 0, 2049, 201, 1793, 2312, 2049, 160, 
15, 10, 1, 2308, 1793, 2320, 2049, 162, 15, 10, 1, 
2316, 2049, 2122, 2049, 1815, 1793, 2335, 1, 133, 2049, 2100, 
2049, 1774, 10, 1, 2328, 1, 21, 2049, 64, 10, 2292, 
2350, 146, 84, 82, 85, 69, 0, 1, -1, 10, 2342, 
2362, 146, 70, 65, 76, 83, 69, 0, 1, 0, 10, 
2353, 2373, 146, 99, 97, 115, 101, 0, 1793, 2378, 67502597, 
11, 10, 1, 2375, 2049, 2100, 4, 1793, 2390, 772, 8, 
2049, 2350, 10, 1, 2385, 1793, 2398, 3, 2049, 2362, 10, 
1, 2394, 2049, 64, 25, 6, 3, 3, 10, 2365, 2417, 
146, 115, 58, 99, 97, 115, 101, 0, 1793, 2423, 67502597, 
2049, 95, 10, 1, 2419, 2049, 2100, 4, 1793, 2435, 772, 
8, 2049, 2350, 10, 1, 2430, 1793, 2443, 3, 2049, 2362, 
10, 1, 2439, 2049, 64, 25, 6, 3, 3, 10, 2407, 
2459, 146, 110, 111, 116, 0, 1, -1, 23, 10, 2452, 
2472, 146, 108, 116, 101, 113, 63, 0, 2049, 2090, 101516555, 
22, 10, 2463, 2486, 146, 103, 116, 101, 113, 63, 0, 
4, 2049, 2472, 10, 2477, 2499, 146, 110, 58, 77, 65, 
88, 0, 1, -5, 15, 10, 2490, 2512, 146, 110, 58, 
77, 73, 78, 0, 1, -4, 15, 10, 2503, 2527, 146, 
110, 58, 122, 101, 114, 111, 63, 0, 1, 0, 11, 
10, 2516, 2543, 146, 110, 58, 45, 122, 101, 114, 111, 
63, 0, 1, 0, 12, 10, 2531, 2562, 146, 110, 58, 
110, 101, 103, 97, 116, 105, 118, 101, 63, 0, 1, 
0, 13, 10, 2547, 2581, 146, 110, 58, 112, 111, 115, 
105, 116, 105, 118, 101, 63, 0, 1, -1, 14, 10, 
2566, 2609, 146, 110, 58, 115, 116, 114, 105, 99, 116, 
108, 121, 45, 112, 111, 115, 105, 116, 105, 118, 101, 
63, 0, 1, 0, 14, 10, 2585, 2624, 146, 110, 58, 
101, 118, 101, 110, 63, 0, 1, 2, 20, 3, 2049, 
2527, 10, 2613, 2641, 146, 110, 58, 111, 100, 100, 63, 
0, 2049, 2624, 2049, 2459, 10, 2631, 2653, 146, 105, 102, 
59, 0, 67502597, 1, 74, 2049, 2100, 25, 6, 771, 10, 
2646, 2670, 146, 45, 105, 102, 59, 0, 67502597, 1, 72, 
2049, 2100, 2049, 2459, 25, 6, 771, 10, 2662, 2688, 152, 
114, 111, 116, 0, 67503109, 10, 2681, 2695, 152, 47, 0, 
197652, 10, 2690, 2704, 152, 109, 111, 100, 0, 788, 10, 
2697, 2715, 146, 110, 58, 112, 111, 119, 0, 1, 1, 
4, 1793, 2723, 67502597, 19, 10, 1, 2720, 2049, 2275, 772, 
10, 2706, 2741, 146, 110, 58, 110, 101, 103, 97, 116, 
101, 0, 1, -1, 19, 10, 2729, 2757, 146, 110, 58, 
115, 113, 117, 97, 114, 101, 0, 4866, 10, 2745, 2769, 
146, 110, 58, 115, 113, 114, 116, 0, 1, 1, 1793, 
2787, 2049, 2090, 197652, 67502597, 18, 1, 2, 197652, 25, 17, 
1, 2773, 7, 10, 1, 2773, 8, 772, 10, 2759, 2801, 
146, 110, 58, 109, 105, 110, 0, 2049, 2090, 13, 1793, 
2808, 3, 10, 1, 2806, 1793, 2814, 772, 10, 1, 2812, 
2049, 64, 10, 2792, 2828, 146, 110, 58, 109, 97, 120, 
0, 2049, 2090, 14, 1793, 2835, 3, 10, 1, 2833, 1793, 
2841, 772, 10, 1, 2839, 2049, 64, 10, 2819, 2855, 146, 
110, 58, 97, 98, 115, 0, 2, 2049, 2562, 1, 2741, 
9, 10, 2846, 2873, 146, 110, 58, 108, 105, 109, 105, 
116, 0, 4, 5, 2049, 2801, 6, 2049, 2828, 10, 2862, 
2890, 146, 110, 58, 105, 110, 99, 0, 1, 1, 17, 
10, 2881, 2903, 146, 110, 58, 100, 101, 99, 0, 1, 
1, 18, 10, 2894, 2921, 146, 110, 58, 98, 101, 116, 
119, 101, 101, 110, 63, 0, 67503109, 1793, 2929, 67503109, 67503109, 
2049, 2873, 10, 1, 2924, 2049, 2110, 11, 10, 2907, 2947, 
146, 118, 58, 105, 110, 99, 45, 98, 121, 0, 1793, 
2951, 4367, 10, 1, 2949, 2049, 2110, 16, 10, 2935, 2969, 
146, 118, 58, 100, 101, 99, 45, 98, 121, 0, 1793, 
2973, 1180687, 10, 1, 2971, 2049, 2110, 16, 10, 2957, 2988, 
146, 118, 58, 105, 110, 99, 0, 1, 1, 4, 2049, 
2947, 10, 2979, 3003, 146, 118, 58, 100, 101, 99, 0, 
1, 1, 4, 2049, 2969, 10, 2994, 3020, 146, 118, 58, 
108, 105, 109, 105, 116, 0, 251790597, 1542, 2049, 2873, 4100, 
10, 3009, 3034, 146, 118, 58, 111, 110, 0, 2049, 2350, 
4100, 10, 3026, 3047, 146, 118, 58, 111, 102, 102, 0, 
2049, 2362, 4100, 10, 3038, 3060, 146, 97, 108, 108, 111, 
116, 0, 1, 3, 2049, 2947, 10, 3051, 3079, 146, 118, 
58, 112, 114, 101, 115, 101, 114, 118, 101, 0, 983556, 
1793, 3087, 1, 21, 2049, 2100, 10, 1, 3082, 2049, 2100, 
4100, 10, 3065, 3105, 146, 118, 58, 117, 112, 100, 97, 
116, 101, 0, 4, 1793, 3112, 15, 4, 8, 10, 1, 
3108, 2049, 2110, 16, 10, 3093, 3126, 146, 99, 111, 112, 
121, 0, 1793, 3135, 285278725, 1, 33951492, 268767489, 1, 6, 10, 
1, 3128, 2049, 2275, 771, 10, 3118, 3154, 146, 83, 99, 
111, 112, 101, 76, 105, 115, 116, 0, 12622, 12674, 10, 
3141, 3163, 146, 123, 123, 0, 2049, 1570, 2, 1, 3154, 
2049, 59, 16, 10, 3157, 3188, 146, 45, 45, 45, 114, 
101, 118, 101, 97, 108, 45, 45, 45, 0, 2049, 1570, 
1, 3154, 2049, 2890, 16, 10, 3172, 3202, 146, 125, 125, 
0, 1, 3154, 2049, 56, 4, 15, 11, 1793, 3216, 3841, 
3154, 4097, 2, 10, 1, 3211, 1793, 3242, 3841, 3154, 1793, 
3237, 1, 2, 983567, 1, 3154, 2049, 2890, 1641487, 3, 1, 
3226, 7, 10, 1, 3224, 8, 16, 10, 1, 3220, 2049, 
64, 10, 3196, 3256, 133, 83, 116, 97, 114, 116, 0, 
0, 10, 3247, 3265, 133, 69, 110, 100, 0, 0, 10, 
3258, 3280, 146, 116, 101, 114, 109, 105, 110, 97, 116, 
101, 0, 1, 0, 3841, 3265, 16, 10, 3196, 3302, 146, 
98, 117, 102, 102, 101, 114, 58, 115, 116, 97, 114, 
116, 0, 3841, 3256, 10, 3286, 3319, 146, 98, 117, 102, 
102, 101, 114, 58, 101, 110, 100, 0, 3841, 3265, 10, 
3305, 3336, 146, 98, 117, 102, 102, 101, 114, 58, 97, 
100, 100, 0, 3841, 3265, 16, 1, 3265, 2049, 2988, 2049, 
3280, 10, 3322, 3360, 146, 98, 117, 102, 102, 101, 114, 
58, 103, 101, 116, 0, 1, 3265, 2049, 3003, 3841, 3265, 
15, 2049, 3280, 10, 3346, 3386, 146, 98, 117, 102, 102, 
101, 114, 58, 101, 109, 112, 116, 121, 0, 3841, 3256, 
4097, 3265, 2049, 3280, 10, 3370, 3408, 146, 98, 117, 102, 
102, 101, 114, 58, 115, 105, 122, 101, 0, 3841, 3265, 
3841, 3256, 18, 10, 3393, 3428, 146, 98, 117, 102, 102, 
101, 114, 58, 115, 101, 116, 0, 4097, 3256, 2049, 3386, 
10, 3414, 3452, 146, 98, 117, 102, 102, 101, 114, 58, 
112, 114, 101, 115, 101, 114, 118, 101, 0, 3841, 3256, 
3841, 3265, 1793, 3465, 1, 21, 2049, 2100, 4097, 3256, 10, 
1, 3458, 2049, 2100, 4097, 3265, 10, 3433, 3487, 133, 84, 
101, 109, 112, 83, 116, 114, 105, 110, 103, 115, 0, 
32, 3472, 3505, 133, 84, 101, 109, 112, 83, 116, 114, 
105, 110, 103, 77, 97, 120, 0, 512, 3488, 3517, 146, 
83, 84, 82, 73, 78, 71, 83, 0, 2049, 1543, 3841, 
3487, 3841, 3505, 19, 18, 10, 3506, 3537, 133, 67, 117, 
114, 114, 101, 110, 116, 0, 12, 10, 3526, 3552, 146, 
115, 58, 112, 111, 105, 110, 116, 101, 114, 0, 3841, 
3537, 3841, 3505, 19, 2049, 3517, 17, 10, 3539, 3571, 146, 
115, 58, 110, 101, 120, 116, 0, 1, 3537, 2049, 2988, 
3841, 3537, 3841, 3487, 11, 1793, 3587, 1, 0, 4097, 3537, 
10, 1, 3582, 9, 10, 3506, 3601, 146, 115, 58, 116, 
101, 109, 112, 0, 2, 2049, 81, 2049, 2890, 2049, 3552, 
4, 2049, 3126, 2049, 3552, 2049, 3571, 10, 3591, 3627, 146, 
115, 58, 101, 109, 112, 116, 121, 0, 2049, 3552, 2049, 
3571, 1, 0, 67502597, 16, 10, 3616, 3646, 146, 115, 58, 
115, 107, 105, 112, 0, 6, 1793, 3654, 68223234, 1, 786703, 
0, 10, 1, 3649, 2049, 2227, 2049, 2903, 5, 10, 3636, 
3672, 146, 115, 58, 107, 101, 101, 112, 0, 2049, 1815, 
1793, 3681, 1, 3646, 2049, 1774, 10, 1, 3676, 9, 2049, 
1874, 1, 121, 2049, 2100, 2049, 133, 10, 3662, 3705, 158, 
112, 114, 101, 102, 105, 120, 58, 39, 0, 2049, 1815, 
1, 3672, 1, 3601, 2049, 64, 10, 3693, 3724, 146, 115, 
58, 99, 104, 111, 112, 0, 2049, 3601, 2, 2049, 81, 
67502597, 17, 2049, 2903, 1, 0, 4, 16, 10, 3714, 3751, 
146, 115, 58, 114, 101, 118, 101, 114, 115, 101, 0, 
1793, 3793, 2, 2049, 3601, 2049, 3428, 1, 81, 1793, 3769, 
2, 2049, 81, 17, 2049, 2903, 10, 1, 3762, 2049, 2122, 
4, 1793, 3783, 2, 15, 2049, 3336, 2049, 2903, 10, 1, 
3776, 2049, 2275, 3, 2049, 3302, 2049, 3601, 10, 1, 3753, 
2049, 3452, 10, 3738, 3811, 146, 115, 58, 112, 114, 101, 
112, 101, 110, 100, 0, 2049, 3601, 1793, 3835, 2, 2049, 
81, 17, 1793, 3827, 2, 2049, 81, 2049, 2890, 10, 1, 
3821, 2049, 2100, 4, 2049, 3126, 10, 1, 3815, 2049, 2110, 
10, 3798, 3852, 146, 115, 58, 97, 112, 112, 101, 110, 
100, 0, 4, 2049, 3811, 10, 3840, 3870, 146, 115, 58, 
102, 111, 114, 45, 101, 97, 99, 104, 0, 1793, 3885, 
67502597, 6415, 3, 67502597, 67502597, 251987205, 2054, 101777670, 1, 1, 3872, 
7, 10, 1, 3872, 8, 771, 10, 3856, 3904, 146, 115, 
58, 105, 110, 100, 101, 120, 45, 111, 102, 0, 4, 
1793, 3918, 68223234, 1, 6415, 33883396, 101450758, 6404, 3, 1, 3907, 
7, 10, 1, 3907, 1793, 3927, 18, 2049, 2903, 772, 10, 
1, 3922, 1793, 3936, 2049, 81, 67502597, 11, 10, 1, 3931, 
2049, 2159, 1793, 3946, 3, 1, -1, 10, 1, 3942, 9, 
10, 3890, 3970, 146, 115, 58, 99, 111, 110, 116, 97, 
105, 110, 115, 45, 99, 104, 97, 114, 63, 0, 2049, 
3904, 1, -1, 12, 10, 3950, 3986, 146, 115, 58, 104, 
97, 115, 104, 0, 1, 5381, 4, 1793, 3994, 286458116, 33, 
10, 1, 3991, 2049, 3870, 10, 3976, 4006, 133, 83, 114, 
99, 0, 0, 3999, 4014, 133, 84, 97, 114, 0, 0, 
4007, 4022, 133, 80, 97, 100, 0, 0, 4015, 4028, 133, 
73, 0, 0, 4023, 4034, 133, 70, 0, 0, 4029, 4041, 
133, 65, 116, 0, 0, 4035, 4055, 146, 116, 101, 114, 
109, 105, 110, 97, 116, 101, 0, 1, 0, 3841, 4022, 
3841, 4014, 2049, 81, 17, 16, 10, 4042, 4077, 146, 101, 
120, 116, 114, 97, 99, 116, 0, 3841, 4006, 3841, 4028, 
17, 3841, 4022, 3841, 4014, 2049, 81, 2049, 3126, 10, 4066, 
4102, 146, 99, 111, 109, 112, 97, 114, 101, 0, 3841, 
4022, 3841, 4014, 2049, 95, 3841, 4034, 22, 4097, 4034, 3841, 
4034, 1793, 4122, 3841, 4028, 4097, 4041, 10, 1, 4117, 2049, 
72, 10, 4091, 4135, 146, 110, 101, 120, 116, 0, 1, 
4028, 2049, 2988, 10, 3976, 4162, 146, 115, 58, 99, 111, 
110, 116, 97, 105, 110, 115, 45, 115, 116, 114, 105, 
110, 103, 63, 0, 4097, 4014, 4097, 4006, 2049, 3627, 4097, 
4022, 1, 0, 4097, 4028, 1, 0, 4097, 4034, 3841, 4006, 
2049, 81, 1793, 4193, 2049, 4077, 2049, 4055, 2049, 4102, 2049, 
4135, 10, 1, 4184, 2049, 2275, 3841, 4034, 10, 4140, 4207, 
133, 83, 116, 114, 0, 0, 4200, 4219, 146, 101, 120, 
116, 114, 97, 99, 116, 0, 2049, 2090, 3841, 4207, 4, 
2049, 3126, 3841, -1, 67502597, 17, 1, 0, 4, 16, 10, 
4208, 4244, 146, 99, 104, 101, 99, 107, 0, 1, 4219, 
2049, 2100, 1793, 4255, 1, 2890, 2049, 2100, 10, 1, 4250, 
2049, 2100, 3841, 4207, 2049, 3986, 67502597, 11, 10, 4235, 4278, 
146, 108, 111, 99, 97, 116, 105, 111, 110, 0, 67503109, 
67503109, 1793, 4311, 1793, 4306, 4, 1793, 4292, 67502597, 2049, 2527, 
21, 10, 1, 4287, 2049, 2100, 4, 1793, 4302, 772, 2, 
10, 1, 4299, 9, 10, 1, 4284, 2049, 2100, 10, 1, 
4282, 2049, 2100, 10, 4266, 4325, 146, 115, 101, 116, 117, 
112, 0, 2049, 3627, 4097, 4207, 1, 0, 67503109, 67503109, 1, 
81, 1, 3986, 2049, 2122, 2049, 3627, 2049, 3428, 1793, 4349, 
67502597, 2049, 81, 10, 1, 4345, 2049, 2100, 4, 10, 4140, 
4376, 146, 115, 58, 105, 110, 100, 101, 120, 45, 111, 
102, 45, 115, 116, 114, 105, 110, 103, 0, 67502597, 1793, 
4402, 1793, 4395, 2049, 4325, 1793, 4390, 2049, 4244, 2049, 4278, 
10, 1, 4385, 2049, 2275, 10, 1, 4381, 2049, 3452, 771, 
3, 10, 1, 4379, 2049, 2100, 18, 1, 2, 18, 1, 
-1, 2049, 2828, 10, 4355, 4427, 146, 115, 58, 102, 105, 
108, 116, 101, 114, 0, 1793, 4455, 2049, 3627, 2049, 3428, 
4, 1793, 4447, 2049, 2090, 4, 8, 1, 3336, 1, 11, 
2049, 64, 10, 1, 4436, 2049, 3870, 3, 2049, 3302, 10, 
1, 4429, 2049, 3452, 10, 4415, 4469, 146, 115, 58, 109, 
97, 112, 0, 1793, 4491, 2049, 3627, 2049, 3428, 4, 1793, 
4483, 67502597, 8, 2049, 3336, 10, 1, 4478, 2049, 3870, 3, 
2049, 3302, 10, 1, 4471, 2049, 3452, 10, 4460, 4508, 146, 
115, 58, 115, 117, 98, 115, 116, 114, 0, 1793, 4514, 
17, 2049, 3627, 10, 1, 4510, 2049, 2100, 1793, 4526, 67502597, 
1, 3126, 2049, 2100, 10, 1, 4520, 2049, 2110, 67502597, 1793, 
4539, 17, 1, 0, 4, 16, 10, 1, 4533, 2049, 2100, 
10, 4496, 4555, 146, 115, 58, 114, 105, 103, 104, 116, 
0, 67502597, 2049, 81, 67502597, 18, 4, 2049, 4508, 10, 4544, 
4574, 146, 115, 58, 108, 101, 102, 116, 0, 1, 0, 
4, 2049, 4508, 10, 4564, 4598, 146, 115, 58, 98, 101, 
103, 105, 110, 115, 45, 119, 105, 116, 104, 63, 0, 
2, 2049, 81, 1, 13, 2049, 2100, 2049, 4574, 2049, 95, 
10, 4580, 4626, 146, 115, 58, 101, 110, 100, 115, 45, 
119, 105, 116, 104, 63, 0, 2, 2049, 81, 1, 13, 
2049, 2100, 2049, 4555, 2049, 95, 10, 4610, 4648, 146, 115, 
58, 99, 111, 112, 121, 0, 67502597, 2049, 81, 2049, 2890, 
2049, 3126, 10, 4638, 4668, 146, 115, 58, 68, 73, 71, 
73, 84, 83, 0, 2049, 3646, 48, 49, 50, 51, 52, 
53, 54, 55, 56, 57, 0, 1, 4670, 10, 4656, 4705, 
146, 115, 58, 65, 83, 67, 73, 73, 45, 76, 79, 
87, 69, 82, 67, 65, 83, 69, 0, 2049, 3646, 97, 
98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 
109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 
120, 121, 122, 0, 1, 4707, 10, 4684, 4758, 146, 115, 
58, 65, 83, 67, 73, 73, 45, 85, 80, 80, 69, 
82, 67, 65, 83, 69, 0, 2049, 3646, 65, 66, 67, 
68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 
79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 
90, 0, 1, 4760, 10, 4737, 4809, 146, 115, 58, 65, 
83, 67, 73, 73, 45, 76, 69, 84, 84, 69, 82, 
83, 0, 2049, 3646, 97, 98, 99, 100, 101, 102, 103, 
104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 
115, 116, 117, 118, 119, 120, 121, 122, 65, 66, 67, 
68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 
79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 
90, 0, 1, 4811, 10, 4790, 4884, 146, 115, 58, 80, 
85, 78, 67, 84, 85, 65, 84, 73, 79, 78, 0, 
2049, 3646, 95, 33, 34, 35, 36, 37, 38, 39, 40, 
41, 42, 43, 44, 45, 46, 47, 58, 59, 60, 61, 
62, 63, 64, 91, 92, 93, 94, 96, 123, 124, 125, 
126, 0, 1, 4886, 1, 95, 67502597, 16, 10, 4867, 4942, 
133, 115, 58, 87, 72, 73, 84, 69, 83, 80, 65, 
67, 69, 0, 32, 9, 10, 13, 0, 4926, 4952, 158, 
39, 0, 1, 3627, 2049, 146, 10, 4947, 4962, 158, 36, 
0, 1, 0, 2049, 133, 10, 4957, 0, 133, 65, 83, 
67, 73, 73, 58, 78, 85, 76, 0, 4967, 27, 133, 
65, 83, 67, 73, 73, 58, 69, 83, 67, 0, 4980, 
8, 133, 65, 83, 67, 73, 73, 58, 66, 83, 0, 
4993, 9, 133, 65, 83, 67, 73, 73, 58, 72, 84, 
0, 5005, 10, 133, 65, 83, 67, 73, 73, 58, 76, 
70, 0, 5017, 11, 133, 65, 83, 67, 73, 73, 58, 
86, 84, 0, 5029, 12, 133, 65, 83, 67, 73, 73, 
58, 70, 70, 0, 5041, 13, 133, 65, 83, 67, 73, 
73, 58, 67, 82, 0, 5053, 32, 133, 65, 83, 67, 
73, 73, 58, 83, 80, 65, 67, 69, 0, 5065, 127, 
133, 65, 83, 67, 73, 73, 58, 68, 69, 76, 0, 
5080, 1, 133, 65, 83, 67, 73, 73, 58, 83, 79, 
72, 0, 5093, 2, 133, 65, 83, 67, 73, 73, 58, 
83, 84, 88, 0, 5106, 3, 133, 65, 83, 67, 73, 
73, 58, 69, 84, 88, 0, 5119, 4, 133, 65, 83, 
67, 73, 73, 58, 69, 79, 84, 0, 5132, 5, 133, 
65, 83, 67, 73, 73, 58, 69, 78, 81, 0, 5145, 
6, 133, 65, 83, 67, 73, 73, 58, 65, 67, 75, 
0, 5158, 7, 133, 65, 83, 67, 73, 73, 58, 66, 
69, 76, 0, 5171, 14, 133, 65, 83, 67, 73, 73, 
58, 83, 79, 0, 5184, 15, 133, 65, 83, 67, 73, 
73, 58, 83, 73, 0, 5196, 16, 133, 65, 83, 67, 
73, 73, 58, 68, 76, 69, 0, 5208, 17, 133, 65, 
83, 67, 73, 73, 58, 68, 67, 49, 0, 5221, 18, 
133, 65, 83, 67, 73, 73, 58, 68, 67, 50, 0, 
5234, 19, 133, 65, 83, 67, 73, 73, 58, 68, 67, 
51, 0, 5247, 20, 133, 65, 83, 67, 73, 73, 58, 
68, 67, 52, 0, 5260, 21, 133, 65, 83, 67, 73, 
73, 58, 78, 65, 75, 0, 5273, 22, 133, 65, 83, 
67, 73, 73, 58, 83, 89, 78, 0, 5286, 23, 133, 
65, 83, 67, 73, 73, 58, 69, 84, 66, 0, 5299, 
24, 133, 65, 83, 67, 73, 73, 58, 67, 65, 78, 
0, 5312, 25, 133, 65, 83, 67, 73, 73, 58, 69, 
77, 0, 5325, 26, 133, 65, 83, 67, 73, 73, 58, 
83, 85, 66, 0, 5337, 28, 133, 65, 83, 67, 73, 
73, 58, 70, 83, 0, 5350, 29, 133, 65, 83, 67, 
73, 73, 58, 71, 83, 0, 5362, 30, 133, 65, 83, 
67, 73, 73, 58, 82, 83, 0, 5374, 31, 133, 65, 
83, 67, 73, 73, 58, 85, 83, 0, 5386, 5414, 146, 
99, 58, 108, 111, 119, 101, 114, 99, 97, 115, 101, 
63, 0, 1, 97, 1, 122, 2049, 2921, 10, 5398, 5437, 
146, 99, 58, 117, 112, 112, 101, 114, 99, 97, 115, 
101, 63, 0, 1, 65, 1, 90, 2049, 2921, 10, 5421, 
5457, 146, 99, 58, 108, 101, 116, 116, 101, 114, 63, 
0, 1, 5414, 1, 5437, 2049, 2122, 22, 10, 5444, 5477, 
146, 99, 58, 100, 105, 103, 105, 116, 63, 0, 1, 
48, 1, 57, 2049, 2921, 10, 5465, 5498, 146, 99, 58, 
118, 105, 115, 105, 98, 108, 101, 63, 0, 1, 32, 
1, 126, 2049, 2921, 10, 5484, 5517, 146, 99, 58, 118, 
111, 119, 101, 108, 63, 0, 2049, 3646, 97, 101, 105, 
111, 117, 65, 69, 73, 79, 85, 0, 1, 5519, 4, 
2049, 3970, 10, 5505, 5552, 146, 99, 58, 99, 111, 110, 
115, 111, 110, 97, 110, 116, 63, 0, 2, 2049, 5457, 
1793, 5562, 2049, 5517, 2049, 2459, 10, 1, 5557, 1793, 5570, 
3, 2049, 2362, 10, 1, 5566, 2049, 64, 10, 5536, 5592, 
146, 99, 58, 119, 104, 105, 116, 101, 115, 112, 97, 
99, 101, 63, 0, 1, 4942, 4, 2049, 3970, 10, 5575, 
5615, 146, 99, 58, 45, 108, 111, 119, 101, 114, 99, 
97, 115, 101, 63, 0, 2049, 5414, 2049, 2459, 10, 5598, 
5637, 146, 99, 58, 45, 117, 112, 112, 101, 114, 99, 
97, 115, 101, 63, 0, 2049, 5437, 2049, 2459, 10, 5620, 
5655, 146, 99, 58, 45, 100, 105, 103, 105, 116, 63, 
0, 2049, 5477, 2049, 2459, 10, 5642, 5678, 146, 99, 58, 
45, 119, 104, 105, 116, 101, 115, 112, 97, 99, 101, 
63, 0, 2049, 5592, 2049, 2459, 10, 5660, 5698, 146, 99, 
58, 45, 118, 105, 115, 105, 98, 108, 101, 63, 0, 
2049, 5498, 2049, 2459, 10, 5683, 5716, 146, 99, 58, 45, 
118, 111, 119, 101, 108, 63, 0, 2049, 5517, 2049, 2459, 
10, 5703, 5738, 146, 99, 58, 45, 99, 111, 110, 115, 
111, 110, 97, 110, 116, 63, 0, 2049, 5552, 2049, 2459, 
10, 5721, 5757, 146, 99, 58, 116, 111, 45, 117, 112, 
112, 101, 114, 0, 2, 2049, 5414, 25, 3, 1, 32, 
18, 10, 5743, 5780, 146, 99, 58, 116, 111, 45, 108, 
111, 119, 101, 114, 0, 2, 2049, 5437, 25, 3, 1, 
32, 17, 10, 5766, 5804, 146, 99, 58, 116, 111, 45, 
115, 116, 114, 105, 110, 103, 0, 2049, 3646, 46, 0, 
1, 5806, 2049, 3601, 1, 36, 2049, 2110, 10, 5789, 5834, 
146, 99, 58, 116, 111, 103, 103, 108, 101, 45, 99, 
97, 115, 101, 0, 2, 2049, 5414, 1, 5757, 1, 5780, 
2049, 64, 10, 5817, 5859, 146, 99, 58, 116, 111, 45, 
110, 117, 109, 98, 101, 114, 0, 2, 2049, 5477, 1793, 
5868, 1, 48, 18, 10, 1, 5864, 1793, 5876, 3, 1, 
0, 10, 1, 5872, 2049, 64, 10, 5844, 5895, 146, 115, 
58, 116, 111, 45, 117, 112, 112, 101, 114, 0, 1, 
5757, 2049, 4469, 10, 5881, 5914, 146, 115, 58, 116, 111, 
45, 108, 111, 119, 101, 114, 0, 1, 5780, 2049, 4469, 
10, 5900, 5934, 146, 115, 58, 116, 114, 105, 109, 45, 
108, 101, 102, 116, 0, 2049, 3601, 1793, 5948, 2049, 56, 
1, 5592, 1, 2543, 2049, 2122, 21, 10, 1, 5938, 2049, 
2227, 2049, 2903, 10, 5919, 5971, 146, 115, 58, 116, 114, 
105, 109, 45, 114, 105, 103, 104, 116, 0, 2049, 3601, 
2049, 3751, 2049, 5934, 2049, 3751, 10, 5955, 5990, 146, 115, 
58, 116, 114, 105, 109, 0, 2049, 5971, 2049, 5934, 10, 
5980, 6006, 146, 99, 111, 114, 114, 101, 99, 116, 0, 
2, 1, 48, 13, 1793, 6021, 1, 48, 67502597, 18, 1, 
2, 19, 17, 10, 1, 6012, 9, 10, 5980, 6040, 146, 
110, 58, 116, 111, 45, 115, 116, 114, 105, 110, 103, 
0, 1793, 6088, 2049, 1874, 2049, 3428, 2, 2049, 2855, 1793, 
6066, 1, 10, 20, 4, 1, 48, 17, 2049, 6006, 2049, 
3336, 2, 2049, 2543, 10, 1, 6051, 2049, 2227, 3, 2049, 
2562, 1793, 6080, 1, 45, 2049, 3336, 10, 1, 6075, 9, 
2049, 3302, 2049, 3751, 10, 1, 6042, 2049, 3452, 10, 6025, 
6115, 133, 82, 101, 119, 114, 105, 116, 101, 85, 110, 
100, 101, 114, 115, 99, 111, 114, 101, 115, 0, -1, 
6093, 6123, 146, 115, 117, 98, 0, 1, 95, 1793, 6130, 
1, 32, 10, 1, 6127, 2049, 2373, 10, 6116, 6146, 146, 
114, 101, 119, 114, 105, 116, 101, 0, 3841, 6115, 1793, 
6155, 1, 6123, 2049, 4469, 10, 1, 6150, 9, 10, 6135, 
6169, 146, 104, 97, 110, 100, 108, 101, 0, 1, 3705, 
8, 10, 6093, 6185, 158, 112, 114, 101, 102, 105, 120, 
58, 39, 0, 2049, 6146, 2049, 6169, 10, 6173, 6201, 146, 
115, 58, 115, 112, 108, 105, 116, 0, 2049, 2090, 2049, 
3904, 772, 2049, 2090, 2049, 4574, 1, 38, 2049, 2100, 10, 
6190, 6236, 146, 115, 58, 115, 112, 108, 105, 116, 45, 
111, 110, 45, 115, 116, 114, 105, 110, 103, 0, 2049, 
2090, 2049, 4376, 2049, 2890, 772, 2049, 2090, 2049, 4574, 1, 
38, 2049, 2100, 10, 6215, 6265, 146, 115, 58, 114, 101, 
112, 108, 97, 99, 101, 0, 67502597, 2049, 81, 2049, 1874, 
16, 1793, 6281, 2049, 6236, 4, 2049, 1874, 15, 17, 10, 
1, 6273, 2049, 2100, 2049, 3811, 2049, 3852, 10, 6252, 6302, 
133, 83, 112, 108, 105, 116, 45, 79, 110, 0, 0, 
6290, 6313, 146, 109, 97, 116, 99, 104, 63, 0, 3841, 
6302, 11, 10, 6303, 6330, 146, 116, 101, 114, 109, 105, 
110, 97, 116, 101, 0, 1, 0, 67502597, 2049, 2903, 16, 
10, 6317, 6345, 146, 115, 116, 101, 112, 0, 1, 2890, 
2049, 2100, 2049, 6313, 1793, 6359, 2, 2049, 107, 2049, 6330, 
10, 1, 6353, 9, 10, 6252, 6377, 146, 115, 58, 116, 
111, 107, 101, 110, 105, 122, 101, 0, 4097, 6302, 2049, 
3672, 2049, 1874, 1, 0, 2049, 107, 1793, 6399, 2, 2049, 
107, 2, 1, 6345, 2049, 3870, 3, 10, 1, 6389, 2049, 
2100, 2049, 1874, 67502597, 18, 2049, 2903, 67502597, 16, 10, 6363, 
6422, 133, 78, 101, 101, 100, 108, 101, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 6412, 6557, 133, 76, 101, 110, 
0, 0, 6550, 6568, 133, 84, 111, 107, 101, 110, 115, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 6558, 6702, 133, 
84, 80, 0, 0, 6696, 6711, 146, 115, 97, 118, 101, 
0, 2049, 3672, 3841, 6702, 1, 6568, 17, 2049, 2890, 16, 
1, 6702, 2049, 2988, 10, 6703, 6734, 146, 110, 101, 120, 
116, 0, 1793, 6740, 3841, 6557, 17, 10, 1, 6736, 2049, 
2110, 10, 6726, 6754, 146, 100, 111, 110, 101, 63, 0, 
2049, 81, 2049, 2527, 10, 6363, 6783, 146, 115, 58, 116, 
111, 107, 101, 110, 105, 122, 101, 45, 111, 110, 45, 
115, 116, 114, 105, 110, 103, 0, 1, 0, 4097, 6702, 
1793, 6797, 2, 1, 6422, 2049, 4648, 2049, 3852, 10, 1, 
6789, 1793, 6806, 2049, 81, 4097, 6557, 10, 1, 6801, 2049, 
2122, 1793, 6823, 1, 6422, 2049, 6236, 2049, 6711, 2049, 6734, 
2049, 6754, 10, 1, 6812, 2049, 2250, 1, 6568, 3841, 6702, 
2049, 2903, 4097, 6568, 10, 6759, 6848, 146, 102, 111, 114, 
45, 101, 97, 99, 104, 0, 4, 2049, 56, 1, 13, 
2049, 2100, 1793, 6864, 5, 2049, 56, 84018692, 525572, 1542, 10, 
1, 6857, 2049, 2275, 771, 10, 6836, 6884, 146, 115, 117, 
98, 115, 116, 105, 116, 117, 116, 101, 0, 2049, 3517, 
1, 129, 18, 10, 6870, 6901, 146, 101, 120, 116, 114, 
97, 99, 116, 0, 2049, 6884, 2049, 4648, 10, 6890, 6918, 
146, 116, 111, 107, 101, 110, 105, 122, 101, 0, 2049, 
6783, 2049, 3627, 10, 6906, 6934, 146, 99, 111, 109, 98, 
105, 110, 101, 0, 2049, 6884, 2049, 3852, 2049, 3852, 10, 
6923, 6950, 146, 109, 101, 114, 103, 101, 0, 4, 1, 
6934, 2049, 6848, 772, 10, 6941, 6969, 146, 102, 105, 110, 
100, 45, 101, 110, 100, 0, 2, 2049, 81, 2049, 6884, 
2049, 81, 18, 67502597, 17, 10, 6957, 6989, 146, 99, 108, 
101, 97, 110, 0, 2049, 6969, 1, 0, 4, 16, 10, 
6759, 7013, 146, 115, 58, 114, 101, 112, 108, 97, 99, 
101, 45, 97, 108, 108, 0, 1, 3, 1793, 7028, 2049, 
6901, 2049, 6918, 2049, 6950, 2049, 6989, 2049, 3601, 10, 1, 
7017, 2049, 3079, 10, 6996, 7041, 146, 99, 104, 97, 114, 
0, 1, 32, 1793, 7050, 1, 95, 2049, 3336, 10, 1, 
7045, 2049, 2373, 1, 114, 1793, 7063, 1, 13, 2049, 3336, 
10, 1, 7058, 2049, 2373, 1, 110, 1793, 7076, 1, 10, 
2049, 3336, 10, 1, 7071, 2049, 2373, 1, 116, 1793, 7089, 
1, 9, 2049, 3336, 10, 1, 7084, 2049, 2373, 1, 48, 
1793, 7102, 1, 0, 2049, 3336, 10, 1, 7097, 2049, 2373, 
1, 94, 1793, 7115, 1, 27, 2049, 3336, 10, 1, 7110, 
2049, 2373, 2049, 3336, 10, 7033, 7130, 146, 116, 121, 112, 
101, 0, 1, 99, 1793, 7138, 4, 2049, 3336, 10, 1, 
7134, 2049, 2373, 1, 115, 1793, 7152, 4, 1, 3336, 2049, 
3870, 10, 1, 7146, 2049, 2373, 1, 110, 1793, 7168, 4, 
2049, 6040, 1, 3336, 2049, 3870, 10, 1, 7160, 2049, 2373, 
3, 10, 7122, 7184, 146, 104, 97, 110, 100, 108, 101, 
0, 1, 92, 1793, 7193, 2049, 56, 2049, 7041, 10, 1, 
7188, 2049, 2373, 1, 37, 1793, 7206, 2049, 56, 2049, 7130, 
10, 1, 7201, 2049, 2373, 2049, 3336, 10, 6996, 7225, 146, 
115, 58, 102, 111, 114, 109, 97, 116, 0, 1793, 7254, 
2049, 3627, 1793, 7249, 2049, 3428, 1793, 7244, 2049, 56, 25, 
2049, 7184, 1, 7235, 7, 10, 1, 7235, 8, 3, 10, 
1, 7231, 2049, 2110, 10, 1, 7227, 2049, 3452, 10, 7213, 
7270, 146, 115, 58, 99, 111, 110, 115, 116, 0, 1, 
3672, 2049, 2100, 2049, 2016, 10, 7259, 7287, 133, 86, 97, 
108, 117, 101, 115, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
7277, 7323, 146, 102, 114, 111, 109, 0, 2049, 81, 2, 
1793, 7341, 1793, 7334, 1, 7287, 4113, 10, 1, 7330, 2049, 
2110, 2049, 2903, 10, 1, 7328, 2049, 2275, 3, 10, 7315, 
7353, 146, 116, 111, 0, 2, 2049, 81, 1793, 7369, 2049, 
56, 1, 97, 18, 2049, 2890, 1, 7287, 266001, 10, 1, 
7358, 2049, 2275, 3, 10, 7259, 7386, 146, 114, 101, 111, 
114, 100, 101, 114, 0, 1, 7323, 2049, 2100, 2049, 7353, 
10, 7375, 7402, 146, 99, 117, 114, 114, 121, 0, 2049, 
1874, 1793, 7412, 4, 2049, 1728, 2049, 1751, 10, 1, 7406, 
2049, 2100, 10, 7393, 7425, 146, 100, 111, 101, 115, 0, 
2049, 1587, 4, 2049, 7402, 2049, 1570, 2049, 160, 16, 1, 
146, 2049, 1646, 10, 7417, 7454, 146, 100, 58, 102, 111, 
114, 45, 101, 97, 99, 104, 0, 1, 2, 1793, 7480, 
6415, 2049, 2090, 1793, 7472, 1793, 7467, 2052, 10, 1, 7465, 
2049, 2100, 10, 1, 7463, 2049, 2100, 1, 7458, 7, 10, 
1, 7458, 8, 3, 10, 7440, 7500, 146, 100, 58, 108, 
111, 111, 107, 117, 112, 45, 120, 116, 0, 1, 0, 
4, 1793, 7525, 2049, 2090, 2049, 160, 2831, 1793, 7518, 4, 
1, 2051, 2049, 2100, 10, 1, 7512, 1, 11, 2049, 64, 
10, 1, 7505, 2049, 7454, 3, 10, 7485, 7543, 146, 97, 
58, 108, 101, 110, 103, 116, 104, 0, 15, 10, 7531, 
7566, 146, 97, 58, 99, 111, 117, 110, 116, 101, 100, 
45, 114, 101, 115, 117, 108, 116, 115, 0, 8, 2049, 
1874, 1793, 7579, 2, 2049, 107, 1, 107, 2049, 2275, 10, 
1, 7571, 2049, 2100, 10, 7545, 7601, 146, 97, 58, 102, 
114, 111, 109, 45, 115, 116, 114, 105, 110, 103, 0, 
2049, 1874, 1793, 7615, 2, 2049, 81, 2049, 107, 1, 107, 
2049, 3870, 10, 1, 7605, 2049, 2100, 10, 7584, 7634, 146, 
97, 58, 102, 111, 114, 45, 101, 97, 99, 104, 0, 
4, 2049, 56, 1, 13, 2049, 2100, 1793, 7650, 5, 2049, 
56, 84018692, 525572, 1542, 10, 1, 7643, 2049, 2275, 771, 10, 
7620, 7665, 146, 97, 58, 100, 117, 112, 0, 2049, 1874, 
1793, 7678, 2, 15, 2049, 107, 1, 107, 2049, 7634, 10, 
1, 7669, 2049, 2100, 10, 7656, 7693, 146, 97, 58, 99, 
111, 112, 121, 0, 1, 3, 1793, 7709, 4097, 3, 2, 
2049, 7543, 2049, 107, 1, 107, 2049, 7634, 10, 1, 7697, 
2049, 3079, 10, 7683, 7729, 146, 97, 58, 116, 111, 45, 
115, 116, 114, 105, 110, 103, 0, 1, 3, 1793, 7742, 
2049, 7665, 1, 0, 2049, 107, 2049, 2890, 10, 1, 7733, 
2049, 3079, 2049, 3601, 10, 7714, 7761, 146, 97, 58, 97, 
112, 112, 101, 110, 100, 0, 2049, 2090, 1, 34, 2049, 
2148, 17, 2049, 1874, 1793, 7786, 2049, 107, 1793, 7781, 1, 
107, 2049, 7634, 10, 1, 7776, 2049, 2148, 10, 1, 7772, 
2049, 2100, 10, 7749, 7804, 146, 97, 58, 112, 114, 101, 
112, 101, 110, 100, 0, 4, 2049, 7761, 10, 7791, 7818, 
146, 97, 58, 99, 104, 111, 112, 0, 2049, 7665, 1, 
-1, 2049, 3060, 2, 2049, 3003, 10, 7808, 7840, 146, 97, 
58, 102, 105, 108, 116, 101, 114, 0, 1793, 7855, 67502597, 
1, 21, 2049, 2100, 4, 1, 107, 1, 11, 2049, 64, 
10, 1, 7842, 2049, 7402, 2049, 1874, 1793, 7870, 67502597, 15, 
2049, 107, 2049, 7634, 10, 1, 7863, 2049, 2100, 2049, 1874, 
67502597, 18, 2049, 2903, 67502597, 16, 10, 7828, 7898, 146, 97, 
58, 99, 111, 110, 116, 97, 105, 110, 115, 63, 0, 
1, 0, 4, 1793, 7910, 4, 5, 67502597, 11, 6, 22, 
10, 1, 7903, 2049, 7634, 772, 10, 7883, 7938, 146, 97, 
58, 99, 111, 110, 116, 97, 105, 110, 115, 45, 115, 
116, 114, 105, 110, 103, 63, 0, 1, 0, 4, 1793, 
7951, 4, 5, 67502597, 2049, 95, 6, 22, 10, 1, 7943, 
2049, 7634, 772, 10, 7916, 7966, 146, 97, 58, 109, 97, 
112, 0, 1793, 7972, 8, 2049, 107, 10, 1, 7968, 2049, 
7402, 2049, 1874, 1793, 7987, 67502597, 15, 2049, 107, 2049, 7634, 
10, 1, 7980, 2049, 2100, 10, 7957, 8005, 146, 97, 58, 
114, 101, 118, 101, 114, 115, 101, 0, 2049, 1874, 1793, 
8039, 2049, 56, 1793, 8017, 17, 2049, 2903, 10, 1, 8013, 
2049, 2110, 2, 2049, 107, 1793, 8033, 2, 15, 2049, 107, 
2049, 2903, 10, 1, 8026, 2049, 2275, 3, 10, 1, 8009, 
2049, 2100, 10, 7992, 8052, 146, 97, 58, 116, 104, 0, 
17, 2049, 2890, 10, 8044, 8067, 146, 97, 58, 102, 101, 
116, 99, 104, 0, 2049, 8052, 15, 10, 8056, 8082, 146, 
97, 58, 115, 116, 111, 114, 101, 0, 2049, 8052, 16, 
10, 8071, 8098, 146, 97, 58, 114, 101, 100, 117, 99, 
101, 0, 1, 13, 2049, 2100, 2049, 7634, 10, 8086, 8117, 
146, 105, 100, 101, 110, 116, 105, 102, 121, 0, 1, 
-1, 4, 1, 0, 1793, 8148, 2049, 2350, 11, 1793, 8142, 
67502597, 1, -1, 11, 1793, 8138, 772, 2, 10, 1, 8135, 
9, 10, 1, 8129, 9, 2049, 2890, 10, 1, 8124, 2049, 
8098, 3, 10, 8086, 8168, 146, 97, 58, 105, 110, 100, 
101, 120, 45, 111, 102, 0, 1, 3, 1793, 8181, 1, 
26, 2049, 7402, 2049, 7966, 2049, 8117, 10, 1, 8172, 2049, 
3079, 10, 8154, 8207, 146, 97, 58, 105, 110, 100, 101, 
120, 45, 111, 102, 45, 115, 116, 114, 105, 110, 103, 
0, 1, 3, 1793, 8220, 1, 95, 2049, 7402, 2049, 7966, 
2049, 8117, 10, 1, 8211, 2049, 3079, 10, 8186, 8235, 146, 
97, 58, 109, 97, 107, 101, 0, 2049, 7566, 2, 2, 
1, 3, 1793, 8246, 2049, 8005, 10, 1, 8243, 2049, 3079, 
4, 2049, 7693, 10, 8225, 8259, 158, 123, 0, 1, 289, 
2049, 158, 1, 1556, 2049, 146, 1, 289, 2049, 158, 10, 
8254, 8277, 158, 125, 0, 1, 305, 2049, 158, 1, 2100, 
2049, 146, 1, 1556, 2049, 146, 1, 13, 2049, 152, 1, 
40, 2049, 152, 1, 2903, 2049, 146, 1, 305, 2049, 158, 
1, 8235, 2049, 146, 10, 8272, 8321, 146, 98, 111, 117, 
110, 100, 115, 63, 0, 67502597, 2049, 7543, 67502597, 13, 10, 
8310, 8335, 146, 99, 111, 112, 121, 0, 2049, 56, 2049, 
107, 10, 8327, 8350, 146, 116, 111, 45, 101, 110, 100, 
0, 2, 2049, 7543, 17, 2049, 2890, 10, 8272, 8367, 146, 
97, 58, 108, 101, 102, 116, 0, 2049, 8321, 1793, 8375, 
771, 1, -1, 10, 1, 8371, 2049, 2653, 2049, 1874, 67502597, 
2049, 107, 1793, 8396, 1, 2890, 2049, 2100, 1, 8335, 2049, 
2275, 3, 10, 1, 8386, 2049, 2100, 10, 8357, 8412, 146, 
97, 58, 114, 105, 103, 104, 116, 0, 2049, 8321, 1793, 
8420, 771, 1, -1, 10, 1, 8416, 2049, 2653, 2049, 1874, 
67502597, 2049, 107, 1793, 8443, 4, 2049, 8350, 67502597, 18, 4, 
1, 8335, 2049, 2275, 3, 10, 1, 8431, 2049, 2100, 10, 
8401, 8460, 146, 97, 58, 109, 105, 100, 100, 108, 101, 
0, 1, 2042, 2049, 2100, 4, 67502597, 2049, 8321, 1793, 8475, 
771, 3, 1, -1, 10, 1, 8470, 2049, 2653, 771, 2049, 
2090, 4, 18, 2049, 2890, 2049, 1874, 67502597, 2049, 107, 1793, 
8510, 772, 1793, 8500, 17, 2049, 2890, 10, 1, 8496, 2049, 
2100, 1, 8335, 2049, 2275, 3, 10, 1, 8493, 2049, 2100, 
10, 8448, 8523, 133, 70, 108, 97, 103, 0, 0, 8515, 
8535, 146, 99, 111, 109, 112, 97, 114, 101, 0, 67440386, 
184946434, 10, 8524, 8548, 146, 108, 101, 110, 103, 116, 104, 
0, 659202, 10, 8538, 8558, 146, 110, 101, 120, 116, 0, 
17043713, 1, 1, 2577, 10, 8550, 8576, 146, 110, 111, 116, 
45, 101, 113, 117, 97, 108, 0, 50529030, 2561, 0, 10, 
8563, 8588, 146, 108, 111, 111, 112, 0, 524549, 8558, 2049, 
8535, 18157313, 8523, 8523, 16, 420610310, 1, 1, 8588, 7, 10, 
8448, 8611, 146, 97, 58, 101, 113, 63, 0, 1048833, -1, 
8523, 2049, 8535, 151066369, -1, 8576, 2049, 8548, 2049, 8588, 251724547, 
8523, 10, 8602, 8636, 146, 97, 58, 45, 101, 113, 63, 
0, 2049, 8611, 2049, 2459, 10, 8626, 8659, 146, 97, 58, 
98, 101, 103, 105, 110, 115, 45, 119, 105, 116, 104, 
63, 0, 1, 3, 1793, 8675, 2, 2049, 7543, 1, 13, 
2049, 2100, 2049, 8367, 2049, 8611, 10, 1, 8663, 2049, 3079, 
10, 8641, 8696, 146, 97, 58, 101, 110, 100, 115, 45, 
119, 105, 116, 104, 63, 0, 1, 3, 1793, 8712, 2, 
2049, 7543, 1, 13, 2049, 2100, 2049, 8412, 2049, 8611, 10, 
1, 8700, 2049, 3079, 10, 8680, 8733, 146, 99, 117, 114, 
114, 101, 110, 116, 45, 108, 105, 110, 101, 0, 2049, 
3517, 1, 1025, 18, 10, 8717, 8755, 146, 99, 111, 117, 
110, 116, 45, 116, 111, 107, 101, 110, 115, 0, 1793, 
8761, 1, 32, 11, 10, 1, 8757, 2049, 4427, 2049, 81, 
10, 8739, 8786, 146, 112, 114, 111, 99, 101, 115, 115, 
45, 116, 111, 107, 101, 110, 115, 0, 1793, 8814, 1, 
32, 2049, 6201, 4, 1793, 8807, 2, 2049, 81, 2049, 2543, 
1, 369, 1, 11, 2049, 64, 10, 1, 8795, 2049, 2100, 
2049, 2890, 10, 1, 8788, 2049, 2275, 2049, 369, 10, 8680, 
8835, 146, 115, 58, 101, 118, 97, 108, 117, 97, 116, 
101, 0, 2049, 8733, 2049, 4648, 2049, 8733, 2, 2049, 8755, 
2049, 8786, 10, 8821, 8853, 133, 76, 80, 0, 0, 8847, 
8863, 133, 73, 110, 100, 101, 120, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 8854, 8903, 146, 110, 
101, 120, 116, 0, 3841, 8853, 1, 8863, 17, 2049, 2988, 
10, 8895, 8919, 146, 112, 114, 101, 112, 0, 1, 8853, 
2049, 2988, 1, 0, 3841, 8853, 1, 8863, 17, 16, 10, 
8911, 8940, 146, 100, 111, 110, 101, 0, 1, 8853, 2049, 
3003, 10, 8821, 8950, 146, 73, 0, 3841, 8853, 1, 8863, 
17, 15, 10, 8945, 8962, 146, 74, 0, 3841, 8853, 1, 
8863, 17, 2049, 2903, 15, 10, 8957, 8976, 146, 75, 0, 
3841, 8853, 1, 8863, 17, 1, 2, 18, 15, 10, 8971, 
9003, 146, 105, 110, 100, 101, 120, 101, 100, 45, 116, 
105, 109, 101, 115, 0, 2049, 8919, 4, 1793, 9019, 25, 
33886721, 1, 2053, 1542, 2049, 8903, 1, 9008, 7, 10, 1, 
9008, 8, 3, 2049, 8940, 10, 8986, 9034, 158, 104, 111, 
111, 107, 0, 2049, 3646, 108, 105, 106, 117, 46, 46, 
46, 46, 0, 1, 9036, 2049, 416, 2049, 1874, 2049, 2890, 
2049, 107, 10, 9026, 9068, 146, 115, 101, 116, 45, 104, 
111, 111, 107, 0, 2049, 2890, 16, 10, 9056, 9082, 146, 
117, 110, 104, 111, 111, 107, 0, 2049, 2890, 2, 2049, 
2890, 4, 16, 10, 9072, 9106, 146, 105, 111, 58, 101, 
110, 117, 109, 101, 114, 97, 116, 101, 0, 27, 10, 
9090, 9120, 146, 105, 111, 58, 113, 117, 101, 114, 121, 
0, 28, 10, 9108, 9135, 146, 105, 111, 58, 105, 110, 
118, 111, 107, 101, 0, 29, 10, 9122, 9152, 146, 105, 
111, 58, 115, 99, 97, 110, 45, 102, 111, 114, 0, 
1, -1, 4, 2049, 9106, 1793, 9183, 2049, 8950, 2049, 9120, 
772, 67502597, 11, 1793, 9179, 1793, 9174, 3, 2049, 8950, 10, 
1, 9170, 2049, 2100, 10, 1, 9168, 9, 10, 1, 9159, 
2049, 9003, 3, 10, 9137, 9198, 146, 99, 58, 112, 117, 
116, 0, 1793, 9200, 1, 0, 2049, 9135, 10, 9189, 9211, 
146, 110, 108, 0, 1, 10, 2049, 9198, 10, 9205, 9222, 
146, 115, 112, 0, 1, 32, 2049, 9198, 10, 9216, 9234, 
146, 116, 97, 98, 0, 1, 9, 2049, 9198, 10, 9227, 
9248, 146, 115, 58, 112, 117, 116, 0, 1, 9198, 2049, 
3870, 10, 9239, 9262, 146, 110, 58, 112, 117, 116, 0, 
2049, 6040, 2049, 9248, 10, 9253, 9276, 146, 114, 101, 115, 
101, 116, 0, 2049, 1556, 25, 771, 1, 9276, 7, 10, 
9267, 9298, 146, 100, 117, 109, 112, 45, 115, 116, 97, 
99, 107, 0, 2049, 1556, 25, 134284547, 9298, 134283782, 9262, 2049, 
9222, 10, 9284, 9316, 146, 70, 82, 69, 69, 0, 2049, 
3517, 1, 1025, 18, 2049, 1874, 18, 10, 9308, 9334, 133, 
67, 108, 111, 99, 107, 0, 0, 9325, 9347, 146, 105, 
100, 101, 110, 116, 105, 102, 121, 0, 3841, 9334, 2049, 
2527, 25, 3, 1, 5, 2049, 9152, 2, 2049, 2562, 1793, 
9402, 3, 2049, 3646, 73, 79, 32, 68, 69, 86, 73, 
67, 69, 32, 84, 89, 80, 69, 32, 48, 48, 48, 
53, 32, 78, 79, 84, 32, 70, 79, 85, 78, 68, 
0, 1, 9365, 2049, 9248, 2049, 9211, 10, 1, 9362, 1793, 
9409, 4097, 9334, 10, 1, 9406, 2049, 64, 10, 9308, 9433, 
146, 99, 108, 111, 99, 107, 58, 111, 112, 101, 114, 
97, 116, 105, 111, 110, 0, 2049, 9347, 3841, 9334, 2049, 
9135, 10, 9414, 9459, 146, 99, 108, 111, 99, 107, 58, 
116, 105, 109, 101, 115, 116, 97, 109, 112, 0, 1, 
0, 2049, 9433, 10, 9440, 9477, 146, 99, 108, 111, 99, 
107, 58, 100, 97, 121, 0, 1, 1, 2049, 9433, 10, 
9464, 9497, 146, 99, 108, 111, 99, 107, 58, 109, 111, 
110, 116, 104, 0, 1, 2, 2049, 9433, 10, 9482, 9516, 
146, 99, 108, 111, 99, 107, 58, 121, 101, 97, 114, 
0, 1, 3, 2049, 9433, 10, 9502, 9535, 146, 99, 108, 
111, 99, 107, 58, 104, 111, 117, 114, 0, 1, 4, 
2049, 9433, 10, 9521, 9556, 146, 99, 108, 111, 99, 107, 
58, 109, 105, 110, 117, 116, 101, 0, 1, 5, 2049, 
9433, 10, 9540, 9577, 146, 99, 108, 111, 99, 107, 58, 
115, 101, 99, 111, 110, 100, 0, 1, 6, 2049, 9433, 
10, 9561, 9599, 146, 99, 108, 111, 99, 107, 58, 117, 
116, 99, 58, 100, 97, 121, 0, 1, 7, 2049, 9433, 
10, 9582, 9623, 146, 99, 108, 111, 99, 107, 58, 117, 
116, 99, 58, 109, 111, 110, 116, 104, 0, 1, 8, 
2049, 9433, 10, 9604, 9646, 146, 99, 108, 111, 99, 107, 
58, 117, 116, 99, 58, 121, 101, 97, 114, 0, 1, 
9, 2049, 9433, 10, 9628, 9669, 146, 99, 108, 111, 99, 
107, 58, 117, 116, 99, 58, 104, 111, 117, 114, 0, 
1, 10, 2049, 9433, 10, 9651, 9694, 146, 99, 108, 111, 
99, 107, 58, 117, 116, 99, 58, 109, 105, 110, 117, 
116, 101, 0, 1, 11, 2049, 9433, 10, 9674, 9719, 146, 
99, 108, 111, 99, 107, 58, 117, 116, 99, 58, 115, 
101, 99, 111, 110, 100, 0, 1, 12, 2049, 9433, 10, 
9699, 9733, 133, 70, 105, 108, 101, 115, 0, 0, 9724, 
9746, 146, 105, 100, 101, 110, 116, 105, 102, 121, 0, 
3841, 9733, 2049, 2527, 25, 3, 1, 4, 2049, 9152, 2, 
2049, 2562, 1793, 9801, 3, 2049, 3646, 73, 79, 32, 68, 
69, 86, 73, 67, 69, 32, 84, 89, 80, 69, 32, 
48, 48, 48, 52, 32, 78, 79, 84, 32, 70, 79, 
85, 78, 68, 0, 1, 9764, 2049, 9248, 2049, 9211, 10, 
1, 9761, 1793, 9808, 4097, 9733, 10, 1, 9805, 2049, 64, 
10, 9699, 9831, 146, 102, 105, 108, 101, 58, 111, 112, 
101, 114, 97, 116, 105, 111, 110, 0, 2049, 9746, 3841, 
9733, 2049, 9135, 10, 9813, 0, 133, 102, 105, 108, 101, 
58, 82, 0, 9838, 1, 133, 102, 105, 108, 101, 58, 
87, 0, 9848, 2, 133, 102, 105, 108, 101, 58, 65, 
0, 9858, 3, 133, 102, 105, 108, 101, 58, 82, 43, 
0, 9868, 9892, 146, 102, 105, 108, 101, 58, 111, 112, 
101, 110, 0, 1, 0, 2049, 9831, 10, 9879, 9911, 146, 
102, 105, 108, 101, 58, 99, 108, 111, 115, 101, 0, 
1, 1, 2049, 9831, 10, 9897, 9929, 146, 102, 105, 108, 
101, 58, 114, 101, 97, 100, 0, 1, 2, 2049, 9831, 
10, 9916, 9948, 146, 102, 105, 108, 101, 58, 119, 114, 
105, 116, 101, 0, 1, 3, 2049, 9831, 10, 9934, 9966, 
146, 102, 105, 108, 101, 58, 116, 101, 108, 108, 0, 
1, 4, 2049, 9831, 10, 9953, 9984, 146, 102, 105, 108, 
101, 58, 115, 101, 101, 107, 0, 1, 5, 2049, 9831, 
10, 9971, 10002, 146, 102, 105, 108, 101, 58, 115, 105, 
122, 101, 0, 1, 6, 2049, 9831, 10, 9989, 10022, 146, 
102, 105, 108, 101, 58, 100, 101, 108, 101, 116, 101, 
0, 1, 7, 2049, 9831, 10, 10007, 10041, 146, 102, 105, 
108, 101, 58, 102, 108, 117, 115, 104, 0, 1, 8, 
2049, 9831, 10, 10027, 10062, 146, 102, 105, 108, 101, 58, 
101, 120, 105, 115, 116, 115, 63, 0, 1, 0, 2049, 
9892, 2, 2049, 2543, 1793, 10076, 2049, 9911, 2049, 2350, 10, 
1, 10071, 1793, 10084, 3, 2049, 2362, 10, 1, 10080, 2049, 
64, 10, 10046, 10114, 146, 102, 105, 108, 101, 58, 111, 
112, 101, 110, 45, 102, 111, 114, 45, 114, 101, 97, 
100, 105, 110, 103, 0, 1, 0, 2049, 9892, 2, 2049, 
10002, 4, 10, 10089, 10147, 146, 102, 105, 108, 101, 58, 
111, 112, 101, 110, 45, 102, 111, 114, 45, 97, 112, 
112, 101, 110, 100, 0, 1, 2, 2049, 9892, 2, 2049, 
10002, 4, 10, 10123, 10181, 146, 102, 105, 108, 101, 58, 
111, 112, 101, 110, 45, 102, 111, 114, 45, 119, 114, 
105, 116, 105, 110, 103, 0, 1, 1, 2049, 9892, 10, 
10156, 10193, 133, 70, 73, 68, 0, 0, 10186, 10202, 133, 
83, 105, 122, 101, 0, 0, 10194, 10213, 133, 65, 99, 
116, 105, 111, 110, 0, 0, 10203, 10224, 133, 66, 117, 
102, 102, 101, 114, 0, 0, 10214, 10234, 146, 45, 101, 
111, 102, 63, 0, 3841, 10193, 2049, 9966, 3841, 10202, 13, 
10, 10225, 10254, 146, 112, 114, 101, 115, 101, 114, 118, 
101, 0, 1, 10193, 1793, 10265, 1, 10202, 1, 21, 2049, 
3079, 10, 1, 10258, 2049, 3079, 10, 10156, 10288, 146, 102, 
105, 108, 101, 58, 114, 101, 97, 100, 45, 108, 105, 
110, 101, 0, 4097, 10193, 1793, 10345, 2049, 1874, 2, 4097, 
10224, 2049, 3428, 1793, 10337, 3841, 10193, 2049, 9929, 2, 2049, 
3336, 1793, 10314, 1, 13, 11, 10, 1, 10310, 1793, 10322, 
1, 10, 11, 10, 1, 10318, 1793, 10330, 1, 0, 11, 
10, 1, 10326, 2049, 2159, 22, 22, 10, 1, 10301, 2049, 
2250, 2049, 3360, 3, 10, 1, 10292, 2049, 3452, 3841, 10224, 
10, 10270, 10374, 146, 102, 105, 108, 101, 58, 102, 111, 
114, 45, 101, 97, 99, 104, 45, 108, 105, 110, 101, 
0, 1793, 10405, 4097, 10213, 2049, 10114, 4097, 10193, 4097, 10202, 
1793, 10396, 3841, 10193, 2049, 10288, 3841, 10213, 8, 2049, 10234, 
10, 1, 10386, 2049, 2227, 3841, 10193, 2049, 9911, 10, 1, 
10376, 2049, 10254, 10, 10352, 10417, 133, 70, 73, 68, 0, 
0, 10352, 10432, 146, 102, 105, 108, 101, 58, 115, 108, 
117, 114, 112, 0, 1793, 10459, 4, 2049, 3428, 2049, 10114, 
4097, 10417, 1793, 10450, 3841, 10417, 2049, 9929, 2049, 3336, 10, 
1, 10443, 2049, 2275, 3841, 10417, 2049, 9911, 10, 1, 10434, 
2049, 3452, 10, 10418, 10477, 146, 102, 105, 108, 101, 58, 
115, 112, 101, 119, 0, 2049, 10181, 4, 1793, 10486, 67502597, 
2049, 9948, 10, 1, 10482, 2049, 3870, 2049, 9911, 10, 10464, 
10500, 133, 70, 80, 85, 0, 0, 10493, 10513, 146, 105, 
100, 101, 110, 116, 105, 102, 121, 0, 3841, 10500, 2049, 
2527, 25, 3, 1, 2, 2049, 9152, 2, 2049, 2562, 1793, 
10568, 3, 2049, 3646, 73, 79, 32, 68, 69, 86, 73, 
67, 69, 32, 84, 89, 80, 69, 32, 48, 48, 48, 
50, 32, 78, 79, 84, 32, 70, 79, 85, 78, 68, 
0, 1, 10531, 2049, 9248, 2049, 9211, 10, 1, 10528, 1793, 
10575, 4097, 10500, 10, 1, 10572, 2049, 64, 10, 10464, 10599, 
146, 102, 108, 111, 97, 116, 58, 111, 112, 101, 114, 
97, 116, 105, 111, 110, 0, 2049, 10513, 3841, 10500, 2049, 
9135, 10, 10580, 10620, 146, 110, 58, 116, 111, 45, 102, 
108, 111, 97, 116, 0, 1, 0, 2049, 10599, 10, 10606, 
10639, 146, 115, 58, 116, 111, 45, 102, 108, 111, 97, 
116, 0, 1, 1, 2049, 10599, 10, 10625, 10659, 146, 102, 
58, 116, 111, 45, 110, 117, 109, 98, 101, 114, 0, 
1, 2, 2049, 10599, 10, 10644, 10679, 146, 102, 58, 116, 
111, 45, 115, 116, 114, 105, 110, 103, 0, 2049, 3627, 
2, 1, 3, 2049, 10599, 10, 10664, 10694, 146, 102, 58, 
43, 0, 1, 4, 2049, 10599, 10, 10687, 10706, 146, 102, 
58, 45, 0, 1, 5, 2049, 10599, 10, 10699, 10718, 146, 
102, 58, 42, 0, 1, 6, 2049, 10599, 10, 10711, 10730, 
146, 102, 58, 47, 0, 1, 7, 2049, 10599, 10, 10723, 
10746, 146, 102, 58, 102, 108, 111, 111, 114, 0, 1, 
8, 2049, 10599, 10, 10735, 10764, 146, 102, 58, 99, 101, 
105, 108, 105, 110, 103, 0, 1, 9, 2049, 10599, 10, 
10751, 10779, 146, 102, 58, 115, 113, 114, 116, 0, 1, 
10, 2049, 10599, 10, 10769, 10793, 146, 102, 58, 101, 113, 
63, 0, 1, 11, 2049, 10599, 10, 10784, 10808, 146, 102, 
58, 45, 101, 113, 63, 0, 1, 12, 2049, 10599, 10, 
10798, 10822, 146, 102, 58, 108, 116, 63, 0, 1, 13, 
2049, 10599, 10, 10813, 10836, 146, 102, 58, 103, 116, 63, 
0, 1, 14, 2049, 10599, 10, 10827, 10852, 146, 102, 58, 
100, 101, 112, 116, 104, 0, 1, 15, 2049, 10599, 10, 
10841, 10866, 146, 102, 58, 100, 117, 112, 0, 1, 16, 
2049, 10599, 10, 10857, 10881, 146, 102, 58, 100, 114, 111, 
112, 0, 1, 17, 2049, 10599, 10, 10871, 10896, 146, 102, 
58, 115, 119, 97, 112, 0, 1, 18, 2049, 10599, 10, 
10886, 10910, 146, 102, 58, 108, 111, 103, 0, 1, 19, 
2049, 10599, 10, 10901, 10926, 146, 102, 58, 112, 111, 119, 
101, 114, 0, 1, 20, 2049, 10599, 10, 10915, 10940, 146, 
102, 58, 115, 105, 110, 0, 1, 21, 2049, 10599, 10, 
10931, 10954, 146, 102, 58, 99, 111, 115, 0, 1, 22, 
2049, 10599, 10, 10945, 10968, 146, 102, 58, 116, 97, 110, 
0, 1, 23, 2049, 10599, 10, 10959, 10983, 146, 102, 58, 
97, 115, 105, 110, 0, 1, 24, 2049, 10599, 10, 10973, 
10998, 146, 102, 58, 97, 99, 111, 115, 0, 1, 25, 
2049, 10599, 10, 10988, 11013, 146, 102, 58, 97, 116, 97, 
110, 0, 1, 26, 2049, 10599, 10, 11003, 11028, 146, 102, 
58, 112, 117, 115, 104, 0, 1, 27, 2049, 10599, 10, 
11018, 11042, 146, 102, 58, 112, 111, 112, 0, 1, 28, 
2049, 10599, 10, 11033, 11059, 146, 102, 58, 97, 100, 101, 
112, 116, 104, 0, 1, 29, 2049, 10599, 10, 11047, 11074, 
146, 102, 58, 111, 118, 101, 114, 0, 2049, 11028, 2049, 
10866, 2049, 11042, 2049, 10896, 10, 11064, 11093, 146, 102, 58, 
116, 117, 99, 107, 0, 2049, 10866, 2049, 11028, 2049, 10896, 
2049, 11042, 10, 11083, 11111, 146, 102, 58, 110, 105, 112, 
0, 2049, 10896, 2049, 10881, 10, 11102, 11131, 146, 102, 58, 
100, 114, 111, 112, 45, 112, 97, 105, 114, 0, 2049, 
10881, 2049, 10881, 10, 11116, 11150, 146, 102, 58, 100, 117, 
112, 45, 112, 97, 105, 114, 0, 2049, 11074, 2049, 11074, 
10, 11136, 11164, 146, 102, 58, 114, 111, 116, 0, 2049, 
11028, 2049, 10896, 2049, 11042, 2049, 10896, 10, 11155, 11185, 158, 
112, 114, 101, 102, 105, 120, 58, 46, 0, 2049, 1815, 
1, 3672, 1, 3601, 2049, 64, 1, 10639, 2049, 146, 10, 
11173, 11210, 146, 102, 58, 115, 113, 117, 97, 114, 101, 
0, 2049, 10866, 2049, 10718, 10, 11198, 11230, 146, 102, 58, 
112, 111, 115, 105, 116, 105, 118, 101, 63, 0, 1, 
0, 2049, 10620, 2049, 10836, 10, 11215, 11252, 146, 102, 58, 
110, 101, 103, 97, 116, 105, 118, 101, 63, 0, 1, 
0, 2049, 10620, 2049, 10822, 10, 11237, 11271, 146, 102, 58, 
110, 101, 103, 97, 116, 101, 0, 1, -1, 2049, 10620, 
2049, 10718, 10, 11259, 11287, 146, 102, 58, 97, 98, 115, 
0, 2049, 10866, 2049, 11252, 1, 11271, 9, 10, 11278, 11304, 
146, 102, 58, 112, 117, 116, 0, 2049, 10679, 2049, 9248, 
10, 11295, 11317, 146, 102, 58, 80, 73, 0, 2049, 3646, 
51, 46, 49, 52, 49, 53, 57, 50, 0, 1, 11319, 
2049, 10639, 10, 11309, 11340, 146, 102, 58, 69, 0, 2049, 
3646, 50, 46, 55, 49, 56, 50, 56, 49, 0, 1, 
11342, 2049, 10639, 10, 11333, 11365, 146, 102, 58, 78, 65, 
78, 0, 2049, 3646, 48, 0, 1, 11367, 2049, 10639, 2049, 
3646, 48, 0, 1, 11375, 2049, 10639, 2049, 10730, 10, 11356, 
11393, 146, 102, 58, 73, 78, 70, 0, 2049, 3646, 49, 
46, 48, 0, 1, 11395, 2049, 10639, 2049, 3646, 48, 0, 
1, 11405, 2049, 10639, 2049, 10730, 10, 11384, 11424, 146, 102, 
58, 45, 73, 78, 70, 0, 2049, 3646, 45, 49, 46, 
48, 0, 1, 11426, 2049, 10639, 2049, 3646, 48, 0, 1, 
11437, 2049, 10639, 2049, 10730, 10, 11414, 11456, 146, 102, 58, 
110, 97, 110, 63, 0, 2049, 10866, 2049, 10808, 10, 11446, 
11471, 146, 102, 58, 105, 110, 102, 63, 0, 2049, 11393, 
2049, 10793, 10, 11461, 11487, 146, 102, 58, 45, 105, 110, 
102, 63, 0, 2049, 11424, 2049, 10793, 10, 11476, 11503, 146, 
102, 58, 114, 111, 117, 110, 100, 0, 2049, 10866, 2049, 
11252, 1793, 11524, 2049, 3646, 48, 46, 53, 0, 1, 11511, 
2049, 10639, 2049, 10706, 2049, 10764, 10, 1, 11509, 1793, 11543, 
2049, 3646, 48, 46, 53, 0, 1, 11530, 2049, 10639, 2049, 
10694, 2049, 10746, 10, 1, 11528, 2049, 64, 10, 11492, 11557, 
146, 102, 58, 109, 105, 110, 0, 2049, 11150, 2049, 10822, 
1, 10881, 1, 11111, 2049, 64, 10, 11548, 11577, 146, 102, 
58, 109, 97, 120, 0, 2049, 11150, 2049, 10836, 1, 10881, 
1, 11111, 2049, 64, 10, 11568, 11599, 146, 102, 58, 108, 
105, 109, 105, 116, 0, 2049, 10896, 2049, 11028, 2049, 11557, 
2049, 11042, 2049, 11577, 10, 11588, 11624, 146, 102, 58, 98, 
101, 116, 119, 101, 101, 110, 63, 0, 2049, 11164, 2049, 
10866, 2049, 11028, 2049, 11164, 2049, 11164, 2049, 11599, 2049, 11042, 
2049, 10793, 10, 11610, 11650, 146, 102, 58, 105, 110, 99, 
0, 2049, 3646, 49, 0, 1, 11652, 2049, 10639, 2049, 10694, 
10, 11641, 11670, 146, 102, 58, 100, 101, 99, 0, 2049, 
3646, 49, 0, 1, 11672, 2049, 10639, 2049, 10706, 10, 11661, 
11691, 146, 102, 58, 99, 97, 115, 101, 0, 2049, 11074, 
2049, 10793, 1793, 11703, 2049, 10881, 8, 1, -1, 10, 1, 
11697, 1793, 11711, 3, 1, 0, 10, 1, 11707, 2049, 64, 
25, 6, 771, 10, 11681, 11729, 146, 102, 58, 115, 105, 
103, 110, 0, 2049, 10866, 2049, 3646, 48, 0, 1, 11733, 
2049, 10639, 2049, 10793, 1793, 11748, 1, 0, 2049, 10881, 10, 
1, 11743, 2049, 2653, 2049, 3646, 48, 0, 1, 11754, 2049, 
10639, 2049, 10836, 1793, 11767, 1, 1, 10, 1, 11764, 1793, 
11774, 1, -1, 10, 1, 11771, 2049, 64, 10, 11719, 2147483645, 
133, 101, 58, 77, 65, 88, 0, 11779, -2147483645, 133, 101, 
58, 77, 73, 78, 0, 11788, -2147483647, 133, 101, 58, 78, 
65, 78, 0, 11797, 2147483646, 133, 101, 58, 73, 78, 70, 
0, 11806, -2147483646, 133, 101, 58, 45, 73, 78, 70, 0, 
11815, 11833, 146, 101, 58, 110, 63, 0, 1, -2147483645, 2049, 
2890, 1, 2147483645, 2049, 2903, 2049, 2921, 10, 11825, 11854, 146, 
101, 58, 109, 97, 120, 63, 0, 1, 2147483645, 11, 10, 
11844, 11868, 146, 101, 58, 109, 105, 110, 63, 0, 1, 
-2147483645, 11, 10, 11858, 11883, 146, 101, 58, 122, 101, 114, 
111, 63, 0, 2049, 2527, 10, 11872, 11896, 146, 101, 58, 
110, 97, 110, 63, 0, 1, -2147483647, 11, 10, 11886, 11910, 
146, 101, 58, 105, 110, 102, 63, 0, 1, 2147483646, 11, 
10, 11900, 11925, 146, 101, 58, 45, 105, 110, 102, 63, 
0, 1, -2147483646, 11, 10, 11914, 11939, 146, 101, 58, 99, 
108, 105, 112, 0, 1, -2147483645, 1, 2147483645, 2049, 2873, 10, 
11929, 11954, 146, 102, 58, 69, 49, 0, 1793, 11956, 2049, 
3646, 49, 46, 101, 53, 0, 1, 11958, 2049, 10639, 10, 
11946, 11985, 146, 102, 58, 115, 105, 103, 110, 101, 100, 
45, 115, 113, 114, 116, 0, 2049, 10866, 2049, 11729, 2049, 
11287, 2049, 10779, 2049, 10620, 2049, 10718, 10, 11968, 12017, 146, 
102, 58, 115, 105, 103, 110, 101, 100, 45, 115, 113, 
117, 97, 114, 101, 0, 2049, 10866, 2049, 11729, 2049, 10866, 
2049, 10718, 2049, 10620, 2049, 10718, 10, 11998, 12042, 146, 102, 
58, 45, 115, 104, 105, 102, 116, 0, 2049, 11954, 2049, 
10718, 10, 12030, 12059, 146, 102, 58, 43, 115, 104, 105, 
102, 116, 0, 2049, 11954, 2049, 10730, 10, 12047, 12077, 146, 
102, 58, 43, 101, 110, 99, 111, 100, 101, 0, 2049, 
11985, 2049, 12042, 10, 12064, 12095, 146, 102, 58, 45, 101, 
110, 99, 111, 100, 101, 0, 2049, 10866, 2049, 11729, 2049, 
12059, 2049, 10866, 2049, 10718, 2049, 10620, 2049, 10718, 10, 11998, 
12120, 146, 102, 58, 116, 111, 45, 101, 0, 2049, 10866, 
2049, 11456, 1793, 12131, 2049, 10881, 1, -2147483647, 10, 1, 12126, 
2049, 2653, 2049, 10866, 2049, 11471, 1793, 12146, 2049, 10881, 1, 
2147483646, 10, 1, 12141, 2049, 2653, 2049, 10866, 2049, 11487, 1793, 
12161, 2049, 10881, 1, -2147483646, 10, 1, 12156, 2049, 2653, 2049, 
12077, 2049, 11503, 2049, 10659, 2049, 11939, 1, -2147483645, 1, 10881, 
2049, 2373, 1, 2147483645, 1, 10881, 2049, 2373, 10, 12110, 12196, 
146, 101, 58, 116, 111, 45, 102, 0, 1, -2147483647, 1, 
11365, 2049, 2373, 1, 2147483646, 1, 11393, 2049, 2373, 1, -2147483646, 
1, 11424, 2049, 2373, 2049, 10620, 2049, 12095, 10, 12186, 12230, 
146, 102, 58, 115, 116, 111, 114, 101, 0, 1, 12120, 
2049, 2100, 16, 10, 12219, 12247, 146, 102, 58, 102, 101, 
116, 99, 104, 0, 15, 2049, 12196, 10, 12236, 12267, 146, 
102, 58, 100, 117, 109, 112, 45, 115, 116, 97, 99, 
107, 0, 2049, 10852, 2, 1, 11028, 2049, 2275, 1793, 12285, 
2049, 11042, 2049, 10866, 2049, 11304, 2049, 9222, 10, 1, 12276, 
2049, 2275, 10, 12251, 12307, 146, 102, 58, 100, 117, 109, 
112, 45, 97, 115, 116, 97, 99, 107, 0, 2049, 11059, 
2, 1, 11042, 2049, 2275, 1793, 12325, 2049, 10866, 2049, 11304, 
2049, 9222, 2049, 11028, 10, 1, 12316, 2049, 2275, 10, 12290, 
12339, 146, 101, 58, 112, 117, 116, 0, 1, 2147483645, 1793, 
12356, 2049, 3646, 101, 58, 77, 65, 88, 0, 1, 12345, 
2049, 9248, 10, 1, 12343, 2049, 2373, 1, -2147483645, 1793, 12377, 
2049, 3646, 101, 58, 77, 73, 78, 0, 1, 12366, 2049, 
9248, 10, 1, 12364, 2049, 2373, 1, 0, 1793, 12396, 2049, 
3646, 101, 58, 48, 0, 1, 12387, 2049, 9248, 10, 1, 
12385, 2049, 2373, 1, -2147483647, 1793, 12417, 2049, 3646, 101, 58, 
78, 65, 78, 0, 1, 12406, 2049, 9248, 10, 1, 12404, 
2049, 2373, 1, 2147483646, 1793, 12438, 2049, 3646, 101, 58, 73, 
78, 70, 0, 1, 12427, 2049, 9248, 10, 1, 12425, 2049, 
2373, 1, -2147483646, 1793, 12460, 2049, 3646, 101, 58, 45, 73, 
78, 70, 0, 1, 12448, 2049, 9248, 10, 1, 12446, 2049, 
2373, 2049, 12196, 2049, 11304, 10, 12330, 12475, 133, 105, 111, 
0, 0, 12469, 12488, 146, 105, 100, 101, 110, 116, 105, 
102, 121, 0, 3841, 12475, 2049, 2527, 25, 3, 1, 10, 
2049, 9152, 2, 2049, 2562, 1793, 12543, 3, 2049, 3646, 73, 
79, 32, 68, 69, 86, 73, 67, 69, 32, 84, 89, 
80, 69, 32, 48, 48, 49, 48, 32, 78, 79, 84, 
32, 70, 79, 85, 78, 68, 0, 1, 12506, 2049, 9248, 
2049, 9211, 10, 1, 12503, 1793, 12550, 4097, 12475, 10, 1, 
12547, 2049, 64, 10, 12330, 12575, 146, 105, 111, 58, 114, 
110, 103, 45, 111, 112, 101, 114, 97, 116, 105, 111, 
110, 0, 2049, 12488, 3841, 12475, 2049, 9135, 10, 12555, 12594, 
146, 110, 58, 114, 97, 110, 100, 111, 109, 0, 2049, 
12575, 10, 12582, 12608, 146, 100, 58, 119, 111, 114, 100, 
115, 0, 1793, 12617, 2049, 164, 2049, 9248, 2049, 9222, 10, 
1, 12610, 2049, 7454, 10, 12597, 12638, 146, 100, 58, 119, 
111, 114, 100, 115, 45, 119, 105, 116, 104, 0, 2049, 
1874, 2049, 4648, 1793, 12669, 2049, 164, 2, 2049, 1874, 2049, 
4162, 1793, 12658, 2049, 9248, 2049, 9222, 10, 1, 12653, 1793, 
12664, 3, 10, 1, 12662, 2049, 64, 10, 1, 12644, 2049, 
7454, 10, 12622, 12693, 146, 100, 105, 115, 112, 108, 97, 
121, 45, 105, 102, 45, 108, 101, 102, 116, 0, 2, 
2049, 1874, 2049, 4598, 1793, 12705, 2049, 9248, 2049, 9222, 10, 
1, 12700, 1793, 12711, 3, 10, 1, 12709, 2049, 64, 10, 
12622, 12742, 146, 100, 58, 119, 111, 114, 100, 115, 45, 
98, 101, 103, 105, 110, 110, 105, 110, 103, 45, 119, 
105, 116, 104, 0, 2049, 1874, 2049, 4648, 1793, 12753, 2049, 
164, 2049, 12693, 10, 1, 12748, 2049, 7454, 10, 0, ]


class Retro:
    def map_in(self, name):
        return self.memory[self.find_entry(name) + 1]

    def __init__(self):
        self.ip = 0
        self.stack = IntegerStack()
        self.address = IntegerStack()
        self.memory = Memory("ngaImage", InitialImage, 1000000)
        self.clock = Clock()
        self.rng = RNG()
        self.files = FileSystem()
        self.floats = FloatStack()
        self.afloats = FloatStack()
        self.Dictionary = self.populate_dictionary()
        self.Cached = self.cache_words()

        self.setup_devices()
        self.instructions = [
            self.i_nop,
            self.i_lit,
            self.i_dup,
            self.i_drop,
            self.i_swap,
            self.i_push,
            self.i_pop,
            self.i_jump,
            self.i_call,
            self.i_ccall,
            self.i_return,
            self.i_eq,
            self.i_neq,
            self.i_lt,
            self.i_gt,
            self.i_fetch,
            self.i_store,
            self.i_add,
            self.i_subtract,
            self.i_multiply,
            self.i_divmod,
            self.i_and,
            self.i_or,
            self.i_xor,
            self.i_shift,
            self.i_zreturn,
            self.i_halt,
            self.i_ienumerate,
            self.i_iquery,
            self.i_iinvoke,
        ]

    def div_mod(self, a, b):
        x = abs(a)
        y = abs(b)
        q, r = divmod(x, y)
        if a < 0 and b < 0:
            r *= -1
        elif a > 0 and b < 0:
            q *= -1
        elif a < 0 and b > 0:
            r *= -1
            q *= -1
        return q, r

    def cache_words(self):
        Cached = dict()
        Cached["interpreter"] = self.map_in("interpret")
        Cached["not_found"] = self.map_in("err:notfound")
        Cached["s:eq?"] = self.map_in("s:eq?")
        Cached["s:to-number"] = self.map_in("s:to-number")
        Cached["s:length"] = self.map_in("s:length")
        Cached["d:lookup"] = self.map_in("d:lookup")
        Cached["d:add-header"] = self.map_in("d:add-header")
        return Cached

    def populate_dictionary(self):
        Dictionary = dict()
        header = self.memory[2]
        while header != 0:
            named = self.extract_string(header + 3)
            if not named in Dictionary:
                Dictionary[named] = header
            header = self.memory[header]
        return Dictionary

    def find_entry(self, named):
        if named in self.Dictionary:
            return self.Dictionary[named]

        header = self.memory[2]
        Done = False
        while header != 0 and not Done:
            if named == self.extract_string(header + 3):
                self.Dictionary[named] = header
                Done = True
            else:
                header = self.memory[header]
        return header

    def get_input(self):
        return ord(sys.stdin.read(1))

    def display_character(self):
        if self.stack.tos() > 0 and self.stack.tos() < 128:
            if self.stack.tos() == 8:
                sys.stdout.write(chr(self.stack.pop()))
                sys.stdout.write(chr(32))
                sys.stdout.write(chr(8))
            else:
                sys.stdout.write(chr(self.stack.pop()))
        else:
            sys.stdout.write("\033[2J\033[1;1H")
            self.stack.pop()
        sys.stdout.flush()

    def i_nop(self):
        pass

    def i_lit(self):
        self.ip += 1
        self.stack.push(self.memory[self.ip])

    def i_dup(self):
        self.stack.dup()

    def i_drop(self):
        self.stack.drop()

    def i_swap(self):
        self.stack.swap()

    def i_push(self):
        self.address.push(self.stack.pop())

    def i_pop(self):
        self.stack.push(self.address.pop())

    def i_jump(self):
        self.ip = self.stack.pop() - 1

    def i_call(self):
        self.address.push(self.ip)
        self.ip = self.stack.pop() - 1

    def i_ccall(self):
        target = self.stack.pop()
        if self.stack.pop() != 0:
            self.address.push(self.ip)
            self.ip = target - 1

    def i_return(self):
        self.ip = self.address.pop()

    def i_eq(self):
        a = self.stack.pop()
        b = self.stack.pop()
        if b == a:
            self.stack.push(-1)
        else:
            self.stack.push(0)

    def i_neq(self):
        a = self.stack.pop()
        b = self.stack.pop()
        if b != a:
            self.stack.push(-1)
        else:
            self.stack.push(0)

    def i_lt(self):
        a = self.stack.pop()
        b = self.stack.pop()
        if b < a:
            self.stack.push(-1)
        else:
            self.stack.push(0)

    def i_gt(self):
        a = self.stack.pop()
        b = self.stack.pop()
        if b > a:
            self.stack.push(-1)
        else:
            self.stack.push(0)

    # The fetch instruction also handles certain
    # introspection queries.
    #
    # Of note is the min and max values for a cell.
    # In most VM implementations, this is limited
    # to 32 bit or 64 bit ranges, but Python allows
    # an unlimited range.
    #
    # I report as if the cells are capped at 128 bits
    # but you can safely ignore this if running on
    # the Python VM. (This does have an impact on
    # floating point values, if using the `e:` words
    # for converting them to/from an encoded format in
    # standard cells, but should not affect anything
    # else in the standard system)

    def i_fetch_query(self, target):
        if target == -1:
            self.stack.push(self.stack.depth())
        elif target == -2:
            self.stack.push(self.address.depth())
        elif target == -3:
            self.stack.push(self.memory.size())
        elif target == -4:
            self.stack.push(-170141183460469231731687303715884105728)
        elif target == -5:
            self.stack.push(170141183460469231731687303715884105727)
        else:
            raise IndexError

    def i_fetch(self):
        target = self.stack.pop()
        if target >= 0:
            self.stack.push(self.memory[target])
        else:
            self.i_fetch_query(target)

    def i_store(self):
        mi = self.stack.pop()
        self.memory[mi] = self.stack.pop()

    def i_add(self):
        t = self.stack.pop()
        v = self.stack.pop()
        self.stack.push(t + v)

    def i_subtract(self):
        t = self.stack.pop()
        v = self.stack.pop()
        self.stack.push(v - t)

    def i_multiply(self):
        t = self.stack.pop()
        v = self.stack.pop()
        self.stack.push(v * t)

    def i_divmod(self):
        t = self.stack.pop()
        v = self.stack.pop()
        b, a = self.div_mod(v, t)
        self.stack.push(a)
        self.stack.push(b)

    def i_and(self):
        t = self.stack.pop()
        m = self.stack.pop()
        self.stack.push(m & t)

    def i_or(self):
        t = self.stack.pop()
        m = self.stack.pop()
        self.stack.push(m | t)

    def i_xor(self):
        t = self.stack.pop()
        m = self.stack.pop()
        self.stack.push(m ^ t)

    def i_shift(self):
        t = self.stack.pop()
        v = self.stack.pop()

        if t < 0:
            v <<= t * -1
        else:
            v >>= t

        self.stack.push(v)

    def i_zreturn(self):
        if self.stack.tos() == 0:
            self.stack.pop()
            self.ip = self.address.pop()

    def i_halt(self):
        self.ip = 9000000

    def i_ienumerate(self):
        self.stack.push(6)

    def i_iquery(self):
        device = self.stack.pop()
        if device == 0:  # generic output
            self.stack.push(0)
            self.stack.push(0)
        if device == 1:  # floating point
            self.stack.push(1)
            self.stack.push(2)
        if device == 2:  # files
            self.stack.push(0)
            self.stack.push(4)
        if device == 3:  # rng
            self.stack.push(0)
            self.stack.push(10)
        if device == 4:  # time
            self.stack.push(0)
            self.stack.push(5)
        if device == 5:  # scripting
            self.stack.push(0)
            self.stack.push(9)

    def file_open_params(self):
        mode = self.stack.pop()
        name = self.extract_string(self.stack.pop())
        return name, mode

    def file_write_params(self):
        slot = self.stack.pop()
        char = self.stack.pop()
        return slot, char

    def setup_devices(self):
        self.files_instr = {
            0: lambda: self.stack.push(self.files.open(self.file_open_params())),
            1: lambda: self.files.close(self.stack.pop()),
            2: lambda: self.stack.push(self.files.read(self.stack.pop())),
            3: lambda: self.files.write(self.file_write_params()),
            4: lambda: self.stack.push(self.files.pos(self.stack.pop())),
            5: lambda: self.files.seek(),
            6: lambda: self.stack.push(self.files.size(self.stack.pop())),
            7: lambda: self.files.delete(self.extract_string(self.stack.pop())),
            8: lambda: 1 + 1,
        }

        self.rng_instr = {0: lambda: self.stack.push(self.rng())}

        self.clock_instr = {
            0: lambda: self.stack.push(int(time.time())),
            1: lambda: self.stack.push(self.clock["day"]),
            2: lambda: self.stack.push(self.clock["month"]),
            3: lambda: self.stack.push(self.clock["year"]),
            4: lambda: self.stack.push(self.clock["hour"]),
            5: lambda: self.stack.push(self.clock["minute"]),
            6: lambda: self.stack.push(self.clock["second"]),
            7: lambda: self.stack.push(self.clock["day_utc"]),
            8: lambda: self.stack.push(self.clock["month_utc"]),
            9: lambda: self.stack.push(self.clock["year_utc"]),
            10: lambda: self.stack.push(self.clock["hour_utc"]),
            11: lambda: self.stack.push(self.clock["minute_utc"]),
            12: lambda: self.stack.push(self.clock["second_utc"]),
        }

        self.float_instr = {
            0: lambda: self.floats.push(float(self.stack.pop())),
            1: lambda: self.floats.push(float(self.extract_string(self.stack.pop()))),
            2: lambda: self.stack.push(int(self.floats.pop())),
            3: lambda: self.inject_string(str(self.floats.pop()), self.stack.pop()),
            4: lambda: self.floats.add(),
            5: lambda: self.floats.sub(),
            6: lambda: self.floats.mul(),
            7: lambda: self.floats.div(),
            8: lambda: self.floats.floor(),
            9: lambda: self.floats.ceil(),
            10: lambda: self.floats.sqrt(),
            11: lambda: self.stack.push(self.floats.eq()),
            12: lambda: self.stack.push(self.floats.neq()),
            13: lambda: self.stack.push(self.floats.lt()),
            14: lambda: self.stack.push(self.floats.gt()),
            15: lambda: self.stack.push(self.floats.depth()),
            16: lambda: self.floats.dup(),
            17: lambda: self.floats.drop(),
            18: lambda: self.floats.swap(),
            19: lambda: self.floats.log(),
            20: lambda: self.floats.pow(),
            21: lambda: self.floats.sin(),
            22: lambda: self.floats.cos(),
            23: lambda: self.floats.tan(),
            24: lambda: self.floats.asin(),
            25: lambda: self.floats.atan(),
            26: lambda: self.floats.acos(),
            27: lambda: self.afloats.push(self.floats.pop()),
            28: lambda: self.floats.push(self.afloats.pop()),
            29: lambda: self.stack.push(self.afloats.depth()),
        }

    def i_iinvoke(self):
        device = self.stack.pop()
        #        print('dev:', device)
        if device == 0:
            self.display_character()
        if device == 1:
            action = self.stack.pop()
            self.float_instr[int(action)]()
        if device == 2:
            action = self.stack.pop()
            self.files_instr[int(action)]()
        if device == 3:
            self.rng_instr[0]()
        if device == 4:
            action = self.stack.pop()
            self.clock_instr[int(action)]()
        if device == 5:
            action = self.stack.pop()
            if action == 0:
                self.stack.push(len(sys.argv) - 2)
            if action == 1:
                a = self.stack.pop()
                b = self.stack.pop()
                self.stack.push(self.inject_string(sys.argv[a + 2], b))
            if action == 2:
                run_file(self.extract_string(self.stack.pop()))
            if action == 3:
                b = self.stack.pop()
                self.stack.push(self.inject_string(sys.argv[0], b))

    def validate_opcode(self, I0, I1, I2, I3):
        if (
            (I0 >= 0 and I0 <= 29)
            and (I1 >= 0 and I1 <= 29)
            and (I2 >= 0 and I2 <= 29)
            and (I3 >= 0 and I3 <= 29)
        ):
            return True
        else:
            return False

    def extract_string(self, at):
        s = ""
        while self.memory[at] != 0:
            s = s + chr(self.memory[at])
            at = at + 1
        return s

    def inject_string(self, s, to):
        for c in s:
            self.memory[to] = ord(c)
            to = to + 1
        self.memory[to] = 0

    def execute(self, word, notfound):
        self.ip = word
        if self.address.depth() == 0:
            self.address.push(0)
        while self.ip < 1000000:
            if self.ip == self.Cached["s:eq?"]:
                a = self.extract_string(self.stack.pop())
                b = self.extract_string(self.stack.pop())
                if a == b:
                    self.stack.push(-1)
                else:
                    self.stack.push(0)
                self.ip = self.address.pop()
            elif self.ip == self.Cached["d:lookup"]:
                name = self.extract_string(self.stack.pop())
                header = self.find_entry(name)
                self.stack.push(header)
                self.memory[self.Cached["d:lookup"] - 20] = header  # "which"
                self.ip = self.address.pop()
            elif self.ip == self.Cached["s:to-number"]:
                n = self.extract_string(self.stack.pop())
                self.stack.push(int(n))
                self.ip = self.address.pop()
            elif self.ip == self.Cached["s:length"]:
                n = len(self.extract_string(self.stack.pop()))
                self.stack.push(n)
                self.ip = self.address.pop()
            else:
                if self.ip == notfound:
                    print("ERROR: word not found!")
                if self.ip == self.Cached["d:add-header"]:
                    self.Dictionary[self.extract_string(self.stack[-3])] = self.memory[
                        3
                    ]
                opcode = self.memory[self.ip]
                I0 = opcode & 0xFF
                I1 = (opcode >> 8) & 0xFF
                I2 = (opcode >> 16) & 0xFF
                I3 = (opcode >> 24) & 0xFF
                if self.validate_opcode(I0, I1, I2, I3):
                    # print("Bytecode: ", I0, I1, I2, I3, "at", self.ip)
                    if I0 != 0:
                        self.instructions[I0]()
                    if I1 != 0:
                        self.instructions[I1]()
                    if I2 != 0:
                        self.instructions[I2]()
                    if I3 != 0:
                        self.instructions[I3]()
                else:
                    print("Invalid Bytecode: ", opcode, "at", self.ip)
                    self.ip = 2000000
            if self.address.depth() == 0:
                self.ip = 2000000
            self.ip = self.ip + 1
        return

    def run(self):
        done = False
        while not done:
            line = input("\nOk> ")
            if line == "bye":
                done = True
            else:
                for token in line.split():
                    self.inject_string(token, 1024)
                    self.stack.push(1024)
                    self.execute(self.Cached["interpreter"], self.Cached["not_found"])

    def run_file(self, file):
        if not os.path.exists(file):
            print("File '{0}' not found".format(file))
            return

        in_block = False
        with open(file, "r") as source:
            for line in source.readlines():
                if line.rstrip() == "~~~":
                    in_block = not in_block
                elif in_block:
                    for token in line.strip().split():
                        self.inject_string(token, 1024)
                        self.stack.push(1024)
                        self.execute(
                            self.Cached["interpreter"], self.Cached["not_found"]
                        )

    def update_image(self):
        import requests
        import shutil

        data = requests.get("https://forthworks.com/retro/ngaImage", stream=True)
        with open("ngaImage", "wb") as f:
            data.raw.decode_content = True
            shutil.copyfileobj(data.raw, f)


if __name__ == "__main__":
    retro = Retro()
    if len(sys.argv) == 1:
        retro.run()

    if len(sys.argv) == 2:
        retro.run_file(sys.argv[1])

    sources = []

    interactive = False

    if len(sys.argv) > 2:
        i = 1
        e = len(sys.argv)
        while i < e:
            param = sys.argv[i]
            if param == "-f":
                i += 1
                sources.append(sys.argv[i])
            elif param == "-u":
                i += 1
                retro.memory.load_image(sys.argv[i])
            elif param == "-i":
                interactive = True
            i += 1

    if len(sys.argv) > 2 and sys.argv[1][0] != "-":
        retro.run_file(sys.argv[1])
    else:
        for source in sources:
            retro.run_file(source)

    if interactive:
        retro.run()
