import re
from pathlib import Path
from .helper import read_notebook, is_notebook, is_script
from .exceptions import UnsupportedFileExtensionError

_pattern = r"^#\s*export\s*$"
_flags = re.IGNORECASE | re.MULTILINE


def export(notebook_path: str, output_path: str, delete_export_comments: bool = False, cell_seperator: str = "\n\n"):
    """
    Export cells to Python script.

    Parameters
    ----------
    notebook_path: str
                    Path of the notebook that contains cells to be exported.
    output_path: str
                    Path of the Python script cells will be exported. Note that if the script is existed, export()
                    will override it. If not existed, it will create the file.
    delete_export_comments: bool
                    Whether keep the '#export' or '# export' comments in the output file.
    cell_seperator: str
                    String that seperates exported cells. Default to 2 newlines.
    Returns
    ------
    None
    """

    notebook_path = Path(notebook_path)
    output_path = Path(output_path)
    # Check whether the notebook exists or not.
    if not notebook_path.exists():
        raise FileNotFoundError(notebook_path)
    if notebook_path.suffix != ".ipynb":
        raise UnsupportedFileExtensionError(notebook_path, [".ipynb"])
    if output_path.suffix != ".py":
        raise UnsupportedFileExtensionError(output_path, [".py"])
    # Read notebook
    cells = read_notebook(notebook_path)
    # Iterate over cells
    sources = []
    for cell in cells:
        #  Find cells that will be exported
        if re.search(_pattern, cell, _flags) is not None:
            if delete_export_comments:
                source = re.sub(_pattern, '', cell, flags=_flags)
            else:
                source = cell
            sources.append(source)
    if len(sources) > 0:
        #  Write source codes to output file
        with open(output_path, 'w', encoding='UTF-8') as out:
            out.write(
                f"# THIS SCRIPT IS AUTOGENERATED FROM {notebook_path}. {cell_seperator}")
            for index, source in enumerate(sources):
                if not delete_export_comments:
                    out.write("\n")
                out.write(source)
                if index == len(sources) - 1:
                    # Put single new line at the end of script.
                    out.write("\n")
                else:
                    # Put cell seperator between different cells.
                    out.write(cell_seperator)
    else:
        print("No cell exported.")
