"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaPowertoolsLayer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
/**
 * Defines a new Lambda Layer with Powertools for python library.
 */
class LambdaPowertoolsLayer extends aws_cdk_lib_1.aws_lambda.LayerVersion {
    constructor(scope, id, props) {
        const runtimeFamily = props?.runtimeFamily ?? aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON;
        const languageName = getLanguageNameFromRuntimeFamily(runtimeFamily);
        const dockerFilePath = path.join(__dirname, `../layer/${languageName}`);
        const compatibleArchitectures = props?.compatibleArchitectures ?? [aws_cdk_lib_1.aws_lambda.Architecture.X86_64];
        const compatibleArchitecturesDescription = compatibleArchitectures.map((arch) => arch.name).join(', ');
        console.log(`path ${dockerFilePath}`);
        super(scope, id, {
            code: aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(dockerFilePath, {
                buildArgs: {
                    PACKAGE_SUFFIX: LambdaPowertoolsLayer.constructBuildArgs(runtimeFamily, props?.includeExtras, props?.version),
                },
                // supports cross-platform docker build
                platform: getDockerPlatformNameFromArchitectures(compatibleArchitectures),
            }),
            layerVersionName: props?.layerVersionName ? props?.layerVersionName : undefined,
            license: 'MIT-0',
            compatibleRuntimes: getRuntimesFromRuntimeFamily(runtimeFamily),
            compatibleArchitectures,
            description: `Lambda Powertools for ${languageName} [${compatibleArchitecturesDescription}]${props?.includeExtras ? ' with extra dependencies' : ''} ${props?.version ? `version ${props?.version}` : 'latest version'}`.trim(),
        });
    }
    /**
     * creates build argument for the Dockerfile.
     * There are multiple combinations between version and extras package that results in different suffix for the installation.
     * With and without version, with and without extras flag.
     * We construct one suffix here because it is easier to do in code than inside the Dockerfile with bash commands.
     * For example, if we set `includeExtras=true` and `version=1.22.0` we get '[all]==1.22.0'.
     *
     */
    static constructBuildArgs(runtimeFamily, includeExtras, version) {
        let suffix = '';
        switch (runtimeFamily) {
            case aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON:
                if (includeExtras) {
                    suffix = '[all]';
                }
                if (version) {
                    suffix = `${suffix}==${version}`;
                }
                break;
            case aws_cdk_lib_1.aws_lambda.RuntimeFamily.NODEJS:
                if (version) {
                    suffix = `@${version}`;
                }
                break;
            default:
                break;
        }
        return suffix;
    }
}
exports.LambdaPowertoolsLayer = LambdaPowertoolsLayer;
_a = JSII_RTTI_SYMBOL_1;
LambdaPowertoolsLayer[_a] = { fqn: "cdk-aws-lambda-powertools-layer.LambdaPowertoolsLayer", version: "3.1.0" };
function getRuntimesFromRuntimeFamily(runtimeFamily) {
    switch (runtimeFamily) {
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON:
            return [
                aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_7,
                aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_8,
                aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            ];
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.NODEJS:
            return [
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_12_X,
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_14_X,
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_16_X,
            ];
        default:
            return [];
    }
}
function getLanguageNameFromRuntimeFamily(runtimeFamily) {
    switch (runtimeFamily) {
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON:
            return 'Python';
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.NODEJS:
            return 'TypeScript';
        default:
            return 'Unknown';
    }
}
// Docker expects a single string for the --platform option.
// getDockerPlatformNameFromArchitectures converts the Architecture enum to a string.
function getDockerPlatformNameFromArchitectures(architectures) {
    if (architectures.length == 1) {
        return architectures[0].dockerPlatform;
    }
    else {
        // if we have multiple architectures, we default to x86_64, hoping for the
        // layer not to have any architecture specific code or at least contain
        // binary code for all architectures
        return aws_lambda_1.Architecture.X86_64.dockerPlatform;
    }
}
//# sourceMappingURL=data:application/json;base64,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