"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const path = require("path");
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * A standard synth with a generated buildspec
 */
class SimpleSynthAction {
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.synthCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b;
        const buildCommand = this.props.buildCommand;
        const synthCommand = this.props.synthCommand;
        const installCommand = this.props.installCommand;
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: (_a = this.props.projectName) !== null && _a !== void 0 ? _a : this.props.projectName,
            environment: this.props.environment,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: _util_1.filterEmpty([
                            this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                            installCommand,
                        ]),
                    },
                    build: {
                        commands: _util_1.filterEmpty([
                            buildCommand,
                            synthCommand,
                        ]),
                    },
                },
                artifacts: renderArtifacts(this),
            }),
            environmentVariables: {
                ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
                ...this.props.environmentVariables,
            },
        });
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_b = this.props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
//# sourceMappingURL=data:application/json;base64,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