"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetsAction = exports.AssetType = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Type of the asset that is being published
 */
var AssetType;
(function (AssetType) {
    /**
     * A file
     */
    AssetType["FILE"] = "file";
    /**
     * A Docker image
     */
    AssetType["DOCKER_IMAGE"] = "docker-image";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
/**
 * Action to publish an asset in the pipeline
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to prepare and publish the asset.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline when you add stacks that use assets.
 */
class PublishAssetsAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.commands = new Array();
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const project = new codebuild.PipelineProject(this, 'Default', {
            projectName: this.props.projectName,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g cdk-assets${installSuffix}`,
                    },
                    build: {
                        commands: core_1.Lazy.listValue({ produce: () => this.commands }),
                    },
                },
            }),
            // Needed to perform Docker builds
            environment: props.assetType === AssetType.DOCKER_IMAGE ? { privileged: true } : undefined,
        });
        const rolePattern = props.assetType === AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        this.action = new codepipeline_actions.CodeBuildAction({
            actionName: props.actionName,
            project,
            input: this.props.cloudAssemblyInput,
        });
    }
    /**
     * Add a single publishing command
     *
     * Manifest path should be relative to the root Cloud Assembly.
     */
    addPublishCommand(relativeManifestPath, assetSelector) {
        const command = `cdk-assets --path "${relativeManifestPath}" --verbose publish "${assetSelector}"`;
        if (!this.commands.includes(command)) {
            this.commands.push(command);
        }
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.PublishAssetsAction = PublishAssetsAction;
//# sourceMappingURL=data:application/json;base64,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