"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const cfn = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const path = require("path");
const construct_internals_1 = require("../private/construct-internals");
/**
 * Action to deploy a CDK Stack
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 */
class DeployCdkStackAction {
    constructor(props) {
        var _a, _b, _c, _d, _e;
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = (_a = props.dependencyStackArtifactIds) !== null && _a !== void 0 ? _a : [];
        this.prepareRunOrder = (_b = props.prepareRunOrder) !== null && _b !== void 0 ? _b : 1;
        this.executeRunOrder = (_c = props.executeRunOrder) !== null && _c !== void 0 ? _c : this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = (_d = props.baseActionName) !== null && _d !== void 0 ? _d : this.stackName;
        const changeSetName = (_e = props.changeSetName) !== null && _e !== void 0 ? _e : 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * Construct a DeployCdkStackAction from a Stack artifact
     */
    static fromStackArtifact(scope, artifact, options) {
        var _a;
        if (!artifact.assumeRoleArn) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const actionRole = roleFromPlaceholderArn(scope, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, artifact.cloudFormationExecutionRoleArn);
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        const templatePath = path.relative(appAsmRoot, fullTemplatePath);
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: (_a = options.stackName) !== null && _a !== void 0 ? _a : artifact.stackName,
            ...options,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    scope = hackyRoleScope(scope, arn);
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = scope.node.tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = scope.node.tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    return iam.Role.fromRoleArn(scope, id, cfnExpressionFromManifestString(arn), { mutable: false });
}
/**
 * MASSIVE HACK
 *
 * We have a bug in the CDK where it's only going to consider Roles that are physically in a
 * different Stack object from the Pipeline "cross-account", and will add the appropriate
 * Bucket/Key policies.
 * https://github.com/aws/aws-cdk/pull/8280 will resolve this, but for now we fake it by hacking
 * up a Stack object to root the role in!
 *
 * Fortunatey, we can just 'new up' an unrooted Stack (unit tests do this all the time) and toss it
 * away. It will never be synthesized, but all the logic happens to work out!
 */
function hackyRoleScope(scope, arn) {
    const parts = core_1.Arn.parse(cxapi.EnvironmentPlaceholders.replace(arn, {
        accountId: '',
        partition: '',
        region: '',
    }));
    return new core_1.Stack(undefined, undefined, {
        env: {
            // Empty string means ARN had a placeholder which means same account as pipeline stack
            account: parts.account || core_1.Stack.of(scope).account,
        },
    });
}
/**
 * Return a CloudFormation expression from a manifest string with placeholders
 */
function cfnExpressionFromManifestString(s) {
    // This implementation relies on the fact that the manifest placeholders are
    // '${AWS::Partition}' etc., and so are the same values as those that are
    // trivially substituable using a `Fn.sub`.
    return core_1.Fn.sub(s);
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
//# sourceMappingURL=data:application/json;base64,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