"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = 'Resource';
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = 'Default';
const PATH_SEP = '/';
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter(x => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error('Unable to calculate a unique id for an empty set of components');
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter(c => encoding_1.unresolved(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(',')}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeNonAlphanumeric(components[0]);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter(x => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join('')
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto.createHash('md5').update(path.join(PATH_SEP)).digest('hex');
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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