"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fingerprint = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The set of exclude patterns, if defined in `options.exclude`
 * 3. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const exclude = options.exclude || [];
    if (exclude.length) {
        _hashField(hash, 'options.exclude', JSON.stringify(exclude));
    }
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        if (!isRootDir && utils_1.shouldExclude(exclude, symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, _contentFingerprint(realPath, stat));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function _contentFingerprint(file, stat) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // tslint:disable-next-line: no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    try {
        let read = 0;
        // tslint:disable-next-line: no-conditional-assignment
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            hash.update(buffer.slice(0, read));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${stat.size}:${hash.digest('hex')}`;
}
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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