"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addDependency = void 0;
const stack_1 = require("./stack");
const stage_1 = require("./stage");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a CloudFormation
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the AWS::CloudFormation::Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    const sourceStage = stage_1.Stage.of(sourceStack);
    const targetStage = stage_1.Stage.of(targetStack);
    if (sourceStage !== targetStage) {
        throw new Error(`You cannot add a dependency from '${source.node.path}' (in ${describeStage(sourceStage)}) to '${target.node.path}' (in ${describeStage(targetStage)}): dependency cannot cross stage boundaries`);
    }
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define a assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}': ${reason}`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element) ? element.nestedStackResource : element;
        if (!resource) {
            throw new Error('assertion failure'); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
/**
 * Return a string representation of the given assembler, for use in error messages
 */
function describeStage(assembly) {
    if (!assembly) {
        return 'an unrooted construct tree';
    }
    if (!assembly.parentStage) {
        return 'the App';
    }
    return `Stage '${assembly.node.path}'`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsbUNBQWdDO0FBQ2hDLG1DQUFnQztBQUNoQyxpQ0FBa0Y7QUFFbEY7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQWtCRztBQUNILFNBQWdCLGFBQWEsQ0FBb0IsTUFBUyxFQUFFLE1BQVMsRUFBRSxNQUFlO0lBQ2xGLElBQUksTUFBTSxLQUFLLE1BQU0sRUFBRTtRQUNuQixPQUFPO0tBQ1Y7SUFDRCxNQUFNLFdBQVcsR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3JDLE1BQU0sV0FBVyxHQUFHLGFBQUssQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDckMsTUFBTSxXQUFXLEdBQUcsYUFBSyxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUMxQyxNQUFNLFdBQVcsR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzFDLElBQUksV0FBVyxLQUFLLFdBQVcsRUFBRTtRQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLHFDQUFxQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksU0FBUyxhQUFhLENBQUMsV0FBVyxDQUFDLFNBQVMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLFNBQVMsYUFBYSxDQUFDLFdBQVcsQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO0tBQ3ROO0lBQ0QseURBQXlEO0lBQ3pELE1BQU0sVUFBVSxHQUFHLDBCQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDM0MsTUFBTSxVQUFVLEdBQUcsMEJBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUMzQyxNQUFNLFdBQVcsR0FBRyw0QkFBcUIsQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDbEUsdUVBQXVFO0lBQ3ZFLG1DQUFtQztJQUNuQyxJQUFJLENBQUMsV0FBVyxFQUFFO1FBQ2QsTUFBTSxjQUFjLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsNENBQTRDO1FBQ2xGLE1BQU0sY0FBYyxHQUFHLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNyQyxjQUFjLENBQUMsc0JBQXNCLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQzlELE9BQU87S0FDVjtJQUNELG9GQUFvRjtJQUNwRix5RUFBeUU7SUFDekUsNEVBQTRFO0lBQzVFLHVCQUF1QjtJQUN2Qix1RUFBdUU7SUFDdkUsNkVBQTZFO0lBQzdFLDJFQUEyRTtJQUMzRSx3QkFBd0I7SUFDeEIsSUFBSSxXQUFXLEtBQUssTUFBTSxFQUFFO1FBQ3hCLE9BQU87S0FDVjtJQUNELHVFQUF1RTtJQUN2RSw2RUFBNkU7SUFDN0UsMkRBQTJEO0lBQzNELElBQUksV0FBVyxLQUFLLE1BQU0sRUFBRTtRQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLGlCQUFpQixXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksc0NBQXNDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxNQUFNLE1BQU0sRUFBRSxDQUFDLENBQUM7S0FDcEk7SUFDRCw0RUFBNEU7SUFDNUUsNkVBQTZFO0lBQzdFLGdEQUFnRDtJQUNoRCxNQUFNLGNBQWMsR0FBRyx3QkFBd0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUN4RCxNQUFNLGNBQWMsR0FBRyx3QkFBd0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUN4RCxjQUFjLENBQUMsc0JBQXNCLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDdEQsU0FBUyx3QkFBd0IsQ0FBQyxPQUE0QjtRQUMxRCxNQUFNLFFBQVEsR0FBRyxhQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRixJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ1gsTUFBTSxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsd0JBQXdCO1NBQ2pFO1FBQ0QsTUFBTSxhQUFhLEdBQUcsYUFBSyxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN6QyxvREFBb0Q7UUFDcEQsSUFBSSxXQUFXLEtBQUssYUFBYSxFQUFFO1lBQy9CLE9BQU8sUUFBUSxDQUFDO1NBQ25CO1FBQ0QsT0FBTyx3QkFBd0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNuRCxDQUFDO0FBQ0wsQ0FBQztBQTFERCxzQ0EwREM7QUFDRDs7R0FFRztBQUNILFNBQVMsYUFBYSxDQUFDLFFBQTJCO0lBQzlDLElBQUksQ0FBQyxRQUFRLEVBQUU7UUFDWCxPQUFPLDRCQUE0QixDQUFDO0tBQ3ZDO0lBQ0QsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUU7UUFDdkIsT0FBTyxTQUFTLENBQUM7S0FDcEI7SUFDRCxPQUFPLFVBQVUsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQztBQUMzQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ2ZuUmVzb3VyY2UgfSBmcm9tICcuL2Nmbi1yZXNvdXJjZSc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJy4vc3RhY2snO1xuaW1wb3J0IHsgU3RhZ2UgfSBmcm9tICcuL3N0YWdlJztcbmltcG9ydCB7IGZpbmRMYXN0Q29tbW9uRWxlbWVudCwgcGF0aFRvVG9wTGV2ZWxTdGFjayBhcyBwYXRoVG9Sb290IH0gZnJvbSAnLi91dGlsJztcbnR5cGUgRWxlbWVudCA9IENmblJlc291cmNlIHwgU3RhY2s7XG4vKipcbiAqIEFkZHMgYSBkZXBlbmRlbmN5IGJldHdlZW4gdHdvIHJlc291cmNlcyBvciBzdGFja3MsIGFjcm9zcyBzdGFjayBhbmQgbmVzdGVkXG4gKiBzdGFjayBib3VuZGFyaWVzLlxuICpcbiAqIFRoZSBhbGdvcml0aG0gY29uc2lzdHMgb2Y6XG4gKiAtIFRyeSB0byBmaW5kIHRoZSBkZWVwZXN0IGNvbW1vbiBzdGFjayBiZXR3ZWVuIHRoZSB0d28gZWxlbWVudHNcbiAqIC0gSWYgdGhlcmUgaXNuJ3QgYSBjb21tb24gc3RhY2ssIGl0IG1lYW5zIHRoZSBlbGVtZW50cyBiZWxvbmcgdG8gdHdvXG4gKiAgIGRpc2pvaW5lZCBzdGFjay10cmVlcyBhbmQgdGhlcmVmb3JlIHdlIGFwcGx5IHRoZSBkZXBlbmRlbmN5IGF0IHRoZVxuICogICBhc3NlbWJseS9hcHAgbGV2ZWwgYmV0d2VlbiB0aGUgdHdvIHRvcGwtbGV2ZWwgc3RhY2tzLlxuICogLSBJZiB3ZSBkaWQgZmluZCBhIGNvbW1vbiBzdGFjaywgd2UgYXBwbHkgdGhlIGRlcGVuZGVuY3kgYXMgYSBDbG91ZEZvcm1hdGlvblxuICogICBcIkRlcGVuZHNPblwiIGJldHdlZW4gdGhlIHJlc291cmNlcyB0aGF0IFwicmVwcmVzZW50XCIgb3VyIHNvdXJjZSBhbmQgdGFyZ2V0XG4gKiAgIGVpdGhlciBkaXJlY3RseSBvciB0aHJvdWdoIHRoZSBBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjayByZXNvdXJjZXMgdGhhdFxuICogICBcImxlYWRcIiB0byB0aGVtLlxuICpcbiAqIEBwYXJhbSBzb3VyY2UgVGhlIHNvdXJjZSByZXNvdXJjZS9zdGFjayAodGhlIGRlcGVkZW50KVxuICogQHBhcmFtIHRhcmdldCBUaGUgdGFyZ2V0IHJlc291cmNlL3N0YWNrICh0aGUgZGVwZW5kZW5jeSlcbiAqIEBwYXJhbSByZWFzb24gT3B0aW9uYWwgcmVzb3VyY2UgdG8gYXNzb2NpYXRlIHdpdGggdGhlIGRlcGVuZGVuY3kgZm9yXG4gKiBkaWFnbm9zdGljc1xuICovXG5leHBvcnQgZnVuY3Rpb24gYWRkRGVwZW5kZW5jeTxUIGV4dGVuZHMgRWxlbWVudD4oc291cmNlOiBULCB0YXJnZXQ6IFQsIHJlYXNvbj86IHN0cmluZykge1xuICAgIGlmIChzb3VyY2UgPT09IHRhcmdldCkge1xuICAgICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IHNvdXJjZVN0YWNrID0gU3RhY2sub2Yoc291cmNlKTtcbiAgICBjb25zdCB0YXJnZXRTdGFjayA9IFN0YWNrLm9mKHRhcmdldCk7XG4gICAgY29uc3Qgc291cmNlU3RhZ2UgPSBTdGFnZS5vZihzb3VyY2VTdGFjayk7XG4gICAgY29uc3QgdGFyZ2V0U3RhZ2UgPSBTdGFnZS5vZih0YXJnZXRTdGFjayk7XG4gICAgaWYgKHNvdXJjZVN0YWdlICE9PSB0YXJnZXRTdGFnZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFlvdSBjYW5ub3QgYWRkIGEgZGVwZW5kZW5jeSBmcm9tICcke3NvdXJjZS5ub2RlLnBhdGh9JyAoaW4gJHtkZXNjcmliZVN0YWdlKHNvdXJjZVN0YWdlKX0pIHRvICcke3RhcmdldC5ub2RlLnBhdGh9JyAoaW4gJHtkZXNjcmliZVN0YWdlKHRhcmdldFN0YWdlKX0pOiBkZXBlbmRlbmN5IGNhbm5vdCBjcm9zcyBzdGFnZSBib3VuZGFyaWVzYCk7XG4gICAgfVxuICAgIC8vIGZpbmQgdGhlIGRlZXBlc3QgY29tbW9uIHN0YWNrIGJldHdlZW4gdGhlIHR3byBlbGVtZW50c1xuICAgIGNvbnN0IHNvdXJjZVBhdGggPSBwYXRoVG9Sb290KHNvdXJjZVN0YWNrKTtcbiAgICBjb25zdCB0YXJnZXRQYXRoID0gcGF0aFRvUm9vdCh0YXJnZXRTdGFjayk7XG4gICAgY29uc3QgY29tbW9uU3RhY2sgPSBmaW5kTGFzdENvbW1vbkVsZW1lbnQoc291cmNlUGF0aCwgdGFyZ2V0UGF0aCk7XG4gICAgLy8gaWYgdGhlcmUgaXMgbm8gY29tbW9uIHN0YWNrLCB0aGVuIGRlZmluZSBhIGFzc2VtYmx5LWxldmVsIGRlcGVuZGVuY3lcbiAgICAvLyBiZXR3ZWVuIHRoZSB0d28gdG9wLWxldmVsIHN0YWNrc1xuICAgIGlmICghY29tbW9uU3RhY2spIHtcbiAgICAgICAgY29uc3QgdG9wTGV2ZWxTb3VyY2UgPSBzb3VyY2VQYXRoWzBdOyAvLyBmaXJzdCBwYXRoIGVsZW1lbnQgaXMgdGhlIHRvcC1sZXZlbCBzdGFja1xuICAgICAgICBjb25zdCB0b3BMZXZlbFRhcmdldCA9IHRhcmdldFBhdGhbMF07XG4gICAgICAgIHRvcExldmVsU291cmNlLl9hZGRBc3NlbWJseURlcGVuZGVuY3kodG9wTGV2ZWxUYXJnZXQsIHJlYXNvbik7XG4gICAgICAgIHJldHVybjtcbiAgICB9XG4gICAgLy8gYXNzZXJ0aW9uOiBhdCB0aGlzIHBvaW50IGlmIHNvdXJjZSBhbmQgdGFyZ2V0IGFyZSBzdGFja3MsIGJvdGggYXJlIG5lc3RlZCBzdGFja3MuXG4gICAgLy8gc2luY2Ugd2UgaGF2ZSBhIGNvbW1vbiBzdGFjaywgaXQgaXMgaW1wb3NzaWJsZSB0aGF0IGJvdGggYXJlIHRvcC1sZXZlbFxuICAgIC8vIHN0YWNrcywgc28gbGV0J3MgZXhhbWluZSB0aGUgdHdvIGNhc2VzIHdoZXJlIG9uZSBvZiB0aGVtIGlzIHRvcC1sZXZlbCBhbmRcbiAgICAvLyB0aGUgb3RoZXIgaXMgbmVzdGVkLlxuICAgIC8vIGNhc2UgMSAtIHNvdXJjZSBpcyB0b3AtbGV2ZWwgYW5kIHRhcmdldCBpcyBuZXN0ZWQ6IHRoaXMgaW1wbGllcyB0aGF0XG4gICAgLy8gYHRhcmdldGAgaXMgYSBkaXJlY3Qgb3IgaW5kaXJlY3QgbmVzdGVkIHN0YWNrIG9mIGBzb3VyY2VgLCBhbmQgYW4gZXhwbGljaXRcbiAgICAvLyBkZXBlbmRlbmN5IGlzIG5vdCByZXF1aXJlZCBiZWNhdXNlIG5lc3RlZCBzdGFja3Mgd2lsbCBhbHdheXMgYmUgZGVwbG95ZWRcbiAgICAvLyBiZWZvcmUgdGhlaXIgcGFyZW50cy5cbiAgICBpZiAoY29tbW9uU3RhY2sgPT09IHNvdXJjZSkge1xuICAgICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vIGNhc2UgMiAtIHNvdXJjZSBpcyBuZXN0ZWQgYW5kIHRhcmdldCBpcyB0b3AtbGV2ZWw6IHRoaXMgaW1wbGllcyB0aGF0XG4gICAgLy8gYHNvdXJjZWAgaXMgYSBkaXJlY3Qgb3IgaW5kaXJlY3QgbmVzdGVkIHN0YWNrIG9mIGB0YXJnZXRgLCBhbmQgdGhpcyBpcyBub3RcbiAgICAvLyBwb3NzaWJsZSAobmVzdGVkIHN0YWNrcyBjYW5ub3QgZGVwZW5kIG9uIHRoZWlyIHBhcmVudHMpLlxuICAgIGlmIChjb21tb25TdGFjayA9PT0gdGFyZ2V0KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIHN0YWNrICcke3NvdXJjZVN0YWNrLm5vZGUucGF0aH0nIGNhbm5vdCBkZXBlbmQgb24gYSBwYXJlbnQgc3RhY2sgJyR7dGFyZ2V0U3RhY2subm9kZS5wYXRofSc6ICR7cmVhc29ufWApO1xuICAgIH1cbiAgICAvLyB3ZSBoYXZlIGEgY29tbW9uIHN0YWNrIGZyb20gd2hpY2ggd2UgY2FuIHJlYWNoIGJvdGggYHNvdXJjZWAgYW5kIGB0YXJnZXRgXG4gICAgLy8gbm93IHdlIG5lZWQgdG8gZmluZCB0d28gcmVzb3VyY2VzIHdoaWNoIGFyZSBkZWZpbmVkIGRpcmVjdGx5IGluIHRoaXMgc3RhY2tcbiAgICAvLyBhbmQgd2hpY2ggY2FuIFwibGVhZCB1c1wiIHRvIHRoZSBzb3VyY2UvdGFyZ2V0LlxuICAgIGNvbnN0IHNvdXJjZVJlc291cmNlID0gcmVzb3VyY2VJbkNvbW1vblN0YWNrRm9yKHNvdXJjZSk7XG4gICAgY29uc3QgdGFyZ2V0UmVzb3VyY2UgPSByZXNvdXJjZUluQ29tbW9uU3RhY2tGb3IodGFyZ2V0KTtcbiAgICBzb3VyY2VSZXNvdXJjZS5fYWRkUmVzb3VyY2VEZXBlbmRlbmN5KHRhcmdldFJlc291cmNlKTtcbiAgICBmdW5jdGlvbiByZXNvdXJjZUluQ29tbW9uU3RhY2tGb3IoZWxlbWVudDogQ2ZuUmVzb3VyY2UgfCBTdGFjayk6IENmblJlc291cmNlIHtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBTdGFjay5pc1N0YWNrKGVsZW1lbnQpID8gZWxlbWVudC5uZXN0ZWRTdGFja1Jlc291cmNlIDogZWxlbWVudDtcbiAgICAgICAgaWYgKCFyZXNvdXJjZSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdhc3NlcnRpb24gZmFpbHVyZScpOyAvLyBzZWUgXCJhc3NlcnRpb25cIiBhYm92ZVxuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHJlc291cmNlU3RhY2sgPSBTdGFjay5vZihyZXNvdXJjZSk7XG4gICAgICAgIC8vIHdlIHJlYWNoZWQgYSByZXNvdXJjZSBkZWZpbmVkIGluIHRoZSBjb21tb24gc3RhY2tcbiAgICAgICAgaWYgKGNvbW1vblN0YWNrID09PSByZXNvdXJjZVN0YWNrKSB7XG4gICAgICAgICAgICByZXR1cm4gcmVzb3VyY2U7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJlc291cmNlSW5Db21tb25TdGFja0ZvcihyZXNvdXJjZVN0YWNrKTtcbiAgICB9XG59XG4vKipcbiAqIFJldHVybiBhIHN0cmluZyByZXByZXNlbnRhdGlvbiBvZiB0aGUgZ2l2ZW4gYXNzZW1ibGVyLCBmb3IgdXNlIGluIGVycm9yIG1lc3NhZ2VzXG4gKi9cbmZ1bmN0aW9uIGRlc2NyaWJlU3RhZ2UoYXNzZW1ibHk6IFN0YWdlIHwgdW5kZWZpbmVkKTogc3RyaW5nIHtcbiAgICBpZiAoIWFzc2VtYmx5KSB7XG4gICAgICAgIHJldHVybiAnYW4gdW5yb290ZWQgY29uc3RydWN0IHRyZWUnO1xuICAgIH1cbiAgICBpZiAoIWFzc2VtYmx5LnBhcmVudFN0YWdlKSB7XG4gICAgICAgIHJldHVybiAndGhlIEFwcCc7XG4gICAgfVxuICAgIHJldHVybiBgU3RhZ2UgJyR7YXNzZW1ibHkubm9kZS5wYXRofSdgO1xufVxuIl19