"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Map = exports.isPositiveInteger = void 0;
const chain_1 = require("../chain");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Returns true if the value passed is a positive integer
 * @param value the value ti validate
 */
exports.isPositiveInteger = (value) => {
    const isFloat = Math.floor(value) !== value;
    const isNotPositiveInteger = value < 0 || value > Number.MAX_SAFE_INTEGER;
    return !isFloat && !isNotPositiveInteger;
};
/**
 * Define a Map state in the state machine
 *
 * A `Map` state can be used to run a set of steps for each element of an input array.
 * A Map state will execute the same steps for multiple entries of an array in the state input.
 *
 * While the Parallel state executes multiple branches of steps using the same input, a Map state
 * will execute the same steps for multiple entries of an array in the state input.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-map-state.html
 */
class Map extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
        this.maxConcurrency = props.maxConcurrency;
        this.itemsPath = props.itemsPath;
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define iterator state machine in Map
     */
    iterator(iterator) {
        const name = `Map ${this.stateId} Iterator`;
        super.addIterator(new state_graph_1.StateGraph(iterator.startState, name));
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.MAP,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderRetryCatch(),
            ...this.renderIterator(),
            ...this.renderItemsPath(),
            MaxConcurrency: this.maxConcurrency,
        };
    }
    /**
     * Validate this state
     */
    validate() {
        const errors = [];
        if (this.iteration === undefined) {
            errors.push('Map state must have a non-empty iterator');
        }
        if (this.maxConcurrency !== undefined && !exports.isPositiveInteger(this.maxConcurrency)) {
            errors.push('maxConcurrency has to be a positive integer');
        }
        return errors;
    }
    renderItemsPath() {
        return {
            ItemsPath: state_1.renderJsonPath(this.itemsPath),
        };
    }
}
exports.Map = Map;
//# sourceMappingURL=data:application/json;base64,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