"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZoneDelegationRecord = exports.MxRecord = exports.CaaAmazonRecord = exports.CaaRecord = exports.CaaTag = exports.SrvRecord = exports.TxtRecord = exports.CnameRecord = exports.AaaaRecord = exports.ARecord = exports.AddressRecordTarget = exports.RecordSet = exports.RecordTarget = exports.RecordType = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    /**
     * route traffic to a resource, such as a web server, using an IPv4 address in dotted decimal
     * notation
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AFormat
     */
    RecordType["A"] = "A";
    /**
     * route traffic to a resource, such as a web server, using an IPv6 address in colon-separated
     * hexadecimal format
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AAAAFormat
     */
    RecordType["AAAA"] = "AAAA";
    /**
     * A CAA record specifies which certificate authorities (CAs) are allowed to issue certificates
     * for a domain or subdomain
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CAAFormat
     */
    RecordType["CAA"] = "CAA";
    /**
     * A CNAME record maps DNS queries for the name of the current record, such as acme.example.com,
     * to another domain (example.com or example.net) or subdomain (acme.example.com or zenith.example.org).
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CNAMEFormat
     */
    RecordType["CNAME"] = "CNAME";
    /**
     * An MX record specifies the names of your mail servers and, if you have two or more mail servers,
     * the priority order.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#MXFormat
     */
    RecordType["MX"] = "MX";
    /**
     * A Name Authority Pointer (NAPTR) is a type of record that is used by Dynamic Delegation Discovery
     * System (DDDS) applications to convert one value to another or to replace one value with another.
     * For example, one common use is to convert phone numbers into SIP URIs.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NAPTRFormat
     */
    RecordType["NAPTR"] = "NAPTR";
    /**
     * An NS record identifies the name servers for the hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NSFormat
     */
    RecordType["NS"] = "NS";
    /**
     * A PTR record maps an IP address to the corresponding domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#PTRFormat
     */
    RecordType["PTR"] = "PTR";
    /**
     * A start of authority (SOA) record provides information about a domain and the corresponding Amazon
     * Route 53 hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SOAFormat
     */
    RecordType["SOA"] = "SOA";
    /**
     * SPF records were formerly used to verify the identity of the sender of email messages.
     * Instead of an SPF record, we recommend that you create a TXT record that contains the applicable value.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SPFFormat
     */
    RecordType["SPF"] = "SPF";
    /**
     * An SRV record Value element consists of four space-separated values. The first three values are
     * decimal numbers representing priority, weight, and port. The fourth value is a domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SRVFormat
     */
    RecordType["SRV"] = "SRV";
    /**
     * A TXT record contains one or more strings that are enclosed in double quotation marks (").
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#TXTFormat
     */
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    /**
     *
     * @param values correspond with the chosen record type (e.g. for 'A' Type, specify one ore more IP addresses)
     * @param aliasTarget alias for targets such as CloudFront distribution to route traffic to
     */
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip addresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment,
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * Target for a DNS A Record
 *
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.A,
            target: props.target,
        });
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.AAAA,
            target: props.target,
        });
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CNAME,
            target: RecordTarget.fromValues(props.domainName),
        });
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.TXT,
            target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))),
        });
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.SRV,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)),
        });
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CAA,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)),
        });
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com',
                },
            ],
        });
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.MX,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`)),
        });
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.NS,
            target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)),
            ttl: props.ttl || core_1.Duration.days(2),
        });
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVjb3JkLXNldC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlY29yZC1zZXQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQTZFLENBQUMsZ0RBQWdEO0FBRzlILDJEQUFtRDtBQUNuRCxpQ0FBMkQ7QUFVM0Q7O0dBRUc7QUFDSCxJQUFZLFVBbUZYO0FBbkZELFdBQVksVUFBVTtJQUNsQjs7Ozs7T0FLRztJQUNILHFCQUFPLENBQUE7SUFDUDs7Ozs7T0FLRztJQUNILDJCQUFhLENBQUE7SUFDYjs7Ozs7T0FLRztJQUNILHlCQUFXLENBQUE7SUFDWDs7Ozs7T0FLRztJQUNILDZCQUFlLENBQUE7SUFDZjs7Ozs7T0FLRztJQUNILHVCQUFTLENBQUE7SUFDVDs7Ozs7O09BTUc7SUFDSCw2QkFBZSxDQUFBO0lBQ2Y7Ozs7T0FJRztJQUNILHVCQUFTLENBQUE7SUFDVDs7OztPQUlHO0lBQ0gseUJBQVcsQ0FBQTtJQUNYOzs7OztPQUtHO0lBQ0gseUJBQVcsQ0FBQTtJQUNYOzs7OztPQUtHO0lBQ0gseUJBQVcsQ0FBQTtJQUNYOzs7OztPQUtHO0lBQ0gseUJBQVcsQ0FBQTtJQUNYOzs7O09BSUc7SUFDSCx5QkFBVyxDQUFBO0FBQ2YsQ0FBQyxFQW5GVyxVQUFVLEdBQVYsa0JBQVUsS0FBVixrQkFBVSxRQW1GckI7QUE0QkQ7O0dBRUc7QUFDSCxNQUFhLFlBQVk7SUFtQnJCOzs7O09BSUc7SUFDSCxZQUFzQyxNQUFpQixFQUFrQixXQUFnQztRQUFuRSxXQUFNLEdBQU4sTUFBTSxDQUFXO1FBQWtCLGdCQUFXLEdBQVgsV0FBVyxDQUFxQjtJQUN6RyxDQUFDO0lBeEJEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxHQUFHLE1BQWdCO1FBQ3hDLE9BQU8sSUFBSSxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxXQUErQjtRQUNuRCxPQUFPLElBQUksWUFBWSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsZUFBZSxDQUFDLEdBQUcsV0FBcUI7UUFDbEQsT0FBTyxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsV0FBVyxDQUFDLENBQUM7SUFDbkQsQ0FBQztDQVFKO0FBMUJELG9DQTBCQztBQWVEOztHQUVHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsZUFBUTtJQUVuQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFCO1FBQzNELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxHQUFHLElBQUksS0FBSyxDQUFDLEdBQUcsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzdHLE1BQU0sU0FBUyxHQUFHLElBQUksZ0NBQVksQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2pELFlBQVksRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVk7WUFDckMsSUFBSSxFQUFFLHdDQUFpQyxDQUFDLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQztZQUM1RixJQUFJLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDdEIsZUFBZSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsTUFBTTtZQUNwQyxXQUFXLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztZQUM1RSxHQUFHO1lBQ0gsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1NBQ3pCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxVQUFVLEdBQUcsU0FBUyxDQUFDLEdBQUcsQ0FBQztJQUNwQyxDQUFDO0NBQ0o7QUFoQkQsOEJBZ0JDO0FBQ0Q7Ozs7R0FJRztBQUNILE1BQWEsbUJBQW9CLFNBQVEsWUFBWTtDQUNwRDtBQURELGtEQUNDO0FBVUQ7Ozs7R0FJRztBQUNILE1BQWEsT0FBUSxTQUFRLFNBQVM7SUFDbEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFtQjtRQUN6RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLEdBQUcsS0FBSztZQUNSLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUN4QixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07U0FDdkIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQUNKO0FBUkQsMEJBUUM7QUFVRDs7OztHQUlHO0FBQ0gsTUFBYSxVQUFXLFNBQVEsU0FBUztJQUNyQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2IsR0FBRyxLQUFLO1lBQ1IsVUFBVSxFQUFFLFVBQVUsQ0FBQyxJQUFJO1lBQzNCLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtTQUN2QixDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFSRCxnQ0FRQztBQVVEOzs7O0dBSUc7QUFDSCxNQUFhLFdBQVksU0FBUSxTQUFTO0lBQ3RDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDN0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDYixHQUFHLEtBQUs7WUFDUixVQUFVLEVBQUUsVUFBVSxDQUFDLEtBQUs7WUFDNUIsTUFBTSxFQUFFLFlBQVksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztTQUNwRCxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFSRCxrQ0FRQztBQVVEOzs7O0dBSUc7QUFDSCxNQUFhLFNBQVUsU0FBUSxTQUFTO0lBQ3BDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBcUI7UUFDM0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDYixHQUFHLEtBQUs7WUFDUixVQUFVLEVBQUUsVUFBVSxDQUFDLEdBQUc7WUFDMUIsTUFBTSxFQUFFLFlBQVksQ0FBQyxVQUFVLENBQUMsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMvRSxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFSRCw4QkFRQztBQStCRDs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsU0FBUztJQUNwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFCO1FBQzNELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2IsR0FBRyxLQUFLO1lBQ1IsVUFBVSxFQUFFLFVBQVUsQ0FBQyxHQUFHO1lBQzFCLE1BQU0sRUFBRSxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxRQUFRLElBQUksQ0FBQyxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1NBQ2pILENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQVJELDhCQVFDO0FBQ0Q7O0dBRUc7QUFDSCxJQUFZLE1BZ0JYO0FBaEJELFdBQVksTUFBTTtJQUNkOzs7T0FHRztJQUNILHlCQUFlLENBQUE7SUFDZjs7O09BR0c7SUFDSCxpQ0FBdUIsQ0FBQTtJQUN2Qjs7O09BR0c7SUFDSCx5QkFBZSxDQUFBO0FBQ25CLENBQUMsRUFoQlcsTUFBTSxHQUFOLGNBQU0sS0FBTixjQUFNLFFBZ0JqQjtBQTJCRDs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsU0FBUztJQUNwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFCO1FBQzNELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2IsR0FBRyxLQUFLO1lBQ1IsVUFBVSxFQUFFLFVBQVUsQ0FBQyxHQUFHO1lBQzFCLE1BQU0sRUFBRSxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLENBQUMsS0FBSyxHQUFHLENBQUMsQ0FBQztTQUMvRixDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFSRCw4QkFRQztBQU1EOzs7Ozs7O0dBT0c7QUFDSCxNQUFhLGVBQWdCLFNBQVEsU0FBUztJQUMxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTJCO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2IsR0FBRyxLQUFLO1lBQ1IsTUFBTSxFQUFFO2dCQUNKO29CQUNJLElBQUksRUFBRSxDQUFDO29CQUNQLEdBQUcsRUFBRSxNQUFNLENBQUMsS0FBSztvQkFDakIsS0FBSyxFQUFFLFlBQVk7aUJBQ3RCO2FBQ0o7U0FDSixDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFiRCwwQ0FhQztBQXVCRDs7OztHQUlHO0FBQ0gsTUFBYSxRQUFTLFNBQVEsU0FBUztJQUNuQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2IsR0FBRyxLQUFLO1lBQ1IsVUFBVSxFQUFFLFVBQVUsQ0FBQyxFQUFFO1lBQ3pCLE1BQU0sRUFBRSxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxRQUFRLElBQUksQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDM0YsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQUNKO0FBUkQsNEJBUUM7QUFVRDs7R0FFRztBQUNILE1BQWEsb0JBQXFCLFNBQVEsU0FBUztJQUMvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWdDO1FBQ3RFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2IsR0FBRyxLQUFLO1lBQ1IsVUFBVSxFQUFFLFVBQVUsQ0FBQyxFQUFFO1lBQ3pCLE1BQU0sRUFBRSxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDO2dCQUNwRSxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxrQ0FBa0M7Z0JBQ3RELENBQUMsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQ2hHLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRyxJQUFJLGVBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1NBQ3JDLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQVhELG9EQVdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiwgSVJlc291cmNlLCBSZXNvdXJjZSwgVG9rZW4gfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IElBbGlhc1JlY29yZFRhcmdldCB9IGZyb20gJy4vYWxpYXMtcmVjb3JkLXRhcmdldCc7XG5pbXBvcnQgeyBJSG9zdGVkWm9uZSB9IGZyb20gJy4vaG9zdGVkLXpvbmUtcmVmJztcbmltcG9ydCB7IENmblJlY29yZFNldCB9IGZyb20gJy4vcm91dGU1My5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgZGV0ZXJtaW5lRnVsbHlRdWFsaWZpZWREb21haW5OYW1lIH0gZnJvbSAnLi91dGlsJztcbi8qKlxuICogQSByZWNvcmQgc2V0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVJlY29yZFNldCBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogVGhlIGRvbWFpbiBuYW1lIG9mIHRoZSByZWNvcmRcbiAgICAgKi9cbiAgICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG59XG4vKipcbiAqIFRoZSByZWNvcmQgdHlwZS5cbiAqL1xuZXhwb3J0IGVudW0gUmVjb3JkVHlwZSB7XG4gICAgLyoqXG4gICAgICogcm91dGUgdHJhZmZpYyB0byBhIHJlc291cmNlLCBzdWNoIGFzIGEgd2ViIHNlcnZlciwgdXNpbmcgYW4gSVB2NCBhZGRyZXNzIGluIGRvdHRlZCBkZWNpbWFsXG4gICAgICogbm90YXRpb25cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNBRm9ybWF0XG4gICAgICovXG4gICAgQSA9ICdBJyxcbiAgICAvKipcbiAgICAgKiByb3V0ZSB0cmFmZmljIHRvIGEgcmVzb3VyY2UsIHN1Y2ggYXMgYSB3ZWIgc2VydmVyLCB1c2luZyBhbiBJUHY2IGFkZHJlc3MgaW4gY29sb24tc2VwYXJhdGVkXG4gICAgICogaGV4YWRlY2ltYWwgZm9ybWF0XG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Sb3V0ZTUzL2xhdGVzdC9EZXZlbG9wZXJHdWlkZS9SZXNvdXJjZVJlY29yZFR5cGVzLmh0bWwjQUFBQUZvcm1hdFxuICAgICAqL1xuICAgIEFBQUEgPSAnQUFBQScsXG4gICAgLyoqXG4gICAgICogQSBDQUEgcmVjb3JkIHNwZWNpZmllcyB3aGljaCBjZXJ0aWZpY2F0ZSBhdXRob3JpdGllcyAoQ0FzKSBhcmUgYWxsb3dlZCB0byBpc3N1ZSBjZXJ0aWZpY2F0ZXNcbiAgICAgKiBmb3IgYSBkb21haW4gb3Igc3ViZG9tYWluXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Sb3V0ZTUzL2xhdGVzdC9EZXZlbG9wZXJHdWlkZS9SZXNvdXJjZVJlY29yZFR5cGVzLmh0bWwjQ0FBRm9ybWF0XG4gICAgICovXG4gICAgQ0FBID0gJ0NBQScsXG4gICAgLyoqXG4gICAgICogQSBDTkFNRSByZWNvcmQgbWFwcyBETlMgcXVlcmllcyBmb3IgdGhlIG5hbWUgb2YgdGhlIGN1cnJlbnQgcmVjb3JkLCBzdWNoIGFzIGFjbWUuZXhhbXBsZS5jb20sXG4gICAgICogdG8gYW5vdGhlciBkb21haW4gKGV4YW1wbGUuY29tIG9yIGV4YW1wbGUubmV0KSBvciBzdWJkb21haW4gKGFjbWUuZXhhbXBsZS5jb20gb3IgemVuaXRoLmV4YW1wbGUub3JnKS5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNDTkFNRUZvcm1hdFxuICAgICAqL1xuICAgIENOQU1FID0gJ0NOQU1FJyxcbiAgICAvKipcbiAgICAgKiBBbiBNWCByZWNvcmQgc3BlY2lmaWVzIHRoZSBuYW1lcyBvZiB5b3VyIG1haWwgc2VydmVycyBhbmQsIGlmIHlvdSBoYXZlIHR3byBvciBtb3JlIG1haWwgc2VydmVycyxcbiAgICAgKiB0aGUgcHJpb3JpdHkgb3JkZXIuXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Sb3V0ZTUzL2xhdGVzdC9EZXZlbG9wZXJHdWlkZS9SZXNvdXJjZVJlY29yZFR5cGVzLmh0bWwjTVhGb3JtYXRcbiAgICAgKi9cbiAgICBNWCA9ICdNWCcsXG4gICAgLyoqXG4gICAgICogQSBOYW1lIEF1dGhvcml0eSBQb2ludGVyIChOQVBUUikgaXMgYSB0eXBlIG9mIHJlY29yZCB0aGF0IGlzIHVzZWQgYnkgRHluYW1pYyBEZWxlZ2F0aW9uIERpc2NvdmVyeVxuICAgICAqIFN5c3RlbSAoREREUykgYXBwbGljYXRpb25zIHRvIGNvbnZlcnQgb25lIHZhbHVlIHRvIGFub3RoZXIgb3IgdG8gcmVwbGFjZSBvbmUgdmFsdWUgd2l0aCBhbm90aGVyLlxuICAgICAqIEZvciBleGFtcGxlLCBvbmUgY29tbW9uIHVzZSBpcyB0byBjb252ZXJ0IHBob25lIG51bWJlcnMgaW50byBTSVAgVVJJcy5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNOQVBUUkZvcm1hdFxuICAgICAqL1xuICAgIE5BUFRSID0gJ05BUFRSJyxcbiAgICAvKipcbiAgICAgKiBBbiBOUyByZWNvcmQgaWRlbnRpZmllcyB0aGUgbmFtZSBzZXJ2ZXJzIGZvciB0aGUgaG9zdGVkIHpvbmVcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNOU0Zvcm1hdFxuICAgICAqL1xuICAgIE5TID0gJ05TJyxcbiAgICAvKipcbiAgICAgKiBBIFBUUiByZWNvcmQgbWFwcyBhbiBJUCBhZGRyZXNzIHRvIHRoZSBjb3JyZXNwb25kaW5nIGRvbWFpbiBuYW1lLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vUm91dGU1My9sYXRlc3QvRGV2ZWxvcGVyR3VpZGUvUmVzb3VyY2VSZWNvcmRUeXBlcy5odG1sI1BUUkZvcm1hdFxuICAgICAqL1xuICAgIFBUUiA9ICdQVFInLFxuICAgIC8qKlxuICAgICAqIEEgc3RhcnQgb2YgYXV0aG9yaXR5IChTT0EpIHJlY29yZCBwcm92aWRlcyBpbmZvcm1hdGlvbiBhYm91dCBhIGRvbWFpbiBhbmQgdGhlIGNvcnJlc3BvbmRpbmcgQW1hem9uXG4gICAgICogUm91dGUgNTMgaG9zdGVkIHpvbmVcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNTT0FGb3JtYXRcbiAgICAgKi9cbiAgICBTT0EgPSAnU09BJyxcbiAgICAvKipcbiAgICAgKiBTUEYgcmVjb3JkcyB3ZXJlIGZvcm1lcmx5IHVzZWQgdG8gdmVyaWZ5IHRoZSBpZGVudGl0eSBvZiB0aGUgc2VuZGVyIG9mIGVtYWlsIG1lc3NhZ2VzLlxuICAgICAqIEluc3RlYWQgb2YgYW4gU1BGIHJlY29yZCwgd2UgcmVjb21tZW5kIHRoYXQgeW91IGNyZWF0ZSBhIFRYVCByZWNvcmQgdGhhdCBjb250YWlucyB0aGUgYXBwbGljYWJsZSB2YWx1ZS5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNTUEZGb3JtYXRcbiAgICAgKi9cbiAgICBTUEYgPSAnU1BGJyxcbiAgICAvKipcbiAgICAgKiBBbiBTUlYgcmVjb3JkIFZhbHVlIGVsZW1lbnQgY29uc2lzdHMgb2YgZm91ciBzcGFjZS1zZXBhcmF0ZWQgdmFsdWVzLiBUaGUgZmlyc3QgdGhyZWUgdmFsdWVzIGFyZVxuICAgICAqIGRlY2ltYWwgbnVtYmVycyByZXByZXNlbnRpbmcgcHJpb3JpdHksIHdlaWdodCwgYW5kIHBvcnQuIFRoZSBmb3VydGggdmFsdWUgaXMgYSBkb21haW4gbmFtZS5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL1JvdXRlNTMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL1Jlc291cmNlUmVjb3JkVHlwZXMuaHRtbCNTUlZGb3JtYXRcbiAgICAgKi9cbiAgICBTUlYgPSAnU1JWJyxcbiAgICAvKipcbiAgICAgKiBBIFRYVCByZWNvcmQgY29udGFpbnMgb25lIG9yIG1vcmUgc3RyaW5ncyB0aGF0IGFyZSBlbmNsb3NlZCBpbiBkb3VibGUgcXVvdGF0aW9uIG1hcmtzIChcIikuXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Sb3V0ZTUzL2xhdGVzdC9EZXZlbG9wZXJHdWlkZS9SZXNvdXJjZVJlY29yZFR5cGVzLmh0bWwjVFhURm9ybWF0XG4gICAgICovXG4gICAgVFhUID0gJ1RYVCdcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgYSBSZWNvcmRTZXQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVjb3JkU2V0T3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIGhvc3RlZCB6b25lIGluIHdoaWNoIHRvIGRlZmluZSB0aGUgbmV3IHJlY29yZC5cbiAgICAgKi9cbiAgICByZWFkb25seSB6b25lOiBJSG9zdGVkWm9uZTtcbiAgICAvKipcbiAgICAgKiBUaGUgZG9tYWluIG5hbWUgZm9yIHRoaXMgcmVjb3JkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgem9uZSByb290XG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVjb3JkTmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzb3VyY2UgcmVjb3JkIGNhY2hlIHRpbWUgdG8gbGl2ZSAoVFRMKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IER1cmF0aW9uLm1pbnV0ZXMoMzApXG4gICAgICovXG4gICAgcmVhZG9ubHkgdHRsPzogRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogQSBjb21tZW50IHRvIGFkZCBvbiB0aGUgcmVjb3JkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgbm8gY29tbWVudFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG59XG4vKipcbiAqIFR5cGUgdW5pb24gZm9yIGEgcmVjb3JkIHRoYXQgYWNjZXB0cyBtdWx0aXBsZSB0eXBlcyBvZiB0YXJnZXQuXG4gKi9cbmV4cG9ydCBjbGFzcyBSZWNvcmRUYXJnZXQge1xuICAgIC8qKlxuICAgICAqIFVzZSBzdHJpbmcgdmFsdWVzIGFzIHRhcmdldC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21WYWx1ZXMoLi4udmFsdWVzOiBzdHJpbmdbXSkge1xuICAgICAgICByZXR1cm4gbmV3IFJlY29yZFRhcmdldCh2YWx1ZXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBVc2UgYW4gYWxpYXMgYXMgdGFyZ2V0LlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUFsaWFzKGFsaWFzVGFyZ2V0OiBJQWxpYXNSZWNvcmRUYXJnZXQpIHtcbiAgICAgICAgcmV0dXJuIG5ldyBSZWNvcmRUYXJnZXQodW5kZWZpbmVkLCBhbGlhc1RhcmdldCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFVzZSBpcCBhZGRyZXNzZXMgYXMgdGFyZ2V0LlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUlwQWRkcmVzc2VzKC4uLmlwQWRkcmVzc2VzOiBzdHJpbmdbXSkge1xuICAgICAgICByZXR1cm4gUmVjb3JkVGFyZ2V0LmZyb21WYWx1ZXMoLi4uaXBBZGRyZXNzZXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKlxuICAgICAqIEBwYXJhbSB2YWx1ZXMgY29ycmVzcG9uZCB3aXRoIHRoZSBjaG9zZW4gcmVjb3JkIHR5cGUgKGUuZy4gZm9yICdBJyBUeXBlLCBzcGVjaWZ5IG9uZSBvcmUgbW9yZSBJUCBhZGRyZXNzZXMpXG4gICAgICogQHBhcmFtIGFsaWFzVGFyZ2V0IGFsaWFzIGZvciB0YXJnZXRzIHN1Y2ggYXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gdG8gcm91dGUgdHJhZmZpYyB0b1xuICAgICAqL1xuICAgIHByb3RlY3RlZCBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmFsdWVzPzogc3RyaW5nW10sIHB1YmxpYyByZWFkb25seSBhbGlhc1RhcmdldD86IElBbGlhc1JlY29yZFRhcmdldCkge1xuICAgIH1cbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgUmVjb3JkU2V0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlY29yZFNldFByb3BzIGV4dGVuZHMgUmVjb3JkU2V0T3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIHJlY29yZCB0eXBlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlY29yZFR5cGU6IFJlY29yZFR5cGU7XG4gICAgLyoqXG4gICAgICogVGhlIHRhcmdldCBmb3IgdGhpcyByZWNvcmQsIGVpdGhlciBgUmVjb3JkVGFyZ2V0LmZyb21WYWx1ZXMoKWAgb3JcbiAgICAgKiBgUmVjb3JkVGFyZ2V0LmZyb21BbGlhcygpYC5cbiAgICAgKi9cbiAgICByZWFkb25seSB0YXJnZXQ6IFJlY29yZFRhcmdldDtcbn1cbi8qKlxuICogQSByZWNvcmQgc2V0LlxuICovXG5leHBvcnQgY2xhc3MgUmVjb3JkU2V0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJUmVjb3JkU2V0IHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBSZWNvcmRTZXRQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICBjb25zdCB0dGwgPSBwcm9wcy50YXJnZXQuYWxpYXNUYXJnZXQgPyB1bmRlZmluZWQgOiAoKHByb3BzLnR0bCAmJiBwcm9wcy50dGwudG9TZWNvbmRzKCkpIHx8IDE4MDApLnRvU3RyaW5nKCk7XG4gICAgICAgIGNvbnN0IHJlY29yZFNldCA9IG5ldyBDZm5SZWNvcmRTZXQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgICAgICAgaG9zdGVkWm9uZUlkOiBwcm9wcy56b25lLmhvc3RlZFpvbmVJZCxcbiAgICAgICAgICAgIG5hbWU6IGRldGVybWluZUZ1bGx5UXVhbGlmaWVkRG9tYWluTmFtZShwcm9wcy5yZWNvcmROYW1lIHx8IHByb3BzLnpvbmUuem9uZU5hbWUsIHByb3BzLnpvbmUpLFxuICAgICAgICAgICAgdHlwZTogcHJvcHMucmVjb3JkVHlwZSxcbiAgICAgICAgICAgIHJlc291cmNlUmVjb3JkczogcHJvcHMudGFyZ2V0LnZhbHVlcyxcbiAgICAgICAgICAgIGFsaWFzVGFyZ2V0OiBwcm9wcy50YXJnZXQuYWxpYXNUYXJnZXQgJiYgcHJvcHMudGFyZ2V0LmFsaWFzVGFyZ2V0LmJpbmQodGhpcyksXG4gICAgICAgICAgICB0dGwsXG4gICAgICAgICAgICBjb21tZW50OiBwcm9wcy5jb21tZW50LFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5kb21haW5OYW1lID0gcmVjb3JkU2V0LnJlZjtcbiAgICB9XG59XG4vKipcbiAqIFRhcmdldCBmb3IgYSBETlMgQSBSZWNvcmRcbiAqXG4gKiBAZGVwcmVjYXRlZCBVc2UgUmVjb3JkVGFyZ2V0XG4gKi9cbmV4cG9ydCBjbGFzcyBBZGRyZXNzUmVjb3JkVGFyZ2V0IGV4dGVuZHMgUmVjb3JkVGFyZ2V0IHtcbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQVJlY29yZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBUmVjb3JkUHJvcHMgZXh0ZW5kcyBSZWNvcmRTZXRPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhcmdldDogUmVjb3JkVGFyZ2V0O1xufVxuLyoqXG4gKiBBIEROUyBBIHJlY29yZFxuICpcbiAqIEByZXNvdXJjZSBBV1M6OlJvdXRlNTM6OlJlY29yZFNldFxuICovXG5leHBvcnQgY2xhc3MgQVJlY29yZCBleHRlbmRzIFJlY29yZFNldCB7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFSZWNvcmRQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgcmVjb3JkVHlwZTogUmVjb3JkVHlwZS5BLFxuICAgICAgICAgICAgdGFyZ2V0OiBwcm9wcy50YXJnZXQsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQWFhYVJlY29yZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBYWFhUmVjb3JkUHJvcHMgZXh0ZW5kcyBSZWNvcmRTZXRPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhcmdldDogUmVjb3JkVGFyZ2V0O1xufVxuLyoqXG4gKiBBIEROUyBBQUFBIHJlY29yZFxuICpcbiAqIEByZXNvdXJjZSBBV1M6OlJvdXRlNTM6OlJlY29yZFNldFxuICovXG5leHBvcnQgY2xhc3MgQWFhYVJlY29yZCBleHRlbmRzIFJlY29yZFNldCB7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFhYWFSZWNvcmRQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgcmVjb3JkVHlwZTogUmVjb3JkVHlwZS5BQUFBLFxuICAgICAgICAgICAgdGFyZ2V0OiBwcm9wcy50YXJnZXQsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQ25hbWVSZWNvcmQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ25hbWVSZWNvcmRQcm9wcyBleHRlbmRzIFJlY29yZFNldE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBkb21haW4gbmFtZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG59XG4vKipcbiAqIEEgRE5TIENOQU1FIHJlY29yZFxuICpcbiAqIEByZXNvdXJjZSBBV1M6OlJvdXRlNTM6OlJlY29yZFNldFxuICovXG5leHBvcnQgY2xhc3MgQ25hbWVSZWNvcmQgZXh0ZW5kcyBSZWNvcmRTZXQge1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDbmFtZVJlY29yZFByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICByZWNvcmRUeXBlOiBSZWNvcmRUeXBlLkNOQU1FLFxuICAgICAgICAgICAgdGFyZ2V0OiBSZWNvcmRUYXJnZXQuZnJvbVZhbHVlcyhwcm9wcy5kb21haW5OYW1lKSxcbiAgICAgICAgfSk7XG4gICAgfVxufVxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBUeHRSZWNvcmQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHh0UmVjb3JkUHJvcHMgZXh0ZW5kcyBSZWNvcmRTZXRPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGV4dCB2YWx1ZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmFsdWVzOiBzdHJpbmdbXTtcbn1cbi8qKlxuICogQSBETlMgVFhUIHJlY29yZFxuICpcbiAqIEByZXNvdXJjZSBBV1M6OlJvdXRlNTM6OlJlY29yZFNldFxuICovXG5leHBvcnQgY2xhc3MgVHh0UmVjb3JkIGV4dGVuZHMgUmVjb3JkU2V0IHtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVHh0UmVjb3JkUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgICAgIHJlY29yZFR5cGU6IFJlY29yZFR5cGUuVFhULFxuICAgICAgICAgICAgdGFyZ2V0OiBSZWNvcmRUYXJnZXQuZnJvbVZhbHVlcyguLi5wcm9wcy52YWx1ZXMubWFwKHYgPT4gSlNPTi5zdHJpbmdpZnkodikpKSxcbiAgICAgICAgfSk7XG4gICAgfVxufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIFNSViByZWNvcmQgdmFsdWUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3J2UmVjb3JkVmFsdWUge1xuICAgIC8qKlxuICAgICAqIFRoZSBwcmlvcml0eS5cbiAgICAgKi9cbiAgICByZWFkb25seSBwcmlvcml0eTogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSB3ZWlnaHQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgd2VpZ2h0OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIHBvcnQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBzZXJ2ZXIgaG9zdCBuYW1lLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGhvc3ROYW1lOiBzdHJpbmc7XG59XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciBhIFNydlJlY29yZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTcnZSZWNvcmRQcm9wcyBleHRlbmRzIFJlY29yZFNldE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSB2YWx1ZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmFsdWVzOiBTcnZSZWNvcmRWYWx1ZVtdO1xufVxuLyoqXG4gKiBBIEROUyBTUlYgcmVjb3JkXG4gKlxuICogQHJlc291cmNlIEFXUzo6Um91dGU1Mzo6UmVjb3JkU2V0XG4gKi9cbmV4cG9ydCBjbGFzcyBTcnZSZWNvcmQgZXh0ZW5kcyBSZWNvcmRTZXQge1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTcnZSZWNvcmRQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgcmVjb3JkVHlwZTogUmVjb3JkVHlwZS5TUlYsXG4gICAgICAgICAgICB0YXJnZXQ6IFJlY29yZFRhcmdldC5mcm9tVmFsdWVzKC4uLnByb3BzLnZhbHVlcy5tYXAodiA9PiBgJHt2LnByaW9yaXR5fSAke3Yud2VpZ2h0fSAke3YucG9ydH0gJHt2Lmhvc3ROYW1lfWApKSxcbiAgICAgICAgfSk7XG4gICAgfVxufVxuLyoqXG4gKiBUaGUgQ0FBIHRhZy5cbiAqL1xuZXhwb3J0IGVudW0gQ2FhVGFnIHtcbiAgICAvKipcbiAgICAgKiBFeHBsaWNpdHkgYXV0aG9yaXplcyBhIHNpbmdsZSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkgdG8gaXNzdWUgYVxuICAgICAqIGNlcnRpZmljYXRlIChhbnkgdHlwZSkgZm9yIHRoZSBob3N0bmFtZS5cbiAgICAgKi9cbiAgICBJU1NVRSA9ICdpc3N1ZScsXG4gICAgLyoqXG4gICAgICogRXhwbGljaXR5IGF1dGhvcml6ZXMgYSBzaW5nbGUgY2VydGlmaWNhdGUgYXV0aG9yaXR5IHRvIGlzc3VlIGFcbiAgICAgKiB3aWxkY2FyZCBjZXJ0aWZpY2F0ZSAoYW5kIG9ubHkgd2lsZGNhcmQpIGZvciB0aGUgaG9zdG5hbWUuXG4gICAgICovXG4gICAgSVNTVUVXSUxEID0gJ2lzc3Vld2lsZCcsXG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIGEgVVJMIHRvIHdoaWNoIGEgY2VydGlmaWNhdGUgYXV0aG9yaXR5IG1heSByZXBvcnQgcG9saWN5XG4gICAgICogdmlvbGF0aW9ucy5cbiAgICAgKi9cbiAgICBJT0RFRiA9ICdpb2RlZidcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBDQUEgcmVjb3JkIHZhbHVlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhYVJlY29yZFZhbHVlIHtcbiAgICAvKipcbiAgICAgKiBUaGUgZmxhZy5cbiAgICAgKi9cbiAgICByZWFkb25seSBmbGFnOiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIHRhZy5cbiAgICAgKi9cbiAgICByZWFkb25seSB0YWc6IENhYVRhZztcbiAgICAvKipcbiAgICAgKiBUaGUgdmFsdWUgYXNzb2NpYXRlZCB3aXRoIHRoZSB0YWcuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQ2FhUmVjb3JkLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhYVJlY29yZFByb3BzIGV4dGVuZHMgUmVjb3JkU2V0T3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIHZhbHVlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSB2YWx1ZXM6IENhYVJlY29yZFZhbHVlW107XG59XG4vKipcbiAqIEEgRE5TIENBQSByZWNvcmRcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpSb3V0ZTUzOjpSZWNvcmRTZXRcbiAqL1xuZXhwb3J0IGNsYXNzIENhYVJlY29yZCBleHRlbmRzIFJlY29yZFNldCB7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENhYVJlY29yZFByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICByZWNvcmRUeXBlOiBSZWNvcmRUeXBlLkNBQSxcbiAgICAgICAgICAgIHRhcmdldDogUmVjb3JkVGFyZ2V0LmZyb21WYWx1ZXMoLi4ucHJvcHMudmFsdWVzLm1hcCh2ID0+IGAke3YuZmxhZ30gJHt2LnRhZ30gXCIke3YudmFsdWV9XCJgKSksXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQ2FhQW1hem9uUmVjb3JkLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhYUFtYXpvblJlY29yZFByb3BzIGV4dGVuZHMgUmVjb3JkU2V0T3B0aW9ucyB7XG59XG4vKipcbiAqIEEgRE5TIEFtYXpvbiBDQUEgcmVjb3JkLlxuICpcbiAqIEEgQ0FBIHJlY29yZCB0byByZXN0cmljdCBjZXJ0aWZpY2F0ZSBhdXRob3JpdGllcyBhbGxvd2VkXG4gKiB0byBpc3N1ZSBjZXJ0aWZpY2F0ZXMgZm9yIGEgZG9tYWluIHRvIEFtYXpvbiBvbmx5LlxuICpcbiAqIEByZXNvdXJjZSBBV1M6OlJvdXRlNTM6OlJlY29yZFNldFxuICovXG5leHBvcnQgY2xhc3MgQ2FhQW1hem9uUmVjb3JkIGV4dGVuZHMgQ2FhUmVjb3JkIHtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2FhQW1hem9uUmVjb3JkUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgICAgIHZhbHVlczogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgZmxhZzogMCxcbiAgICAgICAgICAgICAgICAgICAgdGFnOiBDYWFUYWcuSVNTVUUsXG4gICAgICAgICAgICAgICAgICAgIHZhbHVlOiAnYW1hem9uLmNvbScsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBNWCByZWNvcmQgdmFsdWUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTXhSZWNvcmRWYWx1ZSB7XG4gICAgLyoqXG4gICAgICogVGhlIHByaW9yaXR5LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByaW9yaXR5OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIG1haWwgc2VydmVyIGhvc3QgbmFtZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBob3N0TmFtZTogc3RyaW5nO1xufVxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBNeFJlY29yZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNeFJlY29yZFByb3BzIGV4dGVuZHMgUmVjb3JkU2V0T3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIHZhbHVlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSB2YWx1ZXM6IE14UmVjb3JkVmFsdWVbXTtcbn1cbi8qKlxuICogQSBETlMgTVggcmVjb3JkXG4gKlxuICogQHJlc291cmNlIEFXUzo6Um91dGU1Mzo6UmVjb3JkU2V0XG4gKi9cbmV4cG9ydCBjbGFzcyBNeFJlY29yZCBleHRlbmRzIFJlY29yZFNldCB7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IE14UmVjb3JkUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgICAgIHJlY29yZFR5cGU6IFJlY29yZFR5cGUuTVgsXG4gICAgICAgICAgICB0YXJnZXQ6IFJlY29yZFRhcmdldC5mcm9tVmFsdWVzKC4uLnByb3BzLnZhbHVlcy5tYXAodiA9PiBgJHt2LnByaW9yaXR5fSAke3YuaG9zdE5hbWV9YCkpLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciBhIFpvbmVEZWxlZ2F0aW9uUmVjb3JkXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgWm9uZURlbGVnYXRpb25SZWNvcmRQcm9wcyBleHRlbmRzIFJlY29yZFNldE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIHNlcnZlcnMgdG8gcmVwb3J0IGluIHRoZSBkZWxlZ2F0aW9uIHJlY29yZHMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZVNlcnZlcnM6IHN0cmluZ1tdO1xufVxuLyoqXG4gKiBBIHJlY29yZCB0byBkZWxlZ2F0ZSBmdXJ0aGVyIGxvb2t1cHMgdG8gYSBkaWZmZXJlbnQgc2V0IG9mIG5hbWUgc2VydmVycy5cbiAqL1xuZXhwb3J0IGNsYXNzIFpvbmVEZWxlZ2F0aW9uUmVjb3JkIGV4dGVuZHMgUmVjb3JkU2V0IHtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogWm9uZURlbGVnYXRpb25SZWNvcmRQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgcmVjb3JkVHlwZTogUmVjb3JkVHlwZS5OUyxcbiAgICAgICAgICAgIHRhcmdldDogUmVjb3JkVGFyZ2V0LmZyb21WYWx1ZXMoLi4uVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLm5hbWVTZXJ2ZXJzKVxuICAgICAgICAgICAgICAgID8gcHJvcHMubmFtZVNlcnZlcnMgLy8gQ2FuJ3QgbWFwIGEgc3RyaW5nLWFycmF5IHRva2VuIVxuICAgICAgICAgICAgICAgIDogcHJvcHMubmFtZVNlcnZlcnMubWFwKG5zID0+IChUb2tlbi5pc1VucmVzb2x2ZWQobnMpIHx8IG5zLmVuZHNXaXRoKCcuJykpID8gbnMgOiBgJHtuc30uYCkpLFxuICAgICAgICAgICAgdHRsOiBwcm9wcy50dGwgfHwgRHVyYXRpb24uZGF5cygyKSxcbiAgICAgICAgfSk7XG4gICAgfVxufVxuIl19