"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        let engine;
        if (this.dbCluster && this.dbInstance) {
            throw new Error('Proxy cannot target both database cluster and database instance.');
        }
        else if (this.dbCluster) {
            engine = this.dbCluster.node.defaultChild.engine;
        }
        else if (this.dbInstance) {
            engine = this.dbInstance.node.defaultChild.engine;
        }
        let engineFamily;
        switch (engine) {
            case 'aurora':
            case 'aurora-mysql':
            case 'mysql':
                engineFamily = 'MYSQL';
                break;
            case 'aurora-postgresql':
            case 'postgres':
                engineFamily = 'POSTGRESQL';
                break;
            default:
                throw new Error(`Unsupported engine type - ${engine}`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        props.secret.grantRead(role);
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: [
                {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: props.secret.secretArn,
                },
            ],
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        // Currently(2020-07-04), this property must be set to default.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#TargetGroupName-fn::getatt
        proxyTargetGroup.addOverride('Properties.TargetGroupName', 'default');
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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