"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const version_1 = require("./private/version");
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    // tslint:disable-next-line max-line-length
    constructor(name, singleUserRotationApplication, multiUserRotationApplication, parameterGroupFamilies) {
        this.name = name;
        this.singleUserRotationApplication = singleUserRotationApplication;
        this.multiUserRotationApplication = multiUserRotationApplication;
        this.parameterGroupFamilies = parameterGroupFamilies;
    }
    /**
     * Get the latest parameter group family for this engine. Latest is determined using semver on the engine major version.
     * When `engineVersion` is specified, return the parameter group family corresponding to that engine version.
     * Return undefined if no parameter group family is defined for this engine or for the requested `engineVersion`.
     */
    parameterGroupFamily(engineVersion) {
        if (this.parameterGroupFamilies === undefined) {
            return undefined;
        }
        if (engineVersion) {
            const family = this.parameterGroupFamilies.find(x => engineVersion.startsWith(x.engineMajorVersion));
            if (family) {
                return family.parameterGroupFamily;
            }
        }
        else if (this.parameterGroupFamilies.length > 0) {
            const sorted = this.parameterGroupFamilies.slice().sort((a, b) => {
                return version_1.compare(a.engineMajorVersion, b.engineMajorVersion);
            }).reverse();
            return sorted[0].parameterGroupFamily;
        }
        return undefined;
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/* tslint:disable max-line-length */
DatabaseClusterEngine.AURORA = new DatabaseClusterEngine('aurora', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'aurora5.6' },
]);
DatabaseClusterEngine.AURORA_MYSQL = new DatabaseClusterEngine('aurora-mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.7', parameterGroupFamily: 'aurora-mysql5.7' },
]);
DatabaseClusterEngine.AURORA_POSTGRESQL = new DatabaseClusterEngine('aurora-postgresql', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.6', parameterGroupFamily: 'aurora-postgresql9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'aurora-postgresql10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'aurora-postgresql11' },
]);
//# sourceMappingURL=data:application/json;base64,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