"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterParameterGroup = exports.ParameterGroup = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * A new cluster or instance parameter group
 */
class ParameterGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Parameters of the parameter group
         */
        this.parameters = {};
    }
    /**
     * Imports a parameter group
     */
    static fromParameterGroupName(scope, id, parameterGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.parameterGroupName = parameterGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a parameter to this parameter group
     *
     * @param key The key of the parameter to be added
     * @param value The value of the parameter to be added
     */
    addParameter(key, value) {
        if (!this.parameters) {
            this.parameters = {};
        }
        this.parameters[key] = value;
    }
}
/**
 * A parameter group
 *
 * @resource AWS::RDS::DBParameterGroup
 */
class ParameterGroup extends ParameterGroupBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.parameters = props.parameters ? props.parameters : {};
        const resource = new rds_generated_1.CfnDBParameterGroup(this, 'Resource', {
            description: props.description || `Parameter group for ${props.family}`,
            family: props.family,
            parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
        });
        this.parameterGroupName = resource.ref;
    }
}
exports.ParameterGroup = ParameterGroup;
/**
 * A cluster parameter group
 *
 * @resource AWS::RDS::DBClusterParameterGroup
 */
class ClusterParameterGroup extends ParameterGroupBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.parameters = props.parameters ? props.parameters : {};
        const resource = new rds_generated_1.CfnDBClusterParameterGroup(this, 'Resource', {
            description: props.description || `Cluster parameter group for ${props.family}`,
            family: props.family,
            parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
        });
        this.parameterGroupName = resource.ref;
    }
}
exports.ClusterParameterGroup = ClusterParameterGroup;
//# sourceMappingURL=data:application/json;base64,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