"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.PerformanceInsightRetention = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceEngine = exports.DatabaseInstanceBase = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a new db proxy to this instance.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine extends props_1.DatabaseClusterEngine {
    constructor() {
        super(...arguments);
        /* tslint:enable max-line-length */
        /** To make it a compile-time error to pass a DatabaseClusterEngine where a DatabaseInstanceEngine is expected. */
        this.isDatabaseInstanceEngine = true;
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/* tslint:disable max-line-length */
DatabaseInstanceEngine.MARIADB = new DatabaseInstanceEngine('mariadb', secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER, [
    { engineMajorVersion: '10.0', parameterGroupFamily: 'mariadb10.0' },
    { engineMajorVersion: '10.1', parameterGroupFamily: 'mariadb10.1' },
    { engineMajorVersion: '10.2', parameterGroupFamily: 'mariadb10.2' },
    { engineMajorVersion: '10.3', parameterGroupFamily: 'mariadb10.3' },
]);
DatabaseInstanceEngine.MYSQL = new DatabaseInstanceEngine('mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'mysql5.6' },
    { engineMajorVersion: '5.7', parameterGroupFamily: 'mysql5.7' },
    { engineMajorVersion: '8.0', parameterGroupFamily: 'mysql8.0' },
]);
DatabaseInstanceEngine.ORACLE_EE = new DatabaseInstanceEngine('oracle-ee', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-ee-11.2' },
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-ee-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-ee-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-ee-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-ee-19' },
]);
DatabaseInstanceEngine.ORACLE_SE2 = new DatabaseInstanceEngine('oracle-se2', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-se2-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-se2-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-se2-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-se2-19' },
]);
DatabaseInstanceEngine.ORACLE_SE1 = new DatabaseInstanceEngine('oracle-se1', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se1-11.2' },
]);
DatabaseInstanceEngine.ORACLE_SE = new DatabaseInstanceEngine('oracle-se', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se-11.2' },
]);
DatabaseInstanceEngine.POSTGRES = new DatabaseInstanceEngine('postgres', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.3', parameterGroupFamily: 'postgres9.3' },
    { engineMajorVersion: '9.4', parameterGroupFamily: 'postgres9.4' },
    { engineMajorVersion: '9.5', parameterGroupFamily: 'postgres9.5' },
    { engineMajorVersion: '9.6', parameterGroupFamily: 'postgres9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'postgres10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'postgres11' },
]);
DatabaseInstanceEngine.SQL_SERVER_EE = new DatabaseInstanceEngine('sqlserver-ee', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ee-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ee-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ee-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ee-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_SE = new DatabaseInstanceEngine('sqlserver-se', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-se-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-se-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-se-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-se-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_EX = new DatabaseInstanceEngine('sqlserver-ex', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ex-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ex-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ex-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ex-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_WEB = new DatabaseInstanceEngine('sqlserver-web', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-web-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-web-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-web-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-web-14.0' },
]);
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcPlacement = props.vpcPlacement;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcPlacement);
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'SubnetGroup', {
            dbSubnetGroupDescription: `Subnet group for ${this.node.id} database`,
            subnetIds,
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.numberValue({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const deletionProtection = props.deletionProtection !== undefined ? props.deletionProtection : true;
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention ? props.backupRetention.toDays() : undefined,
            copyTagsToSnapshot: props.copyTagsToSnapshot !== undefined ? props.copyTagsToSnapshot : true,
            dbInstanceClass: `db.${props.instanceType}`,
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection,
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: props.iamAuthentication,
            enablePerformanceInsights: props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: props.optionGroup && props.optionGroup.optionGroupName,
            performanceInsightsKmsKeyId: props.enablePerformanceInsights
                ? props.performanceInsightEncryptionKey && props.performanceInsightEncryptionKey.keyArn
                : undefined,
            performanceInsightsRetentionPeriod: props.enablePerformanceInsights
                ? (props.performanceInsightRetention || PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port ? props.port.toString() : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.vpcPlacement && props.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC,
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new lambda.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const timezoneSupport = [DatabaseInstanceEngine.SQL_SERVER_EE, DatabaseInstanceEngine.SQL_SERVER_EX,
            DatabaseInstanceEngine.SQL_SERVER_SE, DatabaseInstanceEngine.SQL_SERVER_WEB];
        if (props.timezone && !timezoneSupport.includes(props.engine)) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${props.engine.name}`);
        }
        this.sourceCfnProps = {
            ...this.newCfnProps,
            allocatedStorage: props.allocatedStorage ? props.allocatedStorage.toString() : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: props.parameterGroup && props.parameterGroup.parameterGroupName,
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (!props.masterUserPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (props.generateMasterUserPassword) {
            if (!props.masterUsername) { // We need the master username to include it in the generated secret
                throw new Error('`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
            }
            if (props.masterUserPassword) {
                throw new Error('Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        else {
            if (props.masterUsername) { // It's not possible to change the master username of a RDS instance
                throw new Error('Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
            }
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function applyInstanceDeletionPolicy(cfnDbInstance, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDbInstance.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDbInstance.applyRemovalPolicy(removalPolicy);
    }
}
//# sourceMappingURL=data:application/json;base64,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