"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const sqs = require("../../aws-sqs"); // Automatically re-written from '@aws-cdk/aws-sqs'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const log_retention_1 = require("./log-retention");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        this.environment = props.environment || {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additonal managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _a;
                    (_a = this.role) === null || _a === void 0 ? void 0 : _a.addToPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props.runtime);
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
            },
            layers: core_1.Lazy.listValue({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: core_1.Lazy.anyValue({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logretention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            resource.addPropertyOverride('FileSystemConfigs', [
                {
                    LocalMountPath: config.localMountPath,
                    Arn: config.arn,
                },
            ]);
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        return this._currentVersion;
    }
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key, value) {
        this.environment[key] = value;
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logretention.logGroupArn);
        }
        return this._logGroup;
    }
    prepare() {
        super.prepare();
        // if we have a current version resource, override it's logical id
        // so that it includes the hash of the function code and it's configuration.
        if (this._currentVersion) {
            const stack = core_1.Stack.of(this);
            const cfn = this._currentVersion.node.defaultChild;
            const originalLogicalId = stack.resolve(cfn.logicalId);
            const hash = function_hash_1.calculateFunctionHash(this);
            const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
            cfn.overrideLogicalId(`${logicalId}${hash}`);
        }
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        // for backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        if (!this._currentVersion) {
            return {
                variables: this.environment,
            };
        }
        // sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it).
        const variables = {};
        for (const key of Object.keys(this.environment).sort()) {
            variables[key] = this.environment[key];
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        // Pick subnets, make sure they're not Public. Routing through an IGW
        // won't work because the ENIs don't get a Public IP.
        // Why are we not simply forcing vpcSubnets? Because you might still be choosing
        // Isolated networks or selecting among 2 sets of Private subnets by name.
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId)) {
                throw new Error('Not possible to place Lambda Functions in a Public subnet');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, runtime) {
    // mutually exclusive
    if ((!code.inlineCode && !code.s3Location) || (code.inlineCode && code.s3Location)) {
        throw new Error('lambda.Code must specify one of "inlineCode" or "s3Location" but not both');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
//# sourceMappingURL=data:application/json;base64,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