"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositePrincipal = exports.Anyone = exports.AnyPrincipal = exports.AccountRootPrincipal = exports.OpenIdConnectPrincipal = exports.WebIdentityPrincipal = exports.FederatedPrincipal = exports.CanonicalUserPrincipal = exports.OrganizationPrincipal = exports.ServicePrincipal = exports.AccountPrincipal = exports.ArnPrincipal = exports.PrincipalPolicyFragment = exports.PrincipalWithConditions = exports.PrincipalBase = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const region_info_1 = require("../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return { statementAdded: false };
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the
     * passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 */
class PrincipalWithConditions {
    constructor(principal, conditions) {
        this.principal = principal;
        this.grantPrincipal = this;
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        return this.principal.addToPrincipalPolicy(statement);
    }
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            // merge the conditions if one of the additional conditions uses an
            // operator that's already used by the principal's conditions merge the
            // inner structure.
            const existing = mergedConditions[operator];
            if (!existing) {
                mergedConditions[operator] = condition;
                return; // continue
            }
            // if either the existing condition or the new one contain unresolved
            // tokens, fail the merge. this is as far as we go at this point.
            if (cdk.Token.isUnresolved(condition) || cdk.Token.isUnresolved(existing)) {
                throw new Error(`multiple "${operator}" conditions cannot be merged if one of them contains an unresolved token`);
            }
            mergedConditions[operator] = { ...existing, ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    /**
     *
     * @param principalJson JSON of the "Principal" section in a policy statement
     * @param conditions conditions that need to be applied to this policy
     */
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 */
class ArnPrincipal extends PrincipalBase {
    /**
     *
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 */
class AccountPrincipal extends ArnPrincipal {
    /**
     *
     * @param accountId AWS account ID (i.e. 123456789012)
     */
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    /**
     *
     * @param service AWS service (i.e. sqs.amazonaws.com)
     */
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString(),
            ],
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    /**
     *
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde)
     */
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    /**
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     *   root user and IAM users for an account all see the same ID.
     *   (i.e. 79a59df900b949e55d96a1e698fbacedfd6e09d98eacf8f8d5218e7cd47ef2be)
     */
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito,
 * that can be used to provide temporary security credentials to users who have been authenticated.
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 */
class FederatedPrincipal extends PrincipalBase {
    /**
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon,
 * Facebook, Google, etc.
 */
class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     *
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(identityProvider, conditions = {}) {
        super(identityProvider, conditions !== null && conditions !== void 0 ? conditions : {}, 'sts:AssumeRoleWithWebIdentity');
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `WebIdentityPrincipal(${this.federated})`;
    }
}
exports.WebIdentityPrincipal = WebIdentityPrincipal;
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 */
class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     *
     * @param openIdConnectProvider OpenID Connect provider
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(openIdConnectProvider, conditions = {}) {
        super(openIdConnectProvider.openIdConnectProviderArn, conditions !== null && conditions !== void 0 ? conditions : {});
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `OpenIdConnectPrincipal(${this.federated})`;
    }
}
exports.OpenIdConnectPrincipal = OpenIdConnectPrincipal;
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy
 */
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
/**
 * Represents a principal that has multiple types of principals. A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 */
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    /**
     * Adds IAM principals to the composite principal. Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal
     */
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service,
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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