"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeDependable = exports.Grant = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        if (result.success) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
//# sourceMappingURL=data:application/json;base64,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