"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkListener = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const util_1 = require("../shared/util");
const network_listener_action_1 = require("./network-listener-action");
const network_target_group_1 = require("./network-target-group");
/**
 * Define a Network Listener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class NetworkListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        const certs = props.certificates || [];
        const proto = props.protocol || (certs.length > 0 ? enums_1.Protocol.TLS : enums_1.Protocol.TCP);
        util_1.validateNetworkProtocol(proto);
        if (proto === enums_1.Protocol.TLS && certs.filter(v => v != null).length === 0) {
            throw new Error('When the protocol is set to TLS, you must specify certificates');
        }
        if (proto !== enums_1.Protocol.TLS && certs.length > 0) {
            throw new Error('Protocol must be TLS when certificates have been specified');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            protocol: proto,
            port: props.port,
            sslPolicy: props.sslPolicy,
            certificates: props.certificates,
        });
        this.loadBalancer = props.loadBalancer;
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(props.defaultTargetGroups));
        }
    }
    /**
     * Import an existing listener
     */
    static fromNetworkListenerArn(scope, id, networkListenerArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = networkListenerArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     */
    addTargetGroups(_id, ...targetGroups) {
        this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(targetGroups));
    }
    /**
     * Perform the given Action on incoming requests
     *
     * This allows full control of the default Action of the load balancer,
     * including weighted forwarding. See the `NetworkListenerAction` class for
     * all options.
     */
    addAction(_id, props) {
        this.setDefaultAction(props.action);
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates a NetworkTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or imported Load Balancer with specified VPC; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new network_target_group_1.NetworkTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            proxyProtocolV2: props.proxyProtocolV2,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, group);
        return group;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.NetworkListener = NetworkListener;
//# sourceMappingURL=data:application/json;base64,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