"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkListenerAction = void 0;
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 */
class NetworkListenerAction {
    /**
     * Create an instance of NetworkListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Forward to one or more Target Groups
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `NetworkListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.NetworkListenerAction = NetworkListenerAction;
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends NetworkListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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