"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
// tslint:disable:max-line-length
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
// tslint:enable
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_7_DAYS"] = 0] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_14_DAYS"] = 1] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_30_DAYS"] = 2] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_60_DAYS"] = 3] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_90_DAYS"] = 4] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * This is the general purpose performance mode for most file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * This performance mode can scale to higher levels of aggregate throughput and operations per second with a
     * tradeoff of slightly higher latencies.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     *  This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? Array.of({
                transitionToIa: LifecyclePolicy[props.lifecyclePolicy],
            }) : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tag.add(this, 'Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
                this.mountTargetsAvailable = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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