"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connections = void 0;
const peer_1 = require("./peer");
/**
 * Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 */
class Connections {
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * Add a security group to the list of security groups managed by this object
     */
    addSecurityGroup(...securityGroups) {
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * Allow connections to the peer on the given port
     */
    allowTo(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow connections from the peer on the given port
     */
    allowFrom(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow hosts inside the security group to connect to each other on the given port
     */
    allowInternally(portRange, description) {
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * Allow to all IPv4 ranges
     */
    allowToAnyIpv4(portRange, description) {
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow from any IPv4 ranges
     */
    allowFromAnyIpv4(portRange, description) {
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortFrom(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * Allow hosts inside the security group to connect to each other
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * Allow default connections from all IPv4 ranges
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * Allow connections to the security group on their default port
     */
    allowToDefaultPort(other, description) {
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call alloToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortTo(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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