"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.GeoRestriction = exports.ViewerCertificate = exports.LambdaEdgeEventType = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginProtocolPolicy = exports.OriginSslPolicy = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = void 0;
const certificatemanager = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
var PriceClass;
(function (PriceClass) {
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Controls the countries in which your content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
class CloudFrontWebDistribution extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018,
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // tslint:disable-next-line:max-line-length
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error('There must be at least one origin source - either an s3OriginSource or a customOriginSource');
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key],
                    };
                    originHeaders.push(oHeader);
                });
            }
            let s3OriginConfig;
            if (originConfig.s3OriginSource) {
                // first case for backwards compatibility
                if (originConfig.s3OriginSource.originAccessIdentity) {
                    // grant CloudFront OriginAccessIdentity read access to S3 bucket
                    originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                    s3OriginConfig = {
                        originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                    };
                }
                else {
                    s3OriginConfig = {};
                }
            }
            const connectionAttempts = (_a = originConfig.connectionAttempts) !== null && _a !== void 0 ? _a : 3;
            if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
                throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
            }
            const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
            if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
                throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeout
                            && originConfig.customOriginSource.originKeepaliveTimeout.toSeconds() || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeout
                            && originConfig.customOriginSource.originReadTimeout.toSeconds() || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || OriginProtocolPolicy.HTTPS_ONLY,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                    }
                    : undefined,
                connectionAttempts,
                connectionTimeout,
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // tslint:disable-next-line:max-line-length
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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