"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricAggregationType = exports.AdjustmentType = exports.StepScalingAction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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