"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib/");
test('Cloudfront Distribution Included', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "CustomErrorResponses": [
                {
                    "ErrorCode": 403,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/index.html"
                },
                {
                    "ErrorCode": 404,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/index.html"
                }
            ],
            "DefaultCacheBehavior": {
                "ViewerProtocolPolicy": "redirect-to-https"
            },
            "DefaultRootObject": "index.html",
            "HttpVersion": "http2",
            "IPV6Enabled": true,
            "PriceClass": "PriceClass_100",
            "ViewerCertificate": {
                "CloudFrontDefaultCertificate": true
            }
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    let stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup, IP Filter', () => {
    let stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Condition": {
                        "IpAddress": {
                            "aws:SourceIp": [
                                "1.1.1.1"
                            ]
                        }
                    },
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With IP Filter', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Condition": {
                        "IpAddress": {
                            "aws:SourceIp": [
                                "1.1.1.1"
                            ]
                        }
                    },
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Create From Hosted Zone', () => {
    let app = new core_1.App();
    let stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234'
        }
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.cdkspadeploy.com"
            ],
            "ViewerCertificate": {
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName
    });
    // THEN  
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName: exportName
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true
    });
    // THEN  
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName: exportName
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false
    });
    // THEN  
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName: exportName
    }));
});
//# sourceMappingURL=data:application/json;base64,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