"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config != 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config) {
        let bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true) {
            bucketConfig.publicReadAccess = false;
        }
        let bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true) {
            if (typeof this.globalConfig.ipList == 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(bucket.bucketArn + '/*');
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, cert) {
        let cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ],
            //We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: '/' + config.indexDoc,
                    responseCode: 200
                },
                {
                    errorCode: 404,
                    responsePagePath: '/' + config.indexDoc,
                    responseCode: 200
                }]
        };
        if (typeof config.certificateARN != 'undefined' && typeof config.cfAliases != 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases
            };
        }
        if (typeof config.zoneName != 'undefined' && typeof cert != 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        let cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName == 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     * It will also setup error forwarding and unauth forwarding back to indexDoc
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config);
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.domainName
        });
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     * configured by using the details in the hosted zone provided
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config);
        let zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        let cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution))
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: ['www.' + config.zoneName],
            targetDomain: config.zoneName,
        });
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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