"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackResourceRenamer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
/**
 * StackResourceRenamer renames stack name and stack's subordinate resources' custom physical names, so that a CDK stack can be used to create multiple stacks in same AWS environment.
 *
 * @stability stable
 */
class StackResourceRenamer {
    /**
     * Construct a new StackResourceRenamer.
     *
     * @param renameOper RenameOperation is used to rename stack name and resources' custom physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    constructor(renameOper, props = {}) {
        this.renameOper = renameOper;
        //mapping for resources whose physical names donot follow
        //the regular naming conventions: `${resourceType}`+'Name'
        this.irregularNames = {
            Stack: '_stackName',
            Output: '_exportName',
            ScalingPolicy: 'policyName',
            SlackChannelConfiguration: 'configurationName',
            CompositeAlarm: 'alarmName',
            SecurityGroup: 'groupName',
            DBProxy: 'dbProxyName',
        };
        this.defaultNameField = 'name';
        if (props.irregularResourceNames) {
            this.irregularNames = {
                ...this.irregularNames,
                ...props.irregularResourceNames,
            };
        }
        this.excludeResTypes = props.excludeResourceTypes;
        this.includeResTypes = props.includeResourceTypes;
    }
    /**
     * Static method to rename a stack and all its subordinate resources.
     *
     * @param stack The stack (and all its children resources) to be renamed.
     * @param renameOper RenameOperation is used to rename stack name and resources' custom physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    static rename(stack, renameOper, props = {}) {
        cdk.Aspects.of(stack).add(new StackResourceRenamer(renameOper, props));
    }
    /**
     * Implement core.IAspect interface.
     *
     * @param node CFN resources to be renamed.
     * @stability stable
     */
    visit(node) {
        if (node instanceof cdk.Stack) {
            //rename stack
            this.renameResource(node, 'Stack');
        }
        else {
            //rename CFN resources
            let ctorName = node.constructor.name;
            //console.log("==", ctorName)
            if (ctorName.startsWith('Cfn')) {
                this.renameResource(node, ctorName.substring(3));
            }
        }
    }
    /**
     * Rename a CFN resource or stack.
     *
     * @param node CFN resource or stack.
     * @param resTypeName The type name of CFN resource.
     * @stability stable
     */
    renameResource(node, resTypeName) {
        //check include/exclude
        if (this.excludeResTypes && this.excludeResTypes.length > 0 &&
            this.excludeResTypes.indexOf(resTypeName) !== -1) {
            return;
        }
        if (this.includeResTypes && this.includeResTypes.length > 0 &&
            this.includeResTypes.indexOf(resTypeName) === -1) {
            return;
        }
        //find the specific "name" field for CFN resources
        let physicalName = 'name';
        if (this.irregularNames[resTypeName]) {
            physicalName = this.irregularNames[resTypeName];
        }
        else {
            //decapitalize regular resource names
            let [first, ...rest] = resTypeName;
            let decapName = first.toLowerCase() + rest.join('');
            physicalName = `${decapName}Name`;
        }
        if (physicalName.length === 0) {
            return;
        }
        //some protected fields start with underscore
        let underscoreName = '_' + physicalName;
        //rename
        let b = node;
        if (b[physicalName] && b[physicalName].length > 0 && !cdk.Token.isUnresolved(b[physicalName])) {
            b[physicalName] = this.renameOper.rename(b[physicalName], resTypeName);
            //console.log("**** rename: ", b[physicalName]);
        }
        else if (b[underscoreName] && b[underscoreName].length > 0 && !cdk.Token.isUnresolved(b[underscoreName])) {
            b[underscoreName] = this.renameOper.rename(b[underscoreName], resTypeName);
            //console.log("**** rename: ", b[underscoreName]);
        }
        else if (b[this.defaultNameField] && b[this.defaultNameField].length > 0 && !cdk.Token.isUnresolved(b[this.defaultNameField])) {
            b[this.defaultNameField] = this.renameOper.rename(b[this.defaultNameField], resTypeName);
            //console.log("**** rename: ", b[this.defaultNameField]);
        }
    }
}
exports.StackResourceRenamer = StackResourceRenamer;
_a = JSII_RTTI_SYMBOL_1;
StackResourceRenamer[_a] = { fqn: "cdk-stack-resource-rename.StackResourceRenamer", version: "0.0.5" };
//# sourceMappingURL=data:application/json;base64,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