from __future__ import annotations

from typing import Any, Iterator, Union

from flow.record.fieldtypes.net import IPAddress, IPNetwork

from dissect.target.helpers.record import (
    MacInterfaceRecord,
    UnixInterfaceRecord,
    WindowsInterfaceRecord,
)
from dissect.target.plugin import Plugin, export, internal
from dissect.target.target import Target

InterfaceRecord = Union[UnixInterfaceRecord, WindowsInterfaceRecord, MacInterfaceRecord]


class NetworkPlugin(Plugin):
    __namespace__ = "network"

    def __init__(self, target: Target):
        super().__init__(target)
        self._interface_list: list[InterfaceRecord] | None = None

    def check_compatible(self) -> None:
        pass

    def _interfaces(self) -> Iterator[InterfaceRecord]:
        yield from ()

    def _get_record_type(self, field_name: str) -> Iterator[Any]:
        for record in self.interfaces():
            if (output := getattr(record, field_name, None)) is None:
                continue

            if isinstance(output, list):
                yield from output
            else:
                yield output

    @export(record=InterfaceRecord)
    def interfaces(self) -> Iterator[InterfaceRecord]:
        # Only search for the interfaces once
        if self._interface_list is None:
            self._interface_list = list(self._interfaces())

        yield from self._interface_list

    @export
    def ips(self) -> list[IPAddress]:
        return list(self._get_record_type("ip"))

    @export
    def gateways(self) -> list[IPAddress]:
        return list(self._get_record_type("gateway"))

    @export
    def macs(self) -> list[str]:
        return list(self._get_record_type("mac"))

    @export
    def dns(self) -> list[str]:
        return list(self._get_record_type("dns"))

    @internal
    def with_ip(self, ip_addr: str) -> Iterator[InterfaceRecord]:
        for interface in self.interfaces():
            if ip_addr in interface.ip:
                yield interface

    @internal
    def with_mac(self, mac: str) -> Iterator[InterfaceRecord]:
        for interface in self.interfaces():
            if interface.mac == mac:
                yield interface

    @internal
    def in_cidr(self, cidr: str) -> Iterator[InterfaceRecord]:
        cidr = IPNetwork(cidr)
        for interface in self.interfaces():
            if any(ip_addr in cidr for ip_addr in interface.ip):
                yield interface
