"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Modify an InstanceGroup with static ClusterId, InstanceGroupName, and InstanceGroup', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: 'ClusterId',
            instanceGroupName: 'InstanceGroupName',
            instanceGroup: {
                configurations: [{
                        classification: 'Classification',
                        properties: {
                            Key: 'Value',
                        },
                    }],
                eC2InstanceIdsToTerminate: ['InstanceToTerminate'],
                instanceCount: 1,
                shrinkPolicy: {
                    decommissionTimeout: cdk.Duration.seconds(1),
                    instanceResizePolicy: {
                        instanceTerminationTimeout: cdk.Duration.seconds(1),
                        instancesToProtect: ['InstanceToProtect'],
                        instancesToTerminate: ['InstanceToTerminate'],
                    },
                },
            },
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            InstanceGroupName: 'InstanceGroupName',
            InstanceGroup: {
                Configurations: [{
                        Classification: 'Classification',
                        Properties: {
                            Key: 'Value',
                        },
                    }],
                EC2InstanceIdsToTerminate: ['InstanceToTerminate'],
                InstanceCount: 1,
                ShrinkPolicy: {
                    DecommissionTimeout: 1,
                    InstanceResizePolicy: {
                        InstanceTerminationTimeout: 1,
                        InstancesToProtect: ['InstanceToProtect'],
                        InstancesToTerminate: ['InstanceToTerminate'],
                    },
                },
            },
        },
    });
});
test('Modify an InstanceGroup with ClusterId from payload and static InstanceGroupName and InstanceGroupConfiguration', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: sfn.Data.stringAt('$.ClusterId'),
            instanceGroupName: 'InstanceGroupName',
            instanceGroup: {
                instanceCount: 1,
            },
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId',
            'InstanceGroupName': 'InstanceGroupName',
            'InstanceGroup': {
                InstanceCount: 1,
            },
        },
    });
});
test('Modify an InstanceGroup with static ClusterId and InstanceGroupConfigurateion and InstanceGroupName from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: 'ClusterId',
            instanceGroupName: sfn.Data.stringAt('$.InstanceGroupName'),
            instanceGroup: {
                instanceCount: 1,
            },
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId': 'ClusterId',
            'InstanceGroupName.$': '$.InstanceGroupName',
            'InstanceGroup': {
                InstanceCount: 1,
            },
        },
    });
});
test('Modify an InstanceGroup with static ClusterId and InstanceGroupName and InstanceCount from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: 'ClusterId',
            instanceGroupName: 'InstanceGroupName',
            instanceGroup: {
                instanceCount: sfn.Data.numberAt('$.InstanceCount'),
            },
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            InstanceGroupName: 'InstanceGroupName',
            InstanceGroup: {
                'InstanceCount.$': '$.InstanceCount',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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