"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let vpc;
let cluster;
beforeEach(() => {
    // GIVEN
    stack = new core_1.Stack();
    vpc = new ec2.Vpc(stack, 'Vpc');
    cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
    cluster.addCapacity('Capacity', {
        instanceType: new ec2.InstanceType('t3.medium'),
    });
});
test('Cannot create a Fargate task with a fargate-incompatible task definition', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    expect(() => new tasks.RunEcsFargateTask({ cluster, taskDefinition }))
        .toThrowError(/not configured for compatibility with Fargate/);
});
test('Cannot create a Fargate task without a default container', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    expect(() => new tasks.RunEcsFargateTask({ cluster, taskDefinition }))
        .toThrowError(/must have at least one essential container/);
});
test('Running a Fargate Task', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'RunFargate', { task: new tasks.RunEcsFargateTask({
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            cluster,
            taskDefinition,
            containerOverrides: [
                {
                    containerName: 'TheContainer',
                    environment: [
                        { name: 'SOME_KEY', value: sfn.Data.stringAt('$.SomeKey') },
                    ],
                },
            ],
        }) });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'FARGATE',
            NetworkConfiguration: {
                AwsvpcConfiguration: {
                    SecurityGroups: [{ 'Fn::GetAtt': ['RunFargateSecurityGroup709740F2', 'GroupId'] }],
                    Subnets: [
                        { Ref: 'VpcPrivateSubnet1Subnet536B997A' },
                        { Ref: 'VpcPrivateSubnet2Subnet3788AAA1' },
                    ],
                },
            },
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                'Name': 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [{ 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] }],
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: { 'Fn::Join': ['', [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ]] },
                },
            ],
        },
    });
});
test('Running an EC2 Task with bridge network', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: new tasks.RunEcsEc2Task({
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            cluster,
            taskDefinition,
            containerOverrides: [
                {
                    containerName: 'TheContainer',
                    environment: [
                        { name: 'SOME_KEY', value: sfn.Data.stringAt('$.SomeKey') },
                    ],
                },
            ],
        }) });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                'Name': 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [{ 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] }],
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: { 'Fn::Join': ['', [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ]] },
                },
            ],
        },
    });
});
test('Running an EC2 Task with placement strategies', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    const ec2Task = new tasks.RunEcsEc2Task({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        placementStrategies: [
            ecs.PlacementStrategy.spreadAcrossInstances(),
            ecs.PlacementStrategy.packedByCpu(),
            ecs.PlacementStrategy.randomly(),
        ],
        placementConstraints: [ecs.PlacementConstraint.memberOf('blieptuut')],
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: ec2Task });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            PlacementConstraints: [
                { Type: 'memberOf', Expression: 'blieptuut' },
            ],
            PlacementStrategy: [
                { Field: 'instanceId', Type: 'spread' },
                { Field: 'cpu', Type: 'binpack' },
                { Type: 'random' },
            ],
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
test('Running an EC2 Task with overridden number values', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    const ec2Task = new tasks.RunEcsEc2Task({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        containerOverrides: [
            {
                containerName: 'TheContainer',
                command: sfn.Data.listAt('$.TheCommand'),
                cpu: 5,
                memoryLimit: sfn.Data.numberAt('$.MemoryLimit'),
            },
        ],
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: ec2Task });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        'Command.$': '$.TheCommand',
                        'Cpu': 5,
                        'Memory.$': '$.MemoryLimit',
                        'Name': 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
//# sourceMappingURL=data:application/json;base64,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