# -*- coding: utf-8 -*-
import json

from rich.table import Table
from rich.console import Console
from suite_py.lib import logger
from suite_py.lib.handler import prompt_utils
from suite_py.lib.handler.qainit_handler import QainitHandler
from suite_py.lib.handler import git_handler as git
from suite_py.lib.handler.git_handler import GitHandler


class QA:
    def __init__(self, action, project, config, tokens, flags=None):
        self._action = action
        self._project = project
        self._flags = flags
        self._config = config
        self._tokens = tokens
        self._git = GitHandler(project, config)
        self._qainit = QainitHandler(config, tokens)

    def run(self):
        if self._action == "list":
            self._list()
        elif self._action == "create":
            self._create()
        elif self._action == "update":
            self._update()
        elif self._action == "delete":
            self._delete()
        elif self._action == "check":
            self._check()
        elif self._action == "describe":
            self._describe()
        elif self._action == "update-quota":
            self._update_quota()

    def _check(self):
        r = self._qainit.execute("GET", "/api/v1/user")
        logger.info(
            "Checking configuration. If there is an issue, check ~/.suite_py/config.yml file and execute: suite-py login"
        )
        logger.debug(json.dumps(r.json(), indent=2))

    def _list(self):
        if self._flags:
            self._list_others(self._flags)
            return

        table = Table()
        table.add_column("Name", style="purple")
        table.add_column("Hash", style="green")
        table.add_column("Status", style="white")
        console = Console()

        r = self._qainit.execute("GET", "/api/v1/qa")

        qa_list = r.json()["list"]
        for qa in qa_list:
            table.add_row(
                qa["name"],
                qa["hash"],
                qa["status"],
            )
        console.print(table)

    def _list_others(self, user):
        table = Table()
        table.add_column("Name", style="purple")
        table.add_column("Hash", style="green")
        table.add_column("Status", style="white")
        console = Console()
        r = self._qainit.execute("GET", f"/api/v1/qa?user={user}")

        qa_list = r.json()["list"]
        for qa in qa_list:
            table.add_row(
                qa["name"],
                qa["hash"],
                qa["status"],
            )
        console.print(table)

    def _describe(self):
        qa_hash = self._flags["qa_hash"]
        jsonify = self._flags["json"]

        table = Table()
        table.add_column("Microservice", style="purple", no_wrap=True)
        table.add_column("Branch", style="green")
        table.add_column("Drone build")
        table.add_column("Status", style="white")
        console = Console()

        r = self._qainit.execute(
            "GET",
            f"/api/v1/qa/{qa_hash}",
        )
        try:
            resources_list = r.json()["list"]["resources"]
            for resource in resources_list:
                if resource["type"] == "microservice":
                    drone_url = (
                        "[blue][u]"
                        + "https://drone-1.prima.it/primait/"
                        + resource["name"]
                        + "/"
                        + resource["promoted_build"]
                        + "[/u][/blue]"
                    )
                    table.add_row(
                        resource["name"],
                        resource["ref"],
                        drone_url,
                        resource["status"],
                    )
            if jsonify:
                print(json.dumps(r.json(), indent=2))
            else:
                console.print(table)
        except TypeError:
            logger.error("Wrong hash")

    def _delete(self):
        qa_hash = self._flags

        r = self._qainit.execute(
            "DELETE",
            f"/api/v1/qa/{qa_hash}",
        )
        logger.info("QA deletion initiated")
        logger.debug(json.dumps(r.json(), indent=2))

    def _create(self):
        if "staging" in self._qainit.url:
            qa_default_name = (
                f"staging_{git.get_username()}_{self._git.current_branch_name()}"
            )
        else:
            qa_default_name = f"{git.get_username()}_{self._git.current_branch_name()}"

        qa_name = prompt_utils.ask_questions_input(
            "Choose the QA name: ", default_text=qa_default_name
        )

        srv_list = self._qainit.create_services_body(prj_list=self._flags)

        body = {"name": f"{qa_name}", "services": srv_list}
        logger.debug(json.dumps(body))
        r = self._qainit.execute(
            "POST",
            "/api/v1/qa",
            body=json.dumps(body),
        )
        logger.info("QA creation initiated")
        logger.debug(json.dumps(r.json(), indent=2))

    def _update(self):
        qa_hash = self._flags[0]
        prj_list = self._flags[1:]

        srv_list = self._qainit.create_services_body(prj_list)

        body = {"services": srv_list}
        logger.debug(json.dumps(body))
        r = self._qainit.execute(
            "PUT",
            f"/api/v1/qa/{qa_hash}",
            body=json.dumps(body),
        )
        logger.info("QA update initiated")
        logger.debug(json.dumps(r.json(), indent=2))

    def _update_quota(self):
        username = prompt_utils.ask_questions_input("Insert GitHub username: ")
        quota = prompt_utils.ask_questions_input("Insert new quota value: ")

        body = {"github_username": f"{username}", "quota": f"{quota}"}
        logger.debug(json.dumps(body))
        r = self._qainit.execute(
            "POST",
            "/api/v1/user/quota",
            body=json.dumps(body),
        )

        logger.info("Quota updated.")
        logger.debug(json.dumps(r.json(), indent=2))
