import configparser
import logging

import pandas as pd
from finvizfinance.screener import (
    financial,
    overview,
    ownership,
    performance,
    technical,
    valuation,
)

from openbb_terminal.core.config.paths import (
    MISCELLANEOUS_DIRECTORY,
    USER_PRESETS_DIRECTORY,
)
from openbb_terminal.decorators import log_start_end
from openbb_terminal.rich_config import console

logger = logging.getLogger(__name__)

PRESETS_PATH = USER_PRESETS_DIRECTORY / "stocks" / "screener"
PRESETS_PATH_DEFAULT = MISCELLANEOUS_DIRECTORY / "stocks" / "screener"
preset_choices = {
    filepath.name: filepath
    for filepath in PRESETS_PATH.iterdir()
    if filepath.suffix == ".ini"
}
preset_choices.update(
    {
        filepath.name: filepath
        for filepath in PRESETS_PATH_DEFAULT.iterdir()
        if filepath.suffix == ".ini"
    }
)

# pylint: disable=C0302

d_signals = {
    "top_gainers": "Top Gainers",
    "top_losers": "Top Losers",
    "new_high": "New High",
    "new_low": "New Low",
    "most_volatile": "Most Volatile",
    "most_active": "Most Active",
    "unusual_volume": "Unusual Volume",
    "overbought": "Overbought",
    "oversold": "Oversold",
    "downgrades": "Downgrades",
    "upgrades": "Upgrades",
    "earnings_before": "Earnings Before",
    "earnings_after": "Earnings After",
    "recent_insider_buying": "Recent Insider Buying",
    "recent_insider_selling": "Recent Insider Selling",
    "major_news": "Major News",
    "horizontal_sr": "Horizontal S/R",
    "tl_resistance": "TL Resistance",
    "tl_support": "TL Support",
    "wedge_up": "Wedge Up",
    "wedge_down": "Wedge Down",
    "wedge": "Wedge",
    "triangle_ascending": "Triangle Ascending",
    "triangle_descending": "Triangle Descending",
    "channel_up": "Channel Up",
    "channel_down": "Channel Down",
    "channel": "Channel",
    "double_top": "Double Top",
    "double_bottom": "Double Bottom",
    "multiple_top": "Multiple Top",
    "multiple_bottom": "Multiple Bottom",
    "head_shoulders": "Head & Shoulders",
    "head_shoulders_inverse": "Head & Shoulders Inverse",
}


@log_start_end(log=logger)
def get_screener_data(
    preset_loaded: str = "top_gainers",
    data_type: str = "overview",
    limit: int = 10,
    ascend: bool = False,
):
    """Screener Overview

    Parameters
    ----------
    preset_loaded : str
        Loaded preset filter
    data_type : str
        Data type between: overview, valuation, financial, ownership, performance, technical
    limit : int
        Limit of stocks filtered with presets to print
    ascend : bool
        Ascended order of stocks filtered to print

    Returns
    -------
    pd.DataFrame
        Dataframe with loaded filtered stocks
    """
    if data_type == "overview":
        screen = overview.Overview()
    elif data_type == "valuation":
        screen = valuation.Valuation()
    elif data_type == "financial":
        screen = financial.Financial()
    elif data_type == "ownership":
        screen = ownership.Ownership()
    elif data_type == "performance":
        screen = performance.Performance()
    elif data_type == "technical":
        screen = technical.Technical()
    else:
        console.print("Invalid selected screener type")
        return pd.DataFrame()

    if preset_loaded in d_signals:
        screen.set_filter(signal=d_signals[preset_loaded])

        try:
            if limit > 0:
                df_screen = screen.screener_view(limit=limit, ascend=ascend)
            else:
                df_screen = screen.screener_view(ascend=ascend)
        except IndexError:
            console.print("[red]Invalid data provided by the website[/red]\n")
            return pd.DataFrame()
        except AttributeError:
            console.print("[red]Invalid data provided by the website[/red]\n")
            return pd.DataFrame()

    else:
        preset_filter = configparser.RawConfigParser()
        preset_filter.optionxform = str  # type: ignore
        preset_filter.read(preset_choices[preset_loaded])

        d_general = preset_filter["General"]
        d_filters = {
            **preset_filter["Descriptive"],
            **preset_filter["Fundamental"],
            **preset_filter["Technical"],
        }

        for section in ["General", "Descriptive", "Fundamental", "Technical"]:
            for key, val in {**preset_filter[section]}.items():
                if key not in d_check_screener:
                    console.print(
                        f"The screener variable {section}.{key} shouldn't exist!\n"
                    )
                    return pd.DataFrame()

                if val not in d_check_screener[key]:
                    console.print(
                        f"Invalid [{section}] {key}={val}. "
                        f"Choose one of the following options:\n{', '.join(d_check_screener[key])}.\n"
                    )
                    return pd.DataFrame()

        d_filters = {k: v for k, v in d_filters.items() if v}

        screen.set_filter(filters_dict=d_filters)

        if "Order" in d_general:
            if limit > 0:
                df_screen = screen.screener_view(
                    order=d_general["Order"],
                    limit=limit,
                    ascend=ascend,
                )
            else:
                df_screen = screen.screener_view(
                    order=d_general["Order"], ascend=ascend
                )

        else:
            if limit > 0:
                df_screen = screen.screener_view(limit=limit, ascend=ascend)
            else:
                df_screen = screen.screener_view(ascend=ascend)

    return df_screen


d_signals_desc = {
    "top_gainers": "stocks with the highest %% price gain today",
    "top_losers": "stocks with the highest %% price loss today",
    "new_high": "stocks making 52-week high today",
    "new_low": "stocks making 52-week low today",
    "most_volatile": "stocks with the highest widest high/low trading range today",
    "most_active": "stocks with the highest trading volume today",
    "unusual_volume": "stocks with unusually high volume today - the highest relative volume ratio",
    "overbought": "stock is becoming overvalued and may experience a pullback.",
    "oversold": "oversold stocks may represent a buying opportunity for investors",
    "downgrades": "stocks downgraded by analysts today",
    "upgrades": "stocks upgraded by analysts today",
    "earnings_before": "companies reporting earnings today, before market open",
    "earnings_after": "companies reporting earnings today, after market close",
    "recent_insider_buying": "stocks with recent insider buying activity",
    "recent_insider_selling": "stocks with recent insider selling activity",
    "major_news": "stocks with the highest news coverage today",
    "horizontal_sr": "horizontal channel of price range between support and resistance trendlines",
    "tl_resistance": "once a rising trendline is broken",
    "tl_support": "once a falling trendline is broken",
    "wedge_up": "upward trendline support and upward trendline resistance (reversal)",
    "wedge_down": "downward trendline support and downward trendline resistance (reversal)",
    "wedge": "upward trendline support, downward trendline resistance (contiunation)",
    "triangle_ascending": "upward trendline support and horizontal trendline resistance",
    "triangle_descending": "horizontal trendline support and downward trendline resistance",
    "channel_up": "both support and resistance trendlines slope upward",
    "channel_down": "both support and resistance trendlines slope downward",
    "channel": "both support and resistance trendlines are horizontal",
    "double_top": "stock with 'M' shape that indicates a bearish reversal in trend",
    "double_bottom": "stock with 'W' shape that indicates a bullish reversal in trend",
    "multiple_top": "same as double_top hitting more highs",
    "multiple_bottom": "same as double_bottom hitting more lows",
    "head_shoulders": "chart formation that predicts a bullish-to-bearish trend reversal",
    "head_shoulders_inverse": "chart formation that predicts a bearish-to-bullish trend reversal",
}


d_check_screener = {
    "Order": [
        "Any",
        "Signal",
        "Ticker",
        "Company",
        "Sector",
        "Industry",
        "Country",
        "Market Cap.",
        "Price/Earnings",
        "Forward Price/Earnings",
        "PEG (Price/Earnings/Growth)",
        "Price/Sales",
        "Price/Book",
        "Price/Cash",
        "Price/Free Cash Flow",
        "Dividend Yield",
        "Payout Ratio",
        "EPS(ttm)",
        "EPS growth this year",
        "EPS growth next year",
        "EPS growth past 5 years",
        "EPS growth next 5 years",
        "Sales growth past 5 years",
        "EPS growth qtr over qtr",
        "Sales growth qtr over qtr",
        "Shares Outstanding",
        "Shares Float",
        "Insider Ownership",
        "Insider Transactions",
        "Institutional Ownership",
        "Institutional Transactions",
        "Short Interest Share",
        "Short Interest Ratio",
        "Earnings Date",
        "Return on Assets",
        "Return on Equity",
        "Return on Investment",
        "Current Ratio",
        "Quick Ratio",
        "LT Debt/Equity",
        "Total Debt/Equity",
        "Gross Margin",
        "Operating Margin",
        "Net Profit Margin",
        "Analyst Recommendation",
        "Performance (Week)",
        "Performance (Month)",
        "Performance (Quarter)",
        "Performance (Half Year)",
        "Performance (Year)",
        "Performance (Year To Date)",
        "Beta",
        "Average True Range",
        "Volatility (Week)",
        "Volatility (Month)",
        "20-Day SMA (Relative)",
        "50-Day SMA (Relative)",
        "200-Day SMA (Relative)",
        "50-Day High (Relative)",
        "50-Day Low (Relative)",
        "52-Week High (Relative)",
        "52-Week Low (Relative)",
        "Relative Strength Index (14)",
        "Average Volume (3 Month)",
        "Relative Volume",
        "Change",
        "Change from Open",
        "Gap",
        "Volume",
        "Price",
        "Target Price",
        "IPO Date",
    ],
    "Signal": [
        "Any",
        "Top Gainers",
        "Top Losers",
        "New High",
        "New Low",
        "Most Volatile",
        "Most Active",
        "Unusual Volume",
        "Overbought",
        "Oversold",
        "Downgrades",
        "Upgrades",
        "Earnings Before",
        "Earnings After",
        "Recent Insider Buying",
        "Recent Insider Selling",
        "Major News",
        "Horizontal S/R",
        "TL Resistance",
        "TL Support",
        "Wedge Up",
        "Wedge Down",
        "Triangle Ascending",
        "Triangle Descending",
        "Wedge",
        "Channel Up",
        "Channel Down",
        "Channel",
        "Double Top",
        "Double Bottom",
        "Multiple Top",
        "Multiple Bottom",
        "Head & Shoulders",
        "Head & Shoulders Inverse",
    ],
    "Exchange": ["Any", "AMEX", "NASDAQ", "NYSE"],
    "Market Cap.": [
        "Any",
        "Mega ($200bln and more)",
        "Large ($10bln to $200bln)",
        "Mid ($2bl to $10bln)",
        "Small ($300mln to $2bln)",
        "Micro ($50mln to $300mln)",
        "Nano (under $50mln)",
        "+Large (over $50mln)",
        "+Mid (over $2bln)",
        "+Small (over $300mln)",
        "+Micro (over $50mln)",
        "-Large (under $200bln)",
        "-Mid (under $10bln)",
        "-Small (under $2bln)",
        "-Micro (under $300mln)",
    ],
    "Earnings Date": [
        "Any",
        "Today",
        "Today Before Market Open",
        "Today Before Market Close",
        "Tomorrow",
        "Tomorrow Before Market Open",
        "Tomorrow Before Market Close",
        "Yesterday",
        "Yesterday Before Market Open",
        "Yesterday Before Market Close",
        "Next 5 Days",
        "Previous 5 Days",
        "This Week",
        "Next Week",
        "Previous Week",
        "This Month",
    ],
    "Target Price": [
        "Any",
        "50% Above Price",
        "40% Above Price",
        "30% Above Price",
        "20% Above Price",
        "10% Above Price",
        "5% Above Price",
        "Above Price",
        "Below Price",
        "5% Below Price",
        "10% Below Price",
        "20% Below Price",
        "30% Below Price",
        "40% Below Price",
        "50% Below Price",
    ],
    "Index": ["Any", "S&P 500", "DJIA"],
    "Dividend Yield": [
        "Any",
        "None (0%)",
        "Positive (>0%)",
        "High (>5%)",
        "Very High (>10%)",
        "Over 1%",
        "Over 2%",
        "Over 3%",
        "Over 4%",
        "Over 5%",
        "Over 6%",
        "Over 7%",
        "Over 8%",
        "Over 9%",
        "Over 10%",
    ],
    "Average Volume": [
        "Any",
        "Under 50K",
        "Under 100K",
        "Under 500K",
        "Under 750K",
        "Under 1M",
        "Over 50K",
        "Over 100K",
        "Over 200K",
        "Over 300K",
        "Over 400K",
        "Over 500K",
        "Over 750K",
        "Over 1M",
        "Over 2M",
        "100K to 500K",
        "100K to 1M",
        "500K to 1M",
        "500K to 10M",
    ],
    "IPO Date": [
        "Any",
        "Today",
        "Yesterday",
        "In the last week",
        "In the last month",
        "In the last quarter",
        "In the last year",
        "In the last 2 years",
        "In the last 3 years",
        "In the last 5 years",
        "More than a year ago",
        "More that 5 years ago",
        "More than 10 years ago",
        "More than 15 years ago",
        "More than 20 years ago",
        "More than 25 years ago",
    ],
    "Sector": [
        "Any",
        "Basic Materials",
        "Communication Services",
        "Consumer Cyclical",
        "Consumer Defensive",
        "Energy",
        "Financial",
        "Healthcare",
        "Industrials",
        "Real Estate",
        "Technology",
        "Utilities",
    ],
    "Float Short": [
        "Any",
        "Low (<5%)",
        "High(>20%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Relative Volume": [
        "Any",
        "Over 10",
        "Over 5",
        "Over 3",
        "Over 2",
        "Over 1.5",
        "Over 1",
        "Over 0.75",
        "Over 0.5",
        "Over 0.25",
        "Under 2",
        "Under 1",
        "Under 1.5",
        "Under 1",
        "Under 0.75",
        "Under 0.5",
        "Under 0.25",
        "Under 0.1",
    ],
    "Shares Outstanding": [
        "Any",
        "Under 1M",
        "Under 5M",
        "Under 10M",
        "Under 20M",
        "Under 50M",
        "Under 100M",
        "Over 1M",
        "Over 2M",
        "Over 5M",
        "Over 10M",
        "Over 20M",
        "Over 50M",
        "Over 100M",
        "Over 200M",
        "Over 500M",
        "Over 1000M",
    ],
    "Industry": [
        "Any",
        "Agricultural Inputs",
        "Aluminium",
        "Building Materials",
        "Chemicals",
        "Coking Coal",
        "Copper",
        "Gold",
        "Lumber & Wood Production",
        "Other Industrial Metals & Mining",
        "Other Precious Metals & Mining",
        "Paper & Paper Products",
        "Silver",
        "Specialty Chemicals",
        "Steel",
    ],
    "Analyst Recom.": [
        "Any",
        "Strong Buy (1)",
        "Buy or better",
        "Buy",
        "Hold or better",
        "Hold",
        "Hold or worse",
        "Sell",
        "Sell or worse",
        "Strong Sell (5)",
    ],
    "Current Volume": [
        "Any",
        "Under 100K",
        "Under 500K",
        "Under 750K",
        "Under 1M",
        "Over 0",
        "Over 50K",
        "Over 100K",
        "Over 200K",
        "Over 300K",
        "Over 400K",
        "Over 500K",
        "Over 750K",
        "Over 1M",
        "Over 2M",
        "Over 5M",
        "Over 10M",
        "Over 20M",
    ],
    "Float": [
        "Any",
        "Under 1M",
        "Under 5M",
        "Under 10M",
        "Under 20M",
        "Under 50M",
        "Under 100M",
        "Over 1M",
        "Over 2M",
        "Over 5M",
        "Over 10M",
        "Over 20M",
        "Over 50M",
        "Over 100M",
        "Over 200M",
        "Over 500M",
        "Over 1000M",
    ],
    "Country": [
        "Any",
        "Asia",
        "Europe",
        "Latin America",
        "BRIC",
        "Argentina",
        "Australia",
        "Bahamas",
        "Belgium",
        "BeNeLux",
        "Bermuda",
        "Brazil",
        "Canada",
        "Cayman Islands",
        "Chile",
        "China",
        "China & Hong Kong",
        "Colombia",
        "Cyprus",
        "Denmark",
        "Finland",
        "France",
        "Germany",
        "Greece",
        "Hong Kong",
        "Hungary",
        "Iceland",
        "Iceland",
        "India",
        "Indonesia",
        "Ireland",
        "Israel",
        "Italy",
        "Japan",
        "Kazakhstan",
        "Luxembourg",
        "Malaysia",
        "Malta",
        "Mexico",
        "Monaco",
        "Netherlands",
        "New Zealand",
        "Norway",
        "Panama",
        "Peru",
        "Philippines",
        "Portugal",
        "Russia",
        "Singapore",
        "South Africa",
        "South Korea",
        "Spain",
        "Sweden",
        "Switzerland",
        "Taiwan",
        "Turkey",
        "United Arab Emirates",
        "United Kingdom",
        "Uruguay",
        "USA",
        "Foreign (ex-USA)",
    ],
    "Option/Short": ["Any", "Optionable", "Shortable", "Optionable and shortable"],
    "Price": [
        "Any",
        "Under $1",
        "Under $2",
        "Under $3",
        "Under $4",
        "Under $5",
        "Under $7",
        "Under $10",
        "Under $15",
        "Under $20",
        "Under $30",
        "Under $40",
        "Under $50",
        "Over $1",
        "Over $2",
        "Over $3",
        "Over $4",
        "Over $5",
        "Over $7",
        "Over $10",
        "Over $15",
        "Over $20",
        "Over $30",
        "Over $40",
        "Over $50",
        "Over $60",
        "Over $70",
        "Over $80",
        "Over $90",
        "Over $100",
        "$1 to $5",
        "$1 to $10",
        "$1 to $20",
        "$5 to %10",
        "$5 to $20",
        "$5 to $50",
        "$10 to $20",
        "$10 to $50",
        "$20 to $50",
        "$50 to $100",
    ],
    "P/E": [
        "Any",
        "Low (<15)",
        "Profitable (>0)",
        "High (>50)",
        "Under 5",
        "Under 10",
        "Under 15",
        "Under 20",
        "Under 25",
        "Under 30",
        "Under 35",
        "Under 40",
        "Under 45",
        "Under 50",
        "Over 5",
        "Over 10",
        "Over 15",
        "Over 25",
        "Over 30",
        "Over 35",
        "Over 40",
        "Over 45",
        "Over 50",
    ],
    "Price/Cash": [
        "Any",
        "Low (<3)",
        "High (>50)",
        "Under 1",
        "Under 2",
        "Under 3",
        "Under 4",
        "Under 5",
        "Under 6",
        "Under 7",
        "Under 8",
        "Under 9",
        "Under 10",
        "Over 1",
        "Over 2",
        "Over 3",
        "Over 4",
        "Over 5",
        "Over 6",
        "Over 7",
        "Over 8",
        "Over 9",
        "Over 10",
        "Over 20",
        "Over 30",
        "Over 40",
        "Over 50",
    ],
    "EPS growthnext 5 years": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (<10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Return on Equity": [
        "Any",
        "Positive (>0%)",
        "Negative (<0%)",
        "Very Positive (>30%)",
        "Very Negative (<-15%)",
        "Under -50%",
        "Under -40%",
        "Under -35%",
        "Under -30%",
        "Under -25%",
        "Under -20%",
        "Under -15%",
        "Under -10%",
        "Under -5%",
        "Over +50%",
        "Over +45%",
        "Over +40%",
        "Over +35%",
        "Over +30%",
        "Over +25%",
        "Over +20%",
        "Over +15%",
        "Over +10%",
        "Over +5%",
    ],
    "Debt/Equity": [
        "Any",
        "High (>0.5)",
        "Low (<0.1)",
        "Under 1",
        "Under 0.9",
        "Under 0.8",
        "Under 0.7",
        "Under 0.6",
        "Under 0.5",
        "Under 0.4",
        "Under 0.3",
        "Under 0.2",
        "Under 0.1",
        "Over 0.1",
        "Over 0.2",
        "Over 0.3",
        "Over 0.4",
        "Over 0.5",
        "Over 0.6",
        "Over 0.7",
        "Over 0.8",
        "Over 0.9",
        "Over 1",
    ],
    "InsiderOwnership": [
        "Any",
        "Low (<5%)",
        "High (>30%)",
        "Very High (>50%)",
        "Over 10%",
        "Over 20%",
        "Over 30%",
        "Over 40%",
        "Over 50%",
        "Over 60%",
        "Over 70%",
        "Over 80%",
        "Over 90%",
    ],
    "Forward P/E": [
        "Any",
        "Low (<15)",
        "Profitable (>0)",
        "High (>50)",
        "Under 5",
        "Under 10",
        "Under 15",
        "Under 20",
        "Under 25",
        "Under 30",
        "Under 35",
        "Under 40",
        "Under 45",
        "Under 50",
        "Over 5",
        "Over 10",
        "Over 15",
        "Over 20",
        "Over 25",
        "Over 30",
        "Over 35",
        "Over 40",
        "Over 45",
        "Over 50",
    ],
    "Price/Free Cash Flow": [
        "Any",
        "Low (<15)",
        "High (>50)",
        "Under 5",
        "Under 10",
        "Under 15",
        "Under 20",
        "Under 25",
        "Under 30",
        "Under 35",
        "Under 40",
        "Under 45",
        "Under 50",
        "Under 60",
        "Under 70",
        "Under 80",
        "Under 90",
        "Under 100",
        "Over 5",
        "Over 10",
        "Over 15",
        "Over 20",
        "Over 25",
        "Over 30",
        "Over 35",
        "Over 40",
        "Over 45",
        "Over 50",
        "Over 60",
        "Over 70",
        "Over 80",
        "Over 90",
        "Over 100",
    ],
    "Sales growthpast 5 years": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (0-10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Return on Investment": [
        "Any",
        "Positive (>0%)",
        "Negative (<0%)",
        "Very Positive (>25%)",
        "Very Negative (<-10%)",
        "Under -50%",
        "Under -45%",
        "Under -40%",
        "Under -35%",
        "Under -30%",
        "Under -25%",
        "Under -20%",
        "Under -15%",
        "Under -10%",
        "Under -5%",
        "Over +5%",
        "Over +10%",
        "Over +15%",
        "Over +20%",
        "Over +25%",
        "Over +30%",
        "Over +35%",
        "Over +40%",
        "Over +45%",
        "Over +50%",
    ],
    "Gross Margin": [
        "Any",
        "Positive (>0%)",
        "Negative (<0%)",
        "High (>50%)",
        "Under 90%",
        "Under 80%",
        "Under 70%",
        "Under 60%",
        "Under 50%",
        "Under 45%",
        "Under 40%",
        "Under 35%",
        "Under 30%",
        "Under 25%",
        "Under 20%",
        "Under 15%",
        "Under 10%",
        "Under 5%",
        "Under 0%",
        "Under -10%",
        "Under -20%",
        "Under -30%",
        "Under -50%",
        "Under -70%",
        "Under -100%",
        "Over 0%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
        "Over 40%",
        "Over 45%",
        "Over 50%",
        "Over 60%",
        "Over 70%",
        "Over 80%",
        "Over 90%",
    ],
    "InsiderTransactions": [
        "Any",
        "Very Negative (<20%)",
        "Negative (<0%)",
        "Positive (>0%)",
        "Very Positive (>20%)",
        "Under -90%",
        "Under 80%",
        "Under 70%",
        "Under -60%",
        "Under -50%",
        "Under -45%",
        "Under 40%",
        "Under -35%",
        "Under -30%",
        "Under -25%",
        "Under -20%",
        "Under -15%",
        "Under -10%",
        "Under -5%",
        "Over +5%",
        "Over +10%",
        "Over +15%",
        "Over +20%",
        "Over +25%",
        "Over +30%",
        "Over +35%",
        "Over +40%",
        "Over +45%",
        "Over +50%",
        "Over +60%",
        "Over +70%",
        "Over +80%",
        "Over +90%",
    ],
    "PEG": [
        "Any",
        "Low (<1)",
        "High (>2)",
        "Under 1",
        "Under 2",
        "Under 3",
        "Over 1",
        "Over 2",
        "Over 3",
    ],
    "EPS growththis year": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (0-10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "EPS growthqtr over qtr": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (0-10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Current Ratio": [
        "Any",
        "High (>3)",
        "Low (<1)",
        "Under 1",
        "Under 0.5",
        "Over 0.5",
        "Over 1",
        "Over 1.5",
        "Over 2",
        "Over 3",
        "Over 4",
        "Over 5",
        "Over 10",
    ],
    "Operating Margin": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Very Negative (<-20%)",
        "High (>25%)",
        "Under 90%",
        "Under 80%",
        "Under 70%",
        "Under 60%",
        "Under 50%",
        "Under 45%",
        "Under 40%",
        "Under 35%",
        "Under 30%",
        "Under 25%",
        "Under 20%",
        "Under 15%",
        "Under 10%",
        "Under 5%",
        "Under 0%",
        "Under -10%",
        "Under -20%",
        "Under -30%",
        "Under -50%",
        "Under -70%",
        "Under -100%",
        "Over 0%",
        "Over 10%",
        "Under 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
        "Over 35%",
        "Over 40%",
        "Over 45%",
        "Over 50%",
        "Over 60%",
        "Over 70%",
        "Over 80%",
        "Over 90%",
    ],
    "InstitutionalOwnership": [
        "Any",
        "Low (<5%)",
        "High (>90%)",
        "Under 90%",
        "Under 80%",
        "Under 70%",
        "Under 60%",
        "Under 50%",
        "Under 40%",
        "Under 30%",
        "Under 20%",
        "Under 10%",
        "Over 10%",
        "Over 20%",
        "Over 30%",
        "Over 40%",
        "Over 50%",
        "Over 60%",
        "Over 70%",
        "Over 80%",
        "Over 90%",
    ],
    "P/S": [
        "Any",
        "Low (<1)",
        "High (>10)",
        "Under 1",
        "Under 2",
        "Under 3",
        "Under 4",
        "Under 5",
        "Under 6",
        "Under 7",
        "Under 8",
        "Under 9",
        "Under 10",
        "Over 1",
        "Over 2",
        "Over 3",
        "Over 4",
        "Over 5",
        "Over 6",
        "Over 6",
        "Over 7",
        "Over 8",
        "Over 9",
        "Over 10",
    ],
    "EPS growthnext year": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (0-10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Sales growthqtr over qtr": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (0-10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Quick Ratio": [
        "Any",
        "High (>3)",
        "Low (<0.5)",
        "Under 1",
        "Under 0.5",
        "Over 0.5",
        "Over 1",
        "Over 1.5",
        "Over 2",
        "Over 3",
        "Over 4",
        "Over 5",
        "Over 10",
    ],
    "Net Profit Margin": [
        "Any",
        "Positive (>0%)",
        "Negative (<0%)",
        "Very Negative (<-20%)",
        "High (>20%)",
        "Under 90%",
        "Under 80%",
        "Under 70%",
        "Under 60%",
        "Under 50%",
        "Under 45%",
        "Under 40%",
        "Under 35%",
        "Under 30%",
        "Under 25%",
        "Under 20%",
        "Under 15%",
        "Under 10%",
        "Under 5%",
        "Under 0%",
        "Under -10%",
        "Under -20%",
        "Under -30%",
        "Under -50%",
        "Under -70%",
        "Under -100%",
        "Over 0%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
        "Over 35%",
        "Over 40%",
        "Over 45%",
        "Over 50%",
        "Over 60%",
        "Over 70%",
        "Over 80%",
        "Over 90%",
    ],
    "InstitutionalTransactions": [
        "Any",
        "Very Negative (<20%)",
        "Negative (<0%)",
        "Positive (>0%)",
        "Very Positive (>20%)",
        "Under -50%",
        "Under -45%",
        "Under -40%",
        "Under -35%",
        "Under -30%",
        "Under -25%",
        "Under -20%",
        "Under -15%",
        "Under -10%",
        "Under -5%",
        "Over +5%",
        "Over +10%",
        "Over +15%",
        "Over +20%",
        "Over +25%",
        "Over +30%",
        "Over +35%",
        "Over +40%",
        "Over +45%",
        "Over +50%",
    ],
    "P/B": [
        "Any",
        "Low (<1)",
        "High (>5)",
        "Under 1",
        "Under 2",
        "Under 3",
        "Under 4",
        "Under 5",
        "Under 6",
        "Under 7",
        "Under 8",
        "Under 9",
        "Under 10",
        "Over 1",
        "Over 2",
        "Over 3",
        "Over 4",
        "Over 5",
        "Over 6",
        "Over 7",
        "Over 8",
        "Over 9",
        "Over 10",
    ],
    "EPS growthpast 5 years": [
        "Any",
        "Negative (<0%)",
        "Positive (>0%)",
        "Positive Low (0-10%)",
        "High (>25%)",
        "Under 5%",
        "Under 10%",
        "Under 15%",
        "Under 20%",
        "Under 25%",
        "Under 30%",
        "Over 5%",
        "Over 10%",
        "Over 15%",
        "Over 20%",
        "Over 25%",
        "Over 30%",
    ],
    "Return on Assets": [
        "Any",
        "Positive (>0%)",
        "Negative (<0%)",
        "Very Positive (>15%)",
        "Very Negative (<-15%)",
        "Under -50%",
        "Under -45%",
        "Under -40%",
        "Under -35%",
        "Under -30%",
        "Under -25%",
        "Under -20%",
        "Under -15%",
        "Under -10%",
        "Under -5%",
        "Over +5%",
        "Over +10%",
        "Over +15%",
        "Over +20%",
        "Over +25%",
        "Over +30%",
        "Over +35%",
        "Over +40%",
        "Over +45%",
        "Over +50%",
    ],
    "LT Debt/Equity": [
        "Any",
        "High (>0.5)",
        "Low (<0.1)",
        "Under 1",
        "Under 0.9",
        "Under 0.8",
        "Under 0.7",
        "Under 0.6",
        "Under 0.5",
        "Under 0.4",
        "Under 0.3",
        "Under 0.2",
        "Under 0.1",
        "Over 0.1",
        "Over 0.2",
        "Over 0.3",
        "Over 0.4",
        "Over.5",
        "Over 0.6",
        "Over 0.7",
        "Over 0.8",
        "Over 0.9",
        "Over 1",
    ],
    "Payout Ratio": [
        "Any",
        "None (0%)",
        "Positive (>0%)",
        "Low (<20%)",
        "High (>50%)",
        "Over 0%",
        "Over 10%",
        "Over 20%",
        "Over 30%",
        "Over 40%",
        "Over 50%",
        "Over 60%",
        "Over 70%",
        "Over 80%",
        "Over 90%",
        "Over 100%",
        "Under 10%",
        "Under 20%",
        "Under 30%",
        "Under 40%",
        "Under 50%",
        "Under 60%",
        "Under 70%",
        "Under 80%",
        "Under 90%",
        "Under 100%",
    ],
    "Performance": [
        "Any",
        "Today Up",
        "Today Down",
        "Today -15%",
        "Today -10%",
        "Today -5%",
        "Today +5%",
        "Today +10%",
        "Today +15%",
        "Week -30%",
        "Week -20%",
        "Week -10%",
        "Week Down",
        "Week Up",
        "Week +10%",
        "Week +20%",
        "Week +30%",
        "Month -50%",
        "Month -30%",
        "Month -20%",
        "Month -10%",
        "Month Down",
        "Month Up",
        "Month +10%",
        "Month +20%",
        "Month +30%",
        "Month +50%",
        "Quarter -50%",
        "Quarter -30%",
        "Quarter -20%",
        "Quarter -10%",
        "Quarter Down",
        "Quarter Up",
        "Quarter +10%",
        "Quarter +20%",
        "Quarter +30%",
        "Quarter +50%",
        "Half -75%",
        "Half -50%",
        "Half -30%",
        "Half -20%",
        "Half -10%",
        "Half Down",
        "Half Up",
        "Half +10%",
        "Half +20%",
        "Half +30%",
        "Half +50%",
        "Half +100%",
        "Year -75%",
        "Year -50%",
        "Year -30%",
        "Year -20%",
        "Year -10%",
        "Year Down",
        "Year Up",
        "Year +10%",
        "Year +20%",
        "Year +30%",
        "Year +50%",
        "Year +100%",
        "Year +200%",
        "Year +300%",
        "Year +500%",
        "YTD -75%",
        "YTD -50%",
        "YTD -30%",
        "YTD -20%",
        "YTD -10%",
        "YTD -5%",
        "YTD Down",
        "YTD Up",
        "YTD +5%",
        "YTD +10%",
        "YTD +20%",
        "YTD +30",
        "YTD +50%",
        "YTD +100%",
    ],
    "20-Day Simple Moving Average": [
        "Any",
        "Price below SMA20",
        "Price 10% below SMA20",
        "Price 20% below SMA20",
        "Price 30% below SMA20",
        "Price 40% below SMA20",
        "Price 50% below SMA20",
        "Price above SMA20",
        "Price 10% above SMA20",
        "Price 20% above SMA20",
        "Price 30% above SMA20",
        "Price 40% above SMA20",
        "Price 50% above SMA20",
        "Price crossed SMA20",
        "Price crossed SMA20 above",
        "Price crossed SMA20 below",
        "SMA20 crossed SMA50",
        "SMA20 crossed SMA50 above",
        "SMA20 crossed SMA50 below",
        "SMA20 cross SMA200",
        "SMA20 crossed SMA200 below",
        "SMA20 crossed SMA200 above",
        "SMA20 above SMA50",
        "SMA20 below SMA50",
        "SMA20 above SMA200",
        "SMA20 below SMA200",
    ],
    "20-Day High/Low": [
        "Any",
        "New High",
        "New Low",
        "5% or more below High",
        "10% or more below High",
        "15% or more below High",
        "20% or more below High",
        "30% or more below High",
        "40% or more below High",
        "50% or more below High",
        "0-3% below High",
        "0-5% below High",
        "0-10% below High",
        "5% or more above Low",
        "10% or more above Low",
        "15% or more above Low",
        "20% or more above Low",
        "30% or more above Low",
        "40% or more above Low",
        "50% or more above Low",
        "0-3% above Low",
        "0-5% above Low",
        "0-10% above Low",
    ],
    "Beta": [
        "Any",
        "Under 0",
        "Under 0.5",
        "Under 1",
        "Under 1.5",
        "Under 2",
        "Over 0",
        "Over 0.5",
        "Over 1",
        "Over 1.5",
        "Over 2",
        "Over 2.5",
        "Over 3",
        "Over 4",
        "0 to 0.5",
        "0 to 1",
        "0.5 to 1",
        "0.5 to 1.5",
        "1 to 1.5",
        "1 to 2",
    ],
    "Performance 2": [
        "Any",
        "Today Up",
        "Today Down",
        "Today -15%",
        "Today -10%",
        "Today -5%",
        "Today +5%",
        "Today +10%",
        "Today +15%",
        "Week -30%",
        "Week -20%",
        "Week -10%",
        "Week Down",
        "Week Up",
        "Week +10%",
        "Week +20%",
        "Week +30%",
        "Month -50%",
        "Month -30%",
        "Month -20%",
        "Month -10%",
        "Month Down",
        "Month Up",
        "Month +10%",
        "Month +20%",
        "Month +30%",
        "Month +50%",
        "Quarter -50%",
        "Quarter -30%",
        "Quarter -20%",
        "Quarter -10%",
        "Quarter Down",
        "Quarter Up",
        "Quarter +10%",
        "Quarter +20%",
        "Quarter +30%",
        "Quarter +50%",
        "Half -75%",
        "Half -50%",
        "Half -30%",
        "Half -20%",
        "Half -10%",
        "Half Down",
        "Half Up",
        "Half +10%",
        "Half +20%",
        "Half +30%",
        "Half +50%",
        "Half +100%",
        "Year -75%",
        "Year -50%",
        "Year -30%",
        "Year -20%",
        "Year -10%",
        "Year Down",
        "Year Up",
        "Year +10%",
        "Year +20%",
        "Year +30%",
        "Year +50%",
        "Year +100%",
        "Year +200%",
        "Year +300%",
        "Year +500%",
        "YTD -75%",
        "YTD -50%",
        "YTD -30%",
        "YTD -20%",
        "YTD -10%",
        "YTD -5%",
        "YTD Down",
        "YTD Up",
        "YTD +5%",
        "YTD +10%",
        "YTD +20%",
        "YTD +30",
        "YTD +50%",
        "YTD +100%",
    ],
    "50-Day Simple Moving Average": [
        "Any",
        "Price below SMA50",
        "Price 10% below SMA50",
        "Price 20% below SMA50",
        "Price 30% below SMA50",
        "Price 40% below SMA50",
        "Price 50% below SMA50",
        "Price above SMA50",
        "Price 10% above SMA50",
        "Price 20% above SMA50",
        "Price 30% above SMA50",
        "Price 40% above SMA50",
        "Price 50% above SMA50",
        "Price crossed SMA50",
        "Price crossed SMA50 above",
        "Price crossed SMA50 below",
        "SMA50 crossed SMA20",
        "SMA50 crossed SMA20 above",
        "SMA50 crossed SMA20 below",
        "SMA50 cross SMA200",
        "SMA50 crossed SMA200 below",
        "SMA50 crossed SMA200 above",
        "SMA50 above SMA20",
        "SMA50 below SMA20",
        "SMA50 above SMA200",
        "SMA50 below SMA200",
    ],
    "50-Day High/Low": [
        "Any",
        "New High",
        "New Low",
        "5% or more below High",
        "10% or more below High",
        "15% or more below High",
        "20% or more below High",
        "30% or more below High",
        "40% or more below High",
        "50% or more below High",
        "0-3% below High",
        "0-5% below High",
        "0-10% below High",
        "5% or more above Low",
        "10% or more above Low",
        "15% or more above Low",
        "20% or more above Low",
        "30% or more above Low",
        "40% or more above Low",
        "50% or more above Low",
        "0-3% above Low",
        "0-5% above Low",
        "0-10% above Low",
    ],
    "Average True Range": [
        "Any",
        "Over 0.25",
        "Over 0.5",
        "Over 0.75",
        "Over 1",
        "Over 1.5",
        "Over 2. Over 2.5",
        "Over 3",
        "Over 3.5",
        "Over 4",
        "Over 4.5",
        "Over 5",
        "Under 0.25",
        "Under 0.5",
        "Under 0.75",
        "Under 1",
        "Under 1.5",
        "Under 2",
        "Under 2.5",
        "Under 3",
        "Under 3.5",
        "Under 4",
        "Under 4.5",
        "Under 5",
    ],
    "Volatility": [
        "Any",
        "Week - Over 3%",
        "Week - Over 4%",
        "Week - Over 5%",
        "Week - 6%",
        "Week - 7%",
        "Week - 8%",
        "Week - 9%",
        "Week - 10%",
        "Week - 12%",
        "Week - 15%",
        "Month - 2%",
        "Month - 3%",
        "Month - 4%",
        "Month 5%",
        "Month 5%",
        "Month 6%",
        "Month 7%",
        "Month 8%",
        "Month 9%",
        "Month 10%",
        "Month 12%",
        "Month 15%",
    ],
    "200-Day Simple Moving Average": [
        "Any",
        "Price below SMA200",
        "Price 10% below SMA200",
        "Price 20% below SMA200",
        "Price 30% below SMA200",
        "Price 40% below SMA200",
        "Price 50% below SMA200",
        "Price above SMA200",
        "Price 10% above SMA200",
        "Price 20% above SMA200",
        "Price 30% above SMA200",
        "Price 40% above SMA200",
        "Price 50% above SMA200",
        "Price crossed SMA200",
        "Price crossed SMA200 above",
        "Price crossed SMA200 below",
        "SMA200 crossed SMA20",
        "SMA20 crossed SMA20 above",
        "SMA20 crossed SMA20 below",
        "SMA200 cross SMA50",
        "SMA200 crossed SMA50 below",
        "SMA200 crossed SMA50 above",
        "SMA200 above SMA20",
        "SMA200 below SMA20",
        "SMA200 above SMA50",
        "SMA200 below SMA50",
    ],
    "52-Week High/Low": [
        "Any",
        "New High",
        "New Low",
        "5% or more below High",
        "10% or more below High",
        "15% or more below High",
        "20% or more below High",
        "30% or more below High",
        "40% or more below High",
        "50% or more below High",
        "0-3% below High",
        "0-5% below High",
        "0-10% below High",
        "5% or more above Low",
        "10% or more above Low",
        "15% or more above Low",
        "20% or more above Low",
        "30% or more above Low",
        "40% or more above Low",
        "50% or more above Low",
        "0-3% above Low",
        "0-5% above Low",
        "0-10% above Low",
    ],
    "RSI (14)": [
        "Any",
        "Overbought (90)",
        "Overbought (80)",
        "Overbought (70)",
        "Overbought (6)",
        "Oversold (40)",
        "Oversold (30)",
        "Oversold (20)",
        "Oversold (10)",
        "Not Overbought (<60)",
        "Not Overbought (<50)",
        "Not Oversold (>50)",
        "Not Oversold (>40)",
    ],
    "Change": [
        "Any",
        "Up",
        "Up 1%",
        "Up 2%",
        "Up 3%",
        "Up 4%",
        "Up 5%",
        "Up 6%",
        "Up 7%",
        "Up 8%",
        "Up 9%",
        "Up 10%",
        "Up 15%",
        "Up 20%",
        "Down",
        "Down 1%",
        "Down 2%",
        "Down 3%",
        "Down 4%",
        "Down 5%",
        "Down 6%",
        "Down 7%",
        "Down 8%",
        "Down 9%",
        "Down 10%",
        "Down 15%",
        "Down 20%",
    ],
    "Pattern": [
        "Any",
        "Horizontal S/R",
        "Horizontal S/R (Strong)",
        "TL Resistance",
        "TL Resistance (Strong)",
        "TL Support",
        "TL Support (Strong)",
        "Wedge Up",
        "Wedge Up (Strong)",
        "Wedge Down",
        "Wedge Down (Strong)",
        "Triangle Ascending",
        "Triangle Ascending (Strong)",
        "Triangle Descending",
        "Triangle Descending (Strong)",
        "Wedge",
        "Wedge (Strong)",
        "Channel Up",
        "Channel Up (Strong)",
        "Channel Down",
        "Channel Down (Strong)",
        "Channel",
        "Channel (Strong)",
        "Double Top",
        "Double Bottom",
        "Multiple Top",
        "Multiple Bottom",
        "Head & Shoulders",
        "Head & Shoulders Inverse",
    ],
    "Gap": [
        "Any",
        "Up",
        "Up 1%",
        "Up 2%",
        "Up 3%",
        "Up 4%",
        "Up 5%",
        "Up 6%",
        "Up 7%",
        "Up 8%",
        "Up 9%",
        "Up 10%",
        "Up 15%",
        "Up 20%",
        "Down",
        "Down 1%",
        "Down 2%",
        "Down 3%",
        "Down 4%",
        "Down 5%",
        "Down 6%",
        "Down 7%",
        "Down 8%",
        "Down 9%",
        "Down 10%",
        "Down 15%",
        "Down 20%",
    ],
    "Change from Open": [
        "Any",
        "Up",
        "Up 1%",
        "Up 2%",
        "Up 3%",
        "Up 4%",
        "Up 5%",
        "Up 6%",
        "Up 7%",
        "Up 8%",
        "Up 9%",
        "Up 10%",
        "Up 15%",
        "Up 20%",
        "Down",
        "Down 1%",
        "Down 2%",
        "Down 3%",
        "Down 4%",
        "Down 5%",
        "Down 6%",
        "Down 7%",
        "Down 8%",
        "Down 9%",
        "Down 10%",
        "Down 15%",
        "Down 20%",
    ],
    "Candlestick": [
        "Any",
        "Long Lower Shadow",
        "Long Upper Shadow",
        "Hammer",
        "Inverted Hammer",
        "Spinning Top White",
        "Spinning Top Black",
        "Doji",
        "Dragonfly Doji",
        "Gravestone Doji",
        "Marubozu White",
        "Marubozu Black",
    ],
}
