"""OpenBB Terminal SDK."""
# flake8: noqa
# pylint: disable=unused-import,wrong-import-order
# pylint: disable=C0302,W0611,R0902,R0903,C0412,C0301,not-callable
import logging

import openbb_terminal.config_terminal as cfg
from openbb_terminal import helper_funcs as helper  # noqa: F401
from openbb_terminal.base_helpers import load_dotenv_and_reload_configs
from openbb_terminal.config_terminal import theme

from openbb_terminal.cryptocurrency.due_diligence.pycoingecko_model import Coin
from openbb_terminal.dashboards.dashboards_controller import DashboardsController
from openbb_terminal.helper_classes import TerminalStyle  # noqa: F401
from openbb_terminal.reports import widget_helpers as widgets  # noqa: F401
from openbb_terminal.reports.reports_controller import ReportController

import openbb_terminal.sdk_core.sdk_init as lib
from openbb_terminal.sdk_core import (
    controllers as ctrl,
    models as model,
)
from openbb_terminal import feature_flags as obbff
from openbb_terminal.session.user import User

if User.is_guest():
    load_dotenv_and_reload_configs()

logger = logging.getLogger(__name__)
theme.applyMPLstyle()


class OpenBBSDK:
    """OpenBB SDK Class.

    Attributes:
        `login`: Login and load user info.\n
        `logout`: Logout and clear session.\n
        `news`: Get news for a given term and source. [Source: Feedparser]\n
        `whoami`: Display user info.\n
    """

    __version__ = obbff.VERSION

    def __init__(self):
        SDKLogger()
        self.login = lib.sdk_session.login
        self.logout = lib.sdk_session.logout
        self.news = lib.common_feedparser_model.get_news
        self.whoami = lib.sdk_session.whoami

    @property
    def alt(self):
        """Alternative Submodule

        Submodules:
            `covid`: Covid Module
            `oss`: Oss Module
            `realestate`: Realestate Module

        Attributes:
            `hn`: Get top stories from HackerNews.\n
            `hn_chart`: View top stories from HackerNews.\n
        """

        return ctrl.AltController()

    @property
    def crypto(self):
        """Cryptocurrency Submodule

        Submodules:
            `dd`: Due Diligence Module
            `defi`: DeFi Module
            `disc`: Discovery Module
            `nft`: NFT Module
            `onchain`: OnChain Module
            `ov`: Overview Module
            `tools`: Tools Module

        Attributes:
            `candle`: Plot candle chart from dataframe. [Source: Binance]\n
            `chart`: Load data for Technical Analysis\n
            `find`: Find similar coin by coin name,symbol or id.\n
            `load`: Load crypto currency to get data for\n
            `price`: Returns price and confidence interval from pyth live feed. [Source: Pyth]\n
        """

        return ctrl.CryptoController()

    @property
    def econometrics(self):
        """Econometrics Submodule

        Attributes:
            `bgod`: Calculate test statistics for autocorrelation\n
            `bgod_chart`: Show Breusch-Godfrey autocorrelation test\n
            `bols`: The between estimator is an alternative, usually less efficient estimator, can can be used to\n
            `bpag`: Calculate test statistics for heteroscedasticity\n
            `bpag_chart`: Show Breusch-Pagan heteroscedasticity test\n
            `clean`: Clean up NaNs from the dataset\n
            `coint`: Calculate cointegration tests between variable number of input series\n
            `coint_chart`: Estimates long-run and short-run cointegration relationship for series y and x and apply\n
            `comparison`: Compare regression results between Panel Data regressions.\n
            `dwat`: Calculate test statistics for Durbin Watson autocorrelation\n
            `dwat_chart`: Show Durbin-Watson autocorrelation tests\n
            `fdols`: First differencing is an alternative to using fixed effects when there might be correlation.\n
            `fe`: When effects are correlated with the regressors the RE and BE estimators are not consistent.\n
            `get_regression_data`: This function creates a DataFrame with the required regression data as\n
            `granger`: Calculate granger tests\n
            `granger_chart`: Show granger tests\n
            `load`: Load custom file into dataframe.\n
            `norm`: The distribution of returns and generate statistics on the relation to the normal curve.\n
            `norm_chart`: Determine the normality of a timeseries.\n
            `ols`: Performs an OLS regression on timeseries data. [Source: Statsmodels]\n
            `options`: Obtain columns-dataset combinations from loaded in datasets that can be used in other commands\n
            `options_chart`: Plot custom data\n
            `panel`: Based on the regression type, this function decides what regression to run.\n
            `panel_chart`: Based on the regression type, this function decides what regression to run.\n
            `pols`: PooledOLS is just plain OLS that understands that various panel data structures.\n
            `re`: The random effects model is virtually identical to the pooled OLS model except that is accounts for the\n
            `root`: Calculate test statistics for unit roots\n
            `root_chart`: Determine the normality of a timeseries.\n
        """

        return model.EconometricsRoot()

    @property
    def economy(self):
        """Economy Submodule

        Attributes:
            `available_indices`: Get available indices\n
            `bigmac`: Display Big Mac Index for given countries\n
            `bigmac_chart`: Display Big Mac Index for given countries\n
            `country_codes`: Get available country codes for Bigmac index\n
            `currencies`: Scrape data for global currencies\n
            `events`: Get economic calendar for countries between specified dates\n
            `fred`: Get Series data. [Source: FRED]\n
            `fred_chart`: Display (multiple) series from https://fred.stlouisfed.org. [Source: FRED]\n
            `fred_ids`: Get Series IDs. [Source: FRED]\n
            `fred_notes`: Get series notes. [Source: FRED]\n
            `future`: Get futures data. [Source: Finviz]\n
            `futures`: Get futures data.\n
            `get_groups`: Get group available\n
            `glbonds`: Scrape data for global bonds\n
            `index`: Get data on selected indices over time [Source: Yahoo Finance]\n
            `index_chart`: Load (and show) the selected indices over time [Source: Yahoo Finance]\n
            `indices`: Get the top US indices\n
            `macro`: This functions groups the data queried from the EconDB database [Source: EconDB]\n
            `macro_chart`: Show the received macro data about a company [Source: EconDB]\n
            `macro_countries`: This function returns the available countries and respective currencies.\n
            `macro_parameters`: This function returns the available macro parameters with detail.\n
            `overview`: Scrape data for market overview\n
            `perfmap`: Opens Finviz map website in a browser. [Source: Finviz]\n
            `performance`: Get group (sectors, industry or country) performance data. [Source: Finviz]\n
            `rtps`: Get real-time performance sector data\n
            `rtps_chart`: Display Real-Time Performance sector. [Source: AlphaVantage]\n
            `search_index`: Search indices by keyword. [Source: FinanceDatabase]\n
            `spectrum`: Display finviz spectrum in system viewer [Source: Finviz]\n
            `treasury`: Get U.S. Treasury rates [Source: EconDB]\n
            `treasury_chart`: Display U.S. Treasury rates [Source: EconDB]\n
            `treasury_maturities`: Get treasury maturity options [Source: EconDB]\n
            `usbonds`: Scrape data for us bonds\n
            `valuation`: Get group (sectors, industry or country) valuation data. [Source: Finviz]\n
            `ycrv`: Gets yield curve data from FRED\n
            `ycrv_chart`: Display yield curve based on US Treasury rates for a specified date.\n
        """

        return model.EconomyRoot()

    @property
    def etf(self):
        """Etf Submodule

        Submodules:
            `disc`: Discovery Module
            `scr`: Scr Module

        Attributes:
            `candle`: Show candle plot of loaded ticker.\n
            `compare`: Compare selected ETFs\n
            `etf_by_category`: Return a selection of ETFs based on category filtered by total assets.\n
            `etf_by_name`: Get an ETF symbol and name based on ETF string to search. [Source: StockAnalysis]\n
            `holdings`: Get ETF holdings\n
            `ld`: Return a selection of ETFs based on description filtered by total assets.\n
            `ln`: Return a selection of ETFs based on name filtered by total assets. [Source: Finance Database]\n
            `load`: Load a symbol to perform analysis using the string above as a template.\n
            `news`: Get news for a given term. [Source: NewsAPI]\n
            `news_chart`: Prints table showing news for a given term. [Source: NewsAPI]\n
            `overview`: Get overview data for selected etf\n
            `symbols`: Gets all etf names and symbols\n
            `weights`: Return sector weightings allocation of ETF. [Source: FinancialModelingPrep]\n
        """

        return ctrl.EtfController()

    @property
    def forecast(self):
        """Forecasting Submodule

        Attributes:
            `anom`: Get Quantile Anomaly Detection Data\n
            `anom_chart`: Display Quantile Anomaly Detection\n
            `atr`: Calculate the Average True Range of a variable based on a a specific stock ticker.\n
            `autoarima`: Performs Automatic ARIMA forecasting\n
            `autoarima_chart`: Display Automatic ARIMA model.\n
            `autoces`: Performs Automatic Complex Exponential Smoothing forecasting\n
            `autoces_chart`: Display Automatic Complex Exponential Smoothing Model\n
            `autoets`: Performs Automatic ETS forecasting\n
            `autoets_chart`: Display Automatic ETS (Error, Trend, Sesonality) Model\n
            `autoselect`: Performs Automatic Statistical forecasting\n
            `autoselect_chart`: Display Automatic Statistical Forecasting Model\n
            `brnn`: Performs Block RNN forecasting\n
            `brnn_chart`: Display BRNN forecast\n
            `clean`: Clean up NaNs from the dataset\n
            `combine`: Adds the given column of df2 to df1\n
            `corr`: Returns correlation for a given df\n
            `corr_chart`: Plot correlation coefficients for dataset features\n
            `delete`: Delete a column from a dataframe\n
            `delta`: Calculate the %change of a variable based on a specific column\n
            `desc`: Returns statistics for a given df\n
            `desc_chart`: Show descriptive statistics for a dataframe\n
            `ema`: A moving average provides an indication of the trend of the price movement\n
            `expo`: Performs Probabilistic Exponential Smoothing forecasting\n
            `expo_chart`: Display Probabilistic Exponential Smoothing forecast\n
            `export`: Export a dataframe to a file\n
            `linregr`: Perform Linear Regression Forecasting\n
            `linregr_chart`: Display Linear Regression Forecasting\n
            `load`: Load custom file into dataframe.\n
            `mom`: A momentum oscillator, which measures the percentage change between the current\n
            `mstl`: Performs MSTL forecasting\n
            `mstl_chart`: Display MSTL Model\n
            `nbeats`: Perform NBEATS Forecasting\n
            `nbeats_chart`: Display NBEATS forecast\n
            `nhits`: Performs Nhits forecasting\n
            `nhits_chart`: Display Nhits forecast\n
            `plot`: Plot data from a dataset\n
            `plot_chart`: Plot data from a dataset\n
            `regr`: Perform Regression Forecasting\n
            `regr_chart`: Display Regression Forecasting\n
            `rename`: Rename a column in a dataframe\n
            `rnn`: Perform RNN forecasting\n
            `rnn_chart`: Display RNN forecast\n
            `roc`: A momentum oscillator, which measures the percentage change between the current\n
            `rsi`: A momentum indicator that measures the magnitude of recent price changes to evaluate\n
            `rwd`: Performs Random Walk with Drift forecasting\n
            `rwd_chart`: Display Random Walk with Drift Model\n
            `season_chart`: Plot seasonality from a dataset\n
            `seasonalnaive`: Performs Seasonal Naive forecasting\n
            `seasonalnaive_chart`: Display SeasonalNaive Model\n
            `show`: Show a dataframe in a table\n
            `signal`: A price signal based on short/long term price.\n
            `sto`: Stochastic Oscillator %K and %D : A stochastic oscillator is a momentum indicator comparing a particular closing\n
            `tcn`: Perform TCN forecasting\n
            `tcn_chart`: Display TCN forecast\n
            `tft`: Performs Temporal Fusion Transformer forecasting\n
            `tft_chart`: Display Temporal Fusion Transformer forecast\n
            `theta`: Performs Theta forecasting\n
            `theta_chart`: Display Theta forecast\n
            `trans`: Performs Transformer forecasting\n
            `trans_chart`: Display Transformer forecast\n
        """

        return model.ForecastRoot()

    @property
    def forex(self):
        """Forex Submodule

        Submodules:
            `oanda`: Oanda Module

        Attributes:
            `candle`: Show candle plot for fx data.\n
            `fwd`: Gets forward rates from fxempire\n
            `get_currency_list`: Load AV currency codes from a local file.\n
            `load`: Load forex for two given symbols.\n
            `quote`: Get forex quote.\n
        """

        return ctrl.ForexController()

    @property
    def futures(self):
        """Futures Submodule

        Attributes:
            `curve`: Get curve futures [Source: Yahoo Finance]\n
            `curve_chart`: Display curve futures [Source: Yahoo Finance]\n
            `historical`: Get historical futures [Source: Yahoo Finance]\n
            `historical_chart`: Display historical futures [Source: Yahoo Finance]\n
            `search`: Get search futures [Source: Yahoo Finance]\n
        """

        return model.FuturesRoot()

    @property
    def keys(self):
        """Keys Submodule

        Attributes:
            `av`: Set Alpha Vantage key\n
            `binance`: Set Binance key\n
            `bitquery`: Set Bitquery key\n
            `cmc`: Set Coinmarketcap key\n
            `coinbase`: Set Coinbase key\n
            `coinglass`: Set Coinglass key.\n
            `cpanic`: Set Cpanic key.\n
            `degiro`: Set Degiro key\n
            `eodhd`: Set Eodhd key.\n
            `ethplorer`: Set Ethplorer key.\n
            `finnhub`: Set Finnhub key\n
            `fmp`: Set Financial Modeling Prep key\n
            `fred`: Set FRED key\n
            `get_keys_info`: Get info on available APIs to use in set_keys.\n
            `github`: Set GitHub key.\n
            `glassnode`: Set Glassnode key.\n
            `messari`: Set Messari key.\n
            `mykeys`: Get currently set API keys.\n
            `news`: Set News key\n
            `oanda`: Set Oanda key\n
            `openbb`: Set OpenBB Personal Access Token.\n
            `polygon`: Set Polygon key\n
            `quandl`: Set Quandl key\n
            `reddit`: Set Reddit key\n
            `rh`: Set Robinhood key\n
            `santiment`: Set Santiment key.\n
            `set_keys`: Set API keys in bundle.\n
            `shroom`: Set Shroom key\n
            `smartstake`: Set Smartstake key.\n
            `stocksera`: Set Stocksera key.\n
            `tokenterminal`: Set Token Terminal key.\n
            `tradier`: Set Tradier key\n
            `twitter`: Set Twitter key\n
            `walert`: Set Walert key\n
        """

        return model.KeysRoot()

    @property
    def portfolio(self):
        """Portfolio Submodule

        Submodules:
            `alloc`: Alloc Module
            `metric`: Metric Module
            `po`: Portfolio Optimization Module

        Attributes:
            `bench`: Load benchmark into portfolio\n
            `distr`: Display daily returns\n
            `distr_chart`: Display daily returns\n
            `dret`: Get daily returns\n
            `dret_chart`: Display daily returns\n
            `es`: Get portfolio expected shortfall\n
            `holdp`: Get holdings of assets (in percentage)\n
            `holdp_chart`: Display holdings of assets (in percentage)\n
            `holdv`: Get holdings of assets (absolute value)\n
            `holdv_chart`: Display holdings of assets (absolute value)\n
            `load`: Get PortfolioEngine object\n
            `maxdd`: Calculate the drawdown (MDD) of historical series.  Note that the calculation is done\n
            `maxdd_chart`: Display maximum drawdown curve\n
            `mret`: Get monthly returns\n
            `mret_chart`: Display monthly returns\n
            `om`: Get omega ratio\n
            `om_chart`: Display omega ratio\n
            `perf`: Get portfolio performance vs the benchmark\n
            `rbeta`: Get rolling beta using portfolio and benchmark returns\n
            `rbeta_chart`: Display rolling beta\n
            `rsharpe`: Get rolling sharpe ratio\n
            `rsharpe_chart`: Display rolling sharpe\n
            `rsort`: Get rolling sortino\n
            `rsort_chart`: Display rolling sortino\n
            `rvol`: Get rolling volatility\n
            `rvol_chart`: Display rolling volatility\n
            `show`: Get portfolio transactions\n
            `summary`: Get portfolio and benchmark returns summary\n
            `var`: Get portfolio VaR\n
            `yret`: Get yearly returns\n
            `yret_chart`: Display yearly returns\n
        """

        return ctrl.PortfolioController()

    @property
    def qa(self):
        """Quantitative Analysis Submodule

        Attributes:
            `acf`: Plots Auto and Partial Auto Correlation of returns and change in returns\n
            `bw`: Plots box and whisker plots\n
            `calculate_adjusted_var`: Calculates VaR, which is adjusted for skew and kurtosis (Cornish-Fischer-Expansion)\n
            `cdf`: Plots Cumulative Distribution Function\n
            `cusum`: Plots Cumulative sum algorithm (CUSUM) to detect abrupt changes in data\n
            `decompose`: Perform seasonal decomposition\n
            `es`: Gets Expected Shortfall for specified stock dataframe.\n
            `es_chart`: Prints table showing expected shortfall.\n
            `hist`: Plots histogram of data\n
            `kurtosis`: Kurtosis Indicator\n
            `kurtosis_chart`: Plots rolling kurtosis\n
            `line`: Display line plot of data\n
            `normality`: Look at the distribution of returns and generate statistics on the relation to the normal curve.\n
            `normality_chart`: Prints table showing normality statistics\n
            `omega`: Get the omega series\n
            `omega_chart`: Plots the omega ratio\n
            `qqplot`: Plots QQ plot for data against normal quantiles\n
            `quantile`: Overlay Median & Quantile\n
            `quantile_chart`: Plots rolling quantile\n
            `rolling`: Return rolling mean and standard deviation\n
            `rolling_chart`: Plots mean std deviation\n
            `sharpe`: Calculates the sharpe ratio\n
            `sharpe_chart`: Plots Calculated the sharpe ratio\n
            `skew`: Skewness Indicator\n
            `skew_chart`: Plots rolling skew\n
            `sortino`: Calculates the sortino ratio\n
            `sortino_chart`: Plots the sortino ratio\n
            `spread`: Standard Deviation and Variance\n
            `spread_chart`: Plots rolling spread\n
            `summary`: Print summary statistics\n
            `summary_chart`: Prints table showing summary statistics\n
            `unitroot`: Calculate test statistics for unit roots\n
            `unitroot_chart`: Prints table showing unit root test calculations\n
            `var`: Gets value at risk for specified stock dataframe.\n
            `var_chart`: Prints table showing VaR of dataframe.\n
        """

        return model.QaRoot()

    @property
    def stocks(self):
        """Stocks Submodule

        Submodules:
            `ba`: Behavioral Analysis Module
            `ca`: Comparison Analysis Module
            `disc`: Discovery Module
            `dps`: Darkpool Shorts Module
            `fa`: Fundamental Analysis Module
            `gov`: Government Module
            `ins`: Insiders Module
            `options`: Options Module
            `qa`: Quantitative Analysis Module
            `screener`: Screener Module
            `sia`: Sector Industry Analysis Module
            `ta`: Technical Analysis Module
            `th`: Trading Hours Module

        Attributes:
            `candle`: Show candle plot of loaded ticker.\n
            `load`: Load a symbol to perform analysis using the string above as a template.\n
            `process_candle`: Process DataFrame into candle style plot.\n
            `quote`: Gets ticker quote from FMP\n
            `search`: Search selected query for tickers.\n
            `tob`: Get top of book bid and ask for ticker on exchange [CBOE.com]\n
        """

        return ctrl.StocksController()

    @property
    def ta(self):
        """Technical Analysis Submodule

        Attributes:
            `ad`: Calculate AD technical indicator\n
            `ad_chart`: Plots AD technical indicator\n
            `adosc`: Calculate AD oscillator technical indicator\n
            `adosc_chart`: Plots AD Osc Indicator\n
            `adx`: ADX technical indicator\n
            `adx_chart`: Plots ADX indicator\n
            `aroon`: Aroon technical indicator\n
            `aroon_chart`: Plots Aroon indicator\n
            `atr`: Average True Range\n
            `atr_chart`: Plots ATR\n
            `bbands`: Calculate Bollinger Bands\n
            `bbands_chart`: Plots bollinger bands\n
            `cci`: Commodity channel index\n
            `cci_chart`: Plots CCI Indicator\n
            `cg`: Center of gravity\n
            `cg_chart`: Plots center of gravity Indicator\n
            `clenow`: Gets the Clenow Volatility Adjusted Momentum.  this is defined as the regression coefficient on log prices\n
            `clenow_chart`: Prints table and plots clenow momentum\n
            `demark`: Get the integer value for demark sequential indicator\n
            `demark_chart`: Plot demark sequential indicator\n
            `donchian`: Calculate Donchian Channels\n
            `donchian_chart`: Plots donchian channels\n
            `ema`: Gets exponential moving average (EMA) for stock\n
            `fib`: Calculate Fibonacci levels\n
            `fib_chart`: Plots Calculated fibonacci retracement levels\n
            `fisher`: Fisher Transform\n
            `fisher_chart`: Plots Fisher Indicator\n
            `hma`: Gets hull moving average (HMA) for stock\n
            `kc`: Keltner Channels\n
            `kc_chart`: Plots Keltner Channels Indicator\n
            `ma`: Plots MA technical indicator\n
            `ma_chart`: Plots MA technical indicator\n
            `macd`: Moving average convergence divergence\n
            `macd_chart`: Plots MACD signal\n
            `obv`: On Balance Volume\n
            `obv_chart`: Plots OBV technical indicator\n
            `rsi`: Relative strength index\n
            `rsi_chart`: Plots RSI Indicator\n
            `sma`: Gets simple moving average (SMA) for stock\n
            `stoch`: Stochastic oscillator\n
            `stoch_chart`: Plots stochastic oscillator signal\n
            `vwap`: Gets volume weighted average price (VWAP)\n
            `vwap_chart`: Plots VWMA technical indicator\n
            `wma`: Gets weighted moving average (WMA) for stock\n
            `zlma`: Gets zero-lagged exponential moving average (ZLEMA) for stock\n
        """

        return model.TaRoot()


class SDKLogger:
    def __init__(self) -> None:
        self.__check_initialize_logging()

    def __check_initialize_logging(self):
        if not cfg.LOGGING_SUPPRESS:
            self.__initialize_logging()

    @staticmethod
    def __initialize_logging() -> None:
        # pylint: disable=C0415
        from openbb_terminal.core.log.generation.settings_logger import log_all_settings
        from openbb_terminal.loggers import setup_logging

        cfg.LOGGING_SUB_APP = "sdk"
        setup_logging()
        log_all_settings()


openbb = OpenBBSDK()
