import logging
import argparse

# Configure logging
logging.basicConfig(level=logging.INFO, format="%(message)s")
log = logging.getLogger(__name__)


def calculate_parameters(
    forward_overlap: float,
    side_overlap: float,
    agl: float,
    gsd: float = None,
    image_interval: int = 2,
):
    """
    Parameters
    ---------------------------------
    AGL(Altitude above ground level in meter ) = 115
    Forward overlap = 75
    Side overlap = 75

    ## Fixed Parameters
    Image interval = 2 sec
    Vertical FOV = 0.71
    Horizontal FOV = 1.26

    Forward Photo height = AGL * Vertical_FOV = 115*0.71 = 81.65
    Side Photo width = AGL * Horizontal_FOV = 115*1.26 = 144
    forward overlap distance =  forward photo height * forward overlap = 75 / 100 * 81.65 = 61.5
    side overlap distance = side photo width * side overlap = 75 / 100 * 144 = 108
    forward spacing =  forward photo height - forward overlap distance = 81.65 - 61.5 = 20.15
    side spacing = side photo width - side overlap distance = 144 - 108 = 36
    ground speed = forward spacing / image interval = 10

    """

    # Constants ( For DJI Mini 4 Pro)
    VERTICAL_FOV = 0.71
    HORIZONTAL_FOV = 1.26
    GSD_to_AGL_CONST = 29.7

    if gsd:
        agl = gsd * GSD_to_AGL_CONST

    # Calculations
    forward_photo_height = agl * VERTICAL_FOV
    side_photo_width = agl * HORIZONTAL_FOV
    forward_overlap_distance = forward_photo_height * forward_overlap / 100
    side_overlap_distance = side_photo_width * side_overlap / 100
    forward_spacing = forward_photo_height - forward_overlap_distance
    side_spacing = side_photo_width - side_overlap_distance
    ground_speed = forward_spacing / image_interval

    return {
        "forward_photo_height": round(forward_photo_height, 0),
        "side_photo_width": round(side_photo_width, 0),
        "forward_spacing": round(forward_spacing, 2),
        "side_spacing": round(side_spacing, 2),
        "ground_speed": round(ground_speed, 2),
        "altitude_above_ground_level": agl,
    }


def main():
    parser = argparse.ArgumentParser(
        description="Generate parameters for a drone which can be used for flight plans."
    )

    group = parser.add_mutually_exclusive_group(required=True)

    group.add_argument(
        "--altitude_above_ground_level",
        type=float,
        help="The flight altitude in meters.",
    )
    group.add_argument(
        "--gsd",
        type=float,
        help="The flight altitude in meters.",
    )

    parser.add_argument(
        "--forward_overlap",
        type=float,
        default=70.0,
        help="The forward overlap in percentage.",
    )
    parser.add_argument(
        "--side_overlap",
        type=float,
        default=70.0,
        help="The side overlap in percentage.",
    )
    parser.add_argument(
        "--image_interval",
        type=int,
        default=2,
        help="The time interval between two consecutive images in seconds.",
    )

    args = parser.parse_args()

    results = calculate_parameters(
        args.forward_overlap,
        args.side_overlap,
        args.altitude_above_ground_level,
        args.gsd,
        args.image_interval,
    )

    for key, value in results.items():
        log.info(f"{key}: {value}")

    return results


if __name__ == "__main__":
    main()
