"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmtpCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const smtp_credentials_provider_1 = require("./smtp-credentials-provider");
/**
 * This construct creates an IAM user, with a policy permitting emails to be sent via SES from a specified email address, creates an access key associated with this user, and converts the access key to SMTP credentials.
 *
 * The generated SMTP credentials are stored as a parameter in Parameter Store, and the name of
 * this parameter is output as a CloudFormation output. The parameter may be safely deleted, once
 * the credentials have been accessed.
 *
 * @stability stable
 * @example
 *
 * new SmtpCredentials(this, 'SmtpCredentials', {
 *     emailAddress: 'me@charles.fish',
 * });
 */
class SmtpCredentials extends cdk.Construct {
    /**
     * @param scope A reference to the stack which this construct will be created in.
     * @param id A unique identifier, within the context that this construct is created.
     * @param props Configuration defining how this construct should be created.
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { emailAddress } = props;
        const domainName = this.extractDomainName(emailAddress);
        const { userArn, userName } = this.createIamUser(domainName, emailAddress);
        const { accessKey, smtpPassword } = this.createSmtpCredentials(userArn, userName);
        new cdk.CfnOutput(this, "SmtpCredentialsParameterName", {
            value: new ssm.StringParameter(this, "SmtpCredentials", {
                stringValue: JSON.stringify({
                    AccessKey: accessKey,
                    SmtpPassword: smtpPassword,
                }),
            }).parameterName,
        });
    }
    extractDomainName(emailAddress) {
        if (emailAddress.indexOf("@") === -1) {
            throw Error("Invalid email address supplied.");
        }
        return emailAddress.substring(emailAddress.lastIndexOf("@") + 1);
    }
    createIamUser(domainName, emailAddress) {
        const user = new iam.User(this, "SmtpUser");
        user.attachInlinePolicy(new iam.Policy(this, "SmtpPolicy", {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ["ses:SendRawEmail"],
                    resources: [
                        `arn:aws:ses:${process.env.CDK_DEFAULT_REGION}:${process.env.CDK_DEFAULT_ACCOUNT}:identity/${domainName}`,
                    ],
                    conditions: {
                        StringEquals: {
                            "ses:FromAddress": emailAddress,
                        },
                    },
                }),
            ],
        }));
        return user;
    }
    createSmtpCredentials(userArn, userName) {
        const { serviceToken } = new smtp_credentials_provider_1.SmtpCredentialsProvider(this, "SmtpCredentialsProvider", { userArn });
        const credentials = new cdk.CustomResource(this, "SmtpCredentialsLambda", {
            serviceToken,
            properties: {
                UserName: userName,
            },
        });
        return {
            accessKey: credentials.getAttString("AccessKey"),
            smtpPassword: credentials.getAttString("SmtpPassword"),
        };
    }
}
exports.SmtpCredentials = SmtpCredentials;
_a = JSII_RTTI_SYMBOL_1;
SmtpCredentials[_a] = { fqn: "@charlesdotfish/smtp-credentials-cdk-construct.SmtpCredentials", version: "1.0.18" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic210cC1jcmVkZW50aWFscy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9zbXRwLWNyZWRlbnRpYWxzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFFckMsMkVBQXNFOzs7Ozs7Ozs7Ozs7Ozs7QUFTdEUsTUFBYSxlQUFnQixTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7Ozs7O0lBRWhELFlBQ0UsS0FBb0IsRUFDcEIsRUFBVSxFQUNWLEtBQTJCO1FBRTNCLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxFQUFFLFlBQVksRUFBRSxHQUFHLEtBQUssQ0FBQztRQUUvQixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDeEQsTUFBTSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFVBQVUsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUMzRSxNQUFNLEVBQUUsU0FBUyxFQUFFLFlBQVksRUFBRSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FDNUQsT0FBTyxFQUNQLFFBQVEsQ0FDVCxDQUFDO1FBRUYsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtZQUN0RCxLQUFLLEVBQUUsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtnQkFDdEQsV0FBVyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7b0JBQzFCLFNBQVMsRUFBRSxTQUFTO29CQUNwQixZQUFZLEVBQUUsWUFBWTtpQkFDM0IsQ0FBQzthQUNILENBQUMsQ0FBQyxhQUFhO1NBQ2pCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxZQUFvQjtRQUM1QyxJQUFJLFlBQVksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7WUFDcEMsTUFBTSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQztTQUNoRDtRQUVELE9BQU8sWUFBWSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7SUFFTyxhQUFhLENBQUMsVUFBa0IsRUFBRSxZQUFvQjtRQUM1RCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxrQkFBa0IsQ0FDckIsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDakMsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztvQkFDeEIsT0FBTyxFQUFFLENBQUMsa0JBQWtCLENBQUM7b0JBQzdCLFNBQVMsRUFBRTt3QkFDVCxlQUFlLE9BQU8sQ0FBQyxHQUFHLENBQUMsa0JBQWtCLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsYUFBYSxVQUFVLEVBQUU7cUJBQzFHO29CQUNELFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osaUJBQWlCLEVBQUUsWUFBWTt5QkFDaEM7cUJBQ0Y7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFTyxxQkFBcUIsQ0FBQyxPQUFlLEVBQUUsUUFBZ0I7UUFDN0QsTUFBTSxFQUFFLFlBQVksRUFBRSxHQUFHLElBQUksbURBQXVCLENBQ2xELElBQUksRUFDSix5QkFBeUIsRUFDekIsRUFBRSxPQUFPLEVBQUUsQ0FDWixDQUFDO1FBQ0YsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUN4RSxZQUFZO1lBQ1osVUFBVSxFQUFFO2dCQUNWLFFBQVEsRUFBRSxRQUFRO2FBQ25CO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztZQUNMLFNBQVMsRUFBRSxXQUFXLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQztZQUNoRCxZQUFZLEVBQUUsV0FBVyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUM7U0FDdkQsQ0FBQztJQUNKLENBQUM7O0FBN0VILDBDQThFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tIFwiQGF3cy1jZGsvYXdzLWlhbVwiO1xuaW1wb3J0ICogYXMgc3NtIGZyb20gXCJAYXdzLWNkay9hd3Mtc3NtXCI7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcIkBhd3MtY2RrL2NvcmVcIjtcblxuaW1wb3J0IHsgU210cENyZWRlbnRpYWxzUHJvdmlkZXIgfSBmcm9tIFwiLi9zbXRwLWNyZWRlbnRpYWxzLXByb3ZpZGVyXCI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTbXRwQ3JlZGVudGlhbHNQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVtYWlsQWRkcmVzczogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFNtdHBDcmVkZW50aWFscyBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgY29uc3RydWN0b3IoXG4gICAgc2NvcGU6IGNkay5Db25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBwcm9wczogU210cENyZWRlbnRpYWxzUHJvcHNcbiAgKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHsgZW1haWxBZGRyZXNzIH0gPSBwcm9wcztcblxuICAgIGNvbnN0IGRvbWFpbk5hbWUgPSB0aGlzLmV4dHJhY3REb21haW5OYW1lKGVtYWlsQWRkcmVzcyk7XG4gICAgY29uc3QgeyB1c2VyQXJuLCB1c2VyTmFtZSB9ID0gdGhpcy5jcmVhdGVJYW1Vc2VyKGRvbWFpbk5hbWUsIGVtYWlsQWRkcmVzcyk7XG4gICAgY29uc3QgeyBhY2Nlc3NLZXksIHNtdHBQYXNzd29yZCB9ID0gdGhpcy5jcmVhdGVTbXRwQ3JlZGVudGlhbHMoXG4gICAgICB1c2VyQXJuLFxuICAgICAgdXNlck5hbWVcbiAgICApO1xuXG4gICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgXCJTbXRwQ3JlZGVudGlhbHNQYXJhbWV0ZXJOYW1lXCIsIHtcbiAgICAgIHZhbHVlOiBuZXcgc3NtLlN0cmluZ1BhcmFtZXRlcih0aGlzLCBcIlNtdHBDcmVkZW50aWFsc1wiLCB7XG4gICAgICAgIHN0cmluZ1ZhbHVlOiBKU09OLnN0cmluZ2lmeSh7XG4gICAgICAgICAgQWNjZXNzS2V5OiBhY2Nlc3NLZXksXG4gICAgICAgICAgU210cFBhc3N3b3JkOiBzbXRwUGFzc3dvcmQsXG4gICAgICAgIH0pLFxuICAgICAgfSkucGFyYW1ldGVyTmFtZSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgZXh0cmFjdERvbWFpbk5hbWUoZW1haWxBZGRyZXNzOiBzdHJpbmcpIHtcbiAgICBpZiAoZW1haWxBZGRyZXNzLmluZGV4T2YoXCJAXCIpID09PSAtMSkge1xuICAgICAgdGhyb3cgRXJyb3IoXCJJbnZhbGlkIGVtYWlsIGFkZHJlc3Mgc3VwcGxpZWQuXCIpO1xuICAgIH1cblxuICAgIHJldHVybiBlbWFpbEFkZHJlc3Muc3Vic3RyaW5nKGVtYWlsQWRkcmVzcy5sYXN0SW5kZXhPZihcIkBcIikgKyAxKTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlSWFtVXNlcihkb21haW5OYW1lOiBzdHJpbmcsIGVtYWlsQWRkcmVzczogc3RyaW5nKSB7XG4gICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcih0aGlzLCBcIlNtdHBVc2VyXCIpO1xuICAgIHVzZXIuYXR0YWNoSW5saW5lUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgXCJTbXRwUG9saWN5XCIsIHtcbiAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcInNlczpTZW5kUmF3RW1haWxcIl0sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgYGFybjphd3M6c2VzOiR7cHJvY2Vzcy5lbnYuQ0RLX0RFRkFVTFRfUkVHSU9OfToke3Byb2Nlc3MuZW52LkNES19ERUZBVUxUX0FDQ09VTlR9OmlkZW50aXR5LyR7ZG9tYWluTmFtZX1gLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgICAgXCJzZXM6RnJvbUFkZHJlc3NcIjogZW1haWxBZGRyZXNzLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIHJldHVybiB1c2VyO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVTbXRwQ3JlZGVudGlhbHModXNlckFybjogc3RyaW5nLCB1c2VyTmFtZTogc3RyaW5nKSB7XG4gICAgY29uc3QgeyBzZXJ2aWNlVG9rZW4gfSA9IG5ldyBTbXRwQ3JlZGVudGlhbHNQcm92aWRlcihcbiAgICAgIHRoaXMsXG4gICAgICBcIlNtdHBDcmVkZW50aWFsc1Byb3ZpZGVyXCIsXG4gICAgICB7IHVzZXJBcm4gfVxuICAgICk7XG4gICAgY29uc3QgY3JlZGVudGlhbHMgPSBuZXcgY2RrLkN1c3RvbVJlc291cmNlKHRoaXMsIFwiU210cENyZWRlbnRpYWxzTGFtYmRhXCIsIHtcbiAgICAgIHNlcnZpY2VUb2tlbixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgVXNlck5hbWU6IHVzZXJOYW1lLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHJldHVybiB7XG4gICAgICBhY2Nlc3NLZXk6IGNyZWRlbnRpYWxzLmdldEF0dFN0cmluZyhcIkFjY2Vzc0tleVwiKSxcbiAgICAgIHNtdHBQYXNzd29yZDogY3JlZGVudGlhbHMuZ2V0QXR0U3RyaW5nKFwiU210cFBhc3N3b3JkXCIpLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==