"""Coverage reporters."""

import json
import os
import os.path
import sys
import time
import xml.dom.minidom
from collections import defaultdict

from pycobertura import Cobertura
from pycobertura.reporters import TextReporter as CoberturaTextReporter

from nile_coverage import __url__, __version__, logger
from nile_coverage.common import COVERAGE_DIRECTORY, CoverageFile
from nile_coverage.utils import add_files_to_report, get_file_statements


class XmlReporter:
    """Cobertura-style XML reports."""

    def __init__(self, contracts_folder, statements, report_dict):
        """Initialize reporter."""
        self.statements = statements
        self.report_dict = report_dict
        self.contracts_folder = contracts_folder
        self.cairo_path = [contracts_folder]

        # add empty coverage files to report
        add_files_to_report(contracts_folder, self.report_dict)

        self.source_paths = set()
        self.packages = {}
        self.xml_out = None

    def report(self, outfile=None, as_string=False):
        """
        Generate a Cobertura-compatible XML report.

        `outfile` is a file object to write the XML to.
        """
        # Initial setup.
        outfile = outfile or sys.stdout

        # Create the DOM that will store the data.
        impl = xml.dom.minidom.getDOMImplementation()
        self.xml_out = impl.createDocument(None, "coverage", None)

        # Write header stuff.
        xcoverage = self.xml_out.documentElement
        xcoverage.setAttribute("version", __version__)
        xcoverage.setAttribute("timestamp", str(int(time.time() * 1000)))
        xcoverage.appendChild(
            self.xml_out.createComment(f" Generated by nile-coverage: {__url__} ")
        )

        # Call xml_file for each file in the data.
        for file, coverage in self.report_dict.items():
            if file.startswith(self.contracts_folder):
                self.xml_file(
                    CoverageFile(
                        statements=set(self.statements[file]),
                        covered=set(coverage),
                        name=file,
                    )
                )

        xsources = self.xml_out.createElement("sources")
        xcoverage.appendChild(xsources)

        # Populate the XML DOM with the source info.
        for path in sorted(self.source_paths):
            xsource = self.xml_out.createElement("source")
            xsources.appendChild(xsource)
            txt = self.xml_out.createTextNode(path)
            xsource.appendChild(txt)

        lnum_tot, lhits_tot = 0, 0
        bnum_tot, bhits_tot = 0, 0

        xpackages = self.xml_out.createElement("packages")
        xcoverage.appendChild(xpackages)

        # Populate the XML DOM with the package info.
        for pkg_name, pkg_data in sorted(self.packages.items()):
            class_elts, lhits, lnum, bhits, bnum = pkg_data
            xpackage = self.xml_out.createElement("package")
            xpackages.appendChild(xpackage)
            xclasses = self.xml_out.createElement("classes")
            xpackage.appendChild(xclasses)
            for _, class_elt in sorted(class_elts.items()):
                xclasses.appendChild(class_elt)
            xpackage.setAttribute("name", pkg_name.replace(os.sep, "."))
            xpackage.setAttribute("line-rate", "0")
            branch_rate = "0"
            xpackage.setAttribute("branch-rate", branch_rate)
            xpackage.setAttribute("complexity", "0")

            lnum_tot += lnum
            lhits_tot += lhits
            bnum_tot += bnum
            bhits_tot += bhits

        xcoverage.setAttribute("lines-valid", str(lnum_tot))
        xcoverage.setAttribute("lines-covered", str(lhits_tot))
        xcoverage.setAttribute("line-rate", rate(lhits_tot, lnum_tot))
        xcoverage.setAttribute("branches-covered", "0")
        xcoverage.setAttribute("branches-valid", "0")
        xcoverage.setAttribute("branch-rate", "0")
        xcoverage.setAttribute("complexity", "0")

        # Return the string without writing to file
        if as_string:
            return serialize_xml(self.xml_out)

        # Write the output file.
        cwd = os.getcwd()
        output = os.path.join(cwd, outfile)
        with open(output, "w") as f:
            f.write(serialize_xml(self.xml_out))

        # Return the total percentage.
        denom = lnum_tot + bnum_tot
        if denom == 0:
            pct = 0.0
        else:
            pct = 100.0 * (lhits_tot + bhits_tot) / denom
        return pct

    def xml_file(self, cf: CoverageFile):
        """Add to the XML report for a single file."""
        # Create the 'lines' and 'package' XML elements, which
        # are populated later.  Note that a package == a directory.
        cf.name = cf.name.replace("\\", "/")
        self.source_paths.add(cf.name)

        rel_name = cf.name

        if cf.nb_statements == 0:
            statements = get_file_statements([rel_name], self.cairo_path)
            if rel_name in statements:
                cf.statements = list(statements[rel_name])
                cf.nb_statements = len(cf.statements)

        dirname = os.path.dirname(rel_name) or "."
        dirname = "/".join(dirname.split("/")[:6])
        package_name = dirname.replace("/", ".")

        package = self.packages.setdefault(package_name, [{}, 0, 0, 0, 0])

        xclass = self.xml_out.createElement("class")

        xclass.appendChild(self.xml_out.createElement("methods"))

        xlines = self.xml_out.createElement("lines")
        xclass.appendChild(xlines)

        xclass.setAttribute("name", os.path.relpath(rel_name, dirname))
        xclass.setAttribute("filename", rel_name.replace("\\", "/"))
        xclass.setAttribute("complexity", "0")

        # For each statement, create an XML 'line' element.
        for line in sorted(cf.statements):
            xline = self.xml_out.createElement("line")
            xline.setAttribute("number", str(line))

            # Q: can we get info about the number of times a statement is
            # executed?  If so, that should be recorded here.
            if line in cf.covered:
                xline.setAttribute("hits", "1")
            else:
                xline.setAttribute("hits", "0")
            xlines.appendChild(xline)

        class_branches = 0.0
        class_br_hits = 0.0
        branch_rate = "0"
        line_rate = rate(cf.nb_covered, cf.nb_statements)

        # Finalize the statistics that are collected in the XML DOM.
        xclass.setAttribute("line-rate", line_rate)
        xclass.setAttribute("branch-rate", branch_rate)

        package[0][rel_name] = xclass
        package[1] += cf.nb_covered
        package[2] += cf.nb_statements
        package[3] += class_br_hits
        package[4] += class_branches


class TextReporter(XmlReporter):
    """CLI text reports."""

    def report(self):
        """Print the coverage summary of the project."""
        xml_string = XmlReporter.report(self, as_string=True)

        cobertura = Cobertura(xml_string)
        tr = CoberturaTextReporter(cobertura)

        logger.info(f"\n\n{tr.generate()}")


def serialize_xml(dom):
    """Serialize a minidom node to XML."""
    return dom.toprettyxml()


def rate(hit, num):
    """Return the fraction of `hit`/`num`, as a string."""
    if num == 0:
        return "1"
    else:
        return "%.4g" % (float(hit) / num)


def run_report(contracts_folder: str = "", xml: bool = False):
    logger.info("\nGenerating coverage report. This can take a minute...")

    if not os.path.isdir(contracts_folder):
        logger.info(
            f'\n\nNothing to report (couldn\'t find "{contracts_folder}" directory)'
        )

    # Aggregate nile.coverage files
    statements = defaultdict(set)
    report_dict = defaultdict(set)

    files = [
        os.path.join(COVERAGE_DIRECTORY, f)
        for f in os.listdir(COVERAGE_DIRECTORY)
        if os.path.isfile(os.path.join(COVERAGE_DIRECTORY, f))
    ]

    for f in files:
        with open(f, "r") as fp:
            data = json.load(fp)
            for contract in data["lines"]:
                statements[contract].update(data["lines"][contract])
            for contract in data["covered_lines"]:
                report_dict[contract].update(data["covered_lines"][contract])

    if xml:
        reporter = XmlReporter(contracts_folder, statements, report_dict)
        reporter.report(outfile="coverage.xml")
    else:
        reporter = TextReporter(contracts_folder, statements, report_dict)
        reporter.report()
