# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pycmtensor', 'pycmtensor.models']

package_data = \
{'': ['*']}

install_requires = \
['aesara>2.7.4',
 'dill>=0.3.4,<0.4.0',
 'numpy>=1.19.0',
 'pandas>=1.3.5,<2.0.0',
 'pydot>=1.4.2,<2.0.0',
 'scipy>=1.7.1,<2.0.0']

extras_require = \
{'commit': ['pre-commit>=2.17.0,<3.0.0', 'commitizen>=2.21.0,<3.0.0'],
 'dev': ['black==22.6.0',
         'isort==5.10.1',
         'pydocstyle>=6.1.1,<7.0.0',
         'rstcheck>=3.3.1,<4.0.0',
         'Sphinx>=4.4.0,<5.0.0',
         'sphinx-book-theme>=0.2.0,<0.3.0',
         'sphinx-autoapi>=1.8.4,<2.0.0',
         'myst-parser>=0.18.0,<0.19.0',
         'pre-commit>=2.17.0,<3.0.0',
         'commitizen>=2.21.0,<3.0.0'],
 'docs': ['Sphinx>=4.4.0,<5.0.0',
          'sphinx-book-theme>=0.2.0,<0.3.0',
          'sphinx-autoapi>=1.8.4,<2.0.0',
          'myst-parser>=0.18.0,<0.19.0'],
 'lint': ['black==22.6.0',
          'isort==5.10.1',
          'pydocstyle>=6.1.1,<7.0.0',
          'rstcheck>=3.3.1,<4.0.0']}

setup_kwargs = {
    'name': 'pycmtensor',
    'version': '1.1.0',
    'description': 'Python Tensor based package for Deep neural net assisted Discrete Choice Modelling.',
    'long_description': '# PyCMTensor\n\n![](https://img.shields.io/pypi/pyversions/pycmtensor)\n[![PyPI version](https://badge.fury.io/py/pycmtensor.svg)](https://badge.fury.io/py/pycmtensor)\n[![Tests](https://github.com/mwong009/pycmtensor/actions/workflows/tests.yml/badge.svg)](https://github.com/mwong009/pycmtensor/actions/workflows/tests.yml)\n[![Documentation Status](https://readthedocs.org/projects/pycmtensor/badge/?version=develop)](https://pycmtensor.readthedocs.io/en/develop/?badge=develop)\n[![codecov](https://codecov.io/gh/mwong009/pycmtensor/branch/master/graph/badge.svg?token=LFwgggDyjS)](https://codecov.io/gh/mwong009/pycmtensor)\n![Licence](https://img.shields.io/badge/Licence-MIT-blue)\n\nA tensor-based choice modelling Python package with deep learning libraries. \nRead the documentation at <https://pycmtensor.readthedocs.io>\n\n## About PyCMTensor\n\nPyCMTensor is a discrete choice model development platform which is designed with the use of deep learning in mind, enabling users to write more complex models using neural networks.\nPyCMTensor is build on [Aesara](https://github.com/aesara-devs/aesara), a tensor library, and uses many features commonly found in deep learning packages such as Tensorflow and Keras.\n`Aesara` was chosen as the back end mathematical library because of its hackable, open-source nature.\nUsers of [Biogeme](https://biogeme.epfl.ch) will be familiar with the syntax of PyCMTensor.\n\nThis package allows one to incorporate neural networks into discrete choice models that boosts accuracy of model estimates which still being able to produce all the same statistical analysis found in traditional choice modelling software.\n\nPyCMTensor aims to provide developers and researchers with deep learning tools for econometric and travel behaviour modelling with **reproducible** and **interpretable** results.\n\n### PyCMTensor and Biogeme\n\nPyCMTensor improves on Biogeme in situations where much more complex models are necessary, for example, integrating neural networks into discrete choice models.\nPyCMTensor also include the ability to estimate models using 1st order stochastic gradient descent methods by default, such as Nesterov Accelerated Gradient, Adam, or RMSProp.\n## Features\n\n* Estimate complex choice models with neural networks using deep learning algorithms\n* Combines traditional econometric models (e.g. Multinomial Logit) with deep learning models (e.g. ResNets)\n* Shares similar programming syntax with `Biogeme`, allowing easy transition between `Biogeme` and PyCMTensor models\n* Uses tensor based mathematical operations from the advanced features found in the `Aesara` library\n\n---\n\n## Pre-install\n\nTo install PyCMTensor, you need [Conda](https://docs.conda.io/en/latest/miniconda.html) (Full Anaconda works fine, but **miniconda** is recommmended for a minimal installation). \nEnsure that Conda is using at least `Python 3.9`.\n\nOnce Conda is installed, install the required dependencies from conda by running the following command in your terminal:\n\n**Windows**\n\n```console\nconda install mkl-service conda-forge::cxx-compiler conda-forge::m2w64-toolchain -y\n```\n**Linux**\n\n```console\nconda install mkl-service conda-forge::cxx-compiler\n```\n\n**Mac OSX**\n\n```console\nconda install mkl-service Clang\n```\n\n### Install PyCMTensor\n\nThen, run this command in your terminal to download and install the latest branch of PyCMTensor from [PyPI](https://pypi.org/project/pycmtensor/):\n```console\npip install pycmtensor -U\n```\n\n*Optional*: If you want the development version from the Github repository:\n```console\npip install git+https://github.com/mwong009/pycmtensor.git@develop -U\n```\n\nThe development branch is the most recent update of PyCMTensor. If you want a stable branch (master), remove ``@develop`` at the end of the ``.git`` url.\n\n## Usage\n\nPyCMTensor uses syntax very similar to `Biogeme`. Users of `Biogeme` should be familiar \nwith the syntax.\n\nStart an interactive session (`IPython` or Jupyter Notebook) and import:\n```Python\nimport pycmtensor as cmt\n```\n\nSeveral submodules are also important to include:\n```Python\nfrom pycmtensor.expressions import Beta # Beta class for model parameters\nfrom pycmtensor.models import MNLogit   # model library\nfrom pycmtensor.optimizers import Adam  # Optimizers\nfrom pycmtensor.results import Results  # for generating results\n```\n\nFor a full list of submodules and description, refer to [API Reference](/autoapi/index)\n\n## Development\n\nTo set up `PyCMTensor` in a local development environment, you need to set up a virtual environment and install the project requirements. Follow the instructions to install Conda (miniconda), then start a new virtual environment with the provided environment_\\<your OS\\>.yml file.\n\nFor example in windows:\n```console\nconda env create -f environment_windows.yml\n```\n\nNext, activate the virtual environment and install poetry via `pip`.\n```console\nconda activate pycmtensor-dev\npip install poetry\n```\n\nLastly, install the project and development dependencies\n```console\npoetry install -E dev\n```\n\nThe virtual environment needs to be activated and commits are done from the virtural environment.\n\n### Simple example: Swissmetro dataset\n\nUsing the [swissmetro dataset](https://biogeme.epfl.ch/data.html) from Biogeme, we define a simple MNL model. \n\nNote:The following is a replication of the results from Biogeme using the `Adam` optimization algorithm.\n\n\n1. Import the dataset and perform some data santiation\n\t```Python\n\tswissmetro = pd.read_csv("data/swissmetro.dat", sep="\\t")\n\tdb = cmt.Database(name="swissmetro", pandasDatabase=swissmetro, choiceVar="CHOICE")\n\tglobals().update(db.variables)\n\t# Removing some observations\n\tdb.data.drop(db.data[db.data["CHOICE"] == 0].index, inplace=True)\n\tdb.data["CHOICE"] -= 1  # set the first choice index to 0\n\tdb.choices = [0, 1, 2]\n\tdb.autoscale(\n\t\tvariables=["TRAIN_CO", "TRAIN_TT", "CAR_CO", "CAR_TT", "SM_CO", "SM_TT"],\n\t\tdefault=100.0,\n\t\tverbose=False,\n\t)\n\t```\n\n\t``cmt.Database()`` loads the dataset and defines tensor variables automatically.\n\n2. Initialize the model parameters and specify the utility functions and availability conditions\n\t```Python\n\tb_cost = Beta("b_cost", 0.0, None, None, 0)\n\tb_time = Beta("b_time", 0.0, None, None, 0)\n\tasc_train = Beta("asc_train", 0.0, None, None, 0)\n\tasc_car = Beta("asc_car", 0.0, None, None, 0)\n\tasc_sm = Beta("asc_sm", 0.0, None, None, 1)\n\n\tU_1 = b_cost * db["TRAIN_CO"] + b_time * db["TRAIN_TT"] + asc_train\n\tU_2 = b_cost * db["SM_CO"] + b_time * db["SM_TT"] + asc_sm\n\tU_3 = b_cost * db["CAR_CO"] + b_time * db["CAR_TT"] + asc_car\n\t``` \n\n3. Define the Multinomial Logit model\n\t```Python\n\tmymodel = MNLogit(u=U, av=AV, database=db, name="Multinomial Logit")\n\tmymodel.add_params(locals()) # load Betas into the model\n\t```\n\n5. (optional) Define the model hyperparameters\n\t```Python\n\tmymodel.config["patience"] = 9000\n\tmymodel.config["max_epoch"] = 500\n\tmymodel.config["base_lr"] = 0.0012\n\tmymodel.config["max_lr"] = 0.002\n\tmymodel.config["learning_scheduler"] = "ConstantLR"\n\t```\n\n6. Call the training function and save the trained model\n\t```Python\n\tmodel = cmt.train(model=mymodel, database=db, optimizer=Adam)  # we use the Adam Optimizer\n\t```\n\n7. Generate the statistics and correlation matrices\n\t```Python\n\tresults = Results(model, db, prnt=False)\n\tprint(results)\n\tresults.generate_beta_statistics()\n\tresults.print_beta_statistics()\n\tresults.print_correlation_matrix()\n\n\t```\n\n\tSample output: \n\n\t\tPython 3.10.4 | packaged by conda-forge | (main, Mar 30 2022, 08:38:02) [MSC v.1916 64 bit (AMD64)]\n\t\t[2022-08-12 18:51:40] INFO: Building model...\n\t\t[2022-08-12 18:51:52] INFO: Training model...\n\t\t[2022-08-12 18:51:55] INFO: Maximum iterations reached. Terminating...\n\t\t[2022-08-12 18:51:55] INFO: Optimization complete with accuracy of 61.937%.\n\t\t[2022-08-12 18:51:55] INFO: Max log likelihood reached @ epoch 57.\n\n\t\tResults\n\t\t------\n\t\tModel: Multinomial Logit\n\t\tBuild time: 00:00:12\n\t\tEstimation time: 00:00:03\n\t\tEstimation rate: 3400.838 iter/s\n\t\tSeed value: 7577\n\t\tNumber of Beta parameters: 4\n\t\tSample size: 10719\n\t\tExcluded data: None\n\t\tInit loglikelihood: -11093.627\n\t\tFinal loglikelihood: -9165.567\n\t\tFinal loglikelihood reached at: epoch 57\n\t\tLikelihood ratio test: 3856.120\n\t\tAccuracy: 61.937%\n\t\tRho square: 0.174\n\t\tRho bar square: 0.173\n\t\tAkaike Information Criterion: 18339.13\n\t\tBayesian Information Criterion: 18368.25\n\t\tFinal gradient norm: 0.121\n\n\t\tModel statistics:\n\t\t              Value   Std err     t-test   p-value Rob. Std err Rob. t-test Rob. p-value\n\t\tasc_car    0.013287  0.030614   0.434002  0.664287     0.159125    0.083498     0.933456\n\t\tasc_train -0.537674  0.037544 -14.321085       0.0     0.014821  -36.278684          0.0\n\t\tb_cost     0.021882  0.002227   9.824814       0.0     0.005462     4.00618     0.000062\n\t\tb_time    -0.604866  0.035116 -17.224787       0.0     0.514255   -1.176199     0.239515\n\n\t\tCorrelation matrix:\n\t\t             b_cost    b_time  asc_train   asc_car\n\t\tb_cost     1.000000 -0.092697   0.171935  0.269662\n\t\tb_time    -0.092697  1.000000  -0.710780 -0.596636\n\t\tasc_train  0.171935 -0.710780   1.000000  0.603376\n\t\tasc_car    0.269662 -0.596636   0.603376  1.000000\n\n8. Plot the training performance and accuracy\n\n\t![](https://github.com/mwong009/pycmtensor/blob/master/docs/_static/viz/fig.png)\n\n9. Compute the elasticities\n\n\t![](https://github.com/mwong009/pycmtensor/blob/master/docs/_static/viz/els.png)\n\n10. Visualize the computation graph\n\t```Python\n\timport aesara.d3viz as d3v\n\tfrom aesara import printing\n\tprinting.pydotprint(mymodel.cost, "graph.png")\n\t```\n\n\t![](https://github.com/mwong009/pycmtensor/blob/master/docs/_static/viz/print.png)\n\n---\n\n## Credits\n\nPyCMTensor was inspired by [Biogeme](https://biogeme.epfl.ch) and aims to provide deep learning modelling tools for transport modellers and researchers.\n\nThis package template was generated with [Cookiecutter](https://github.com/audreyr/cookiecutter) and the [audreyr/cookiecutter-pypackage](https://github.com/audreyr/cookiecutter-pypackage) project template.\n',
    'author': 'Melvin Wong',
    'author_email': 'm.j.w.wong@tue.nl',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/mwong009/pycmtensor',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '==3.9.12',
}


setup(**setup_kwargs)
