"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const route53 = require("aws-cdk-lib/aws-route53");
const patterns = require("aws-cdk-lib/aws-route53-patterns");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * A CloudFront static website hosted on S3
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName: props.domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const originRequest = new cloudfront.experimental.EdgeFunction(this, 'OriginRequest', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'origin-request-handler')),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_14_X,
        });
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(this.bucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: originRequest,
                    },
                ],
                responseHeadersPolicy: (_b = props.responseHeadersPolicy) !== null && _b !== void 0 ? _b : new cloudfront.ResponseHeadersPolicy(this, 'ResponseHeadersPolicy', {
                    securityHeadersBehavior: StaticWebsite.defaultSecurityHeadersBehavior,
                }),
            },
            defaultRootObject: 'index.html',
            domainNames: [props.domainName],
            certificate,
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2021,
        });
        new route53.ARecord(this, 'ARecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        if (props.backendConfiguration) {
            // Save backend config to bucket, can be queried by the frontend
            new cr.AwsCustomResource(this, 'PutConfig', {
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['s3:PutObject'],
                        resources: [this.bucket.arnForObjects('config.json')],
                    }),
                ]),
                onUpdate: {
                    service: 'S3',
                    action: 'putObject',
                    parameters: {
                        Bucket: this.bucket.bucketName,
                        Key: 'config.json',
                        Body: aws_cdk_lib_1.Stack.of(this).toJsonString(props.backendConfiguration),
                        ContentType: 'application/json',
                        CacheControl: 'max-age=0, no-cache, no-store, must-revalidate',
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('config'),
                },
            });
        }
        if (!props.redirects || props.redirects.length !== 0) {
            const httpsRedirect = new patterns.HttpsRedirect(this, 'HttpsRedirect', {
                targetDomain: props.domainName,
                zone: props.hostedZone,
                recordNames: props.redirects,
            });
            // Force minimum protocol version
            const redirectDistribution = httpsRedirect.node.tryFindChild('RedirectDistribution');
            const cfnDistribution = redirectDistribution.node.tryFindChild('CFDistribution');
            cfnDistribution.addPropertyOverride('DistributionConfig.ViewerCertificate.MinimumProtocolVersion', 'TLSv1.2_2021');
        }
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "cloudstructs.StaticWebsite", version: "0.4.21" };
/**
* Best practice security headers used as default
*/
StaticWebsite.defaultSecurityHeadersBehavior = {
    contentTypeOptions: {
        override: true,
    },
    frameOptions: {
        frameOption: cloudfront.HeadersFrameOption.DENY,
        override: true,
    },
    referrerPolicy: {
        referrerPolicy: cloudfront.HeadersReferrerPolicy.STRICT_ORIGIN_WHEN_CROSS_ORIGIN,
        override: true,
    },
    strictTransportSecurity: {
        accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(31536000),
        includeSubdomains: true,
        preload: true,
        override: true,
    },
    xssProtection: {
        protection: true,
        modeBlock: true,
        override: true,
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvc3RhdGljLXdlYnNpdGUvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFDN0IsNkNBQThDO0FBQzlDLDBEQUEwRDtBQUMxRCx5REFBeUQ7QUFDekQsOERBQThEO0FBQzlELDJDQUEyQztBQUMzQyxpREFBaUQ7QUFDakQsbURBQW1EO0FBQ25ELDZEQUE2RDtBQUM3RCwyREFBMkQ7QUFDM0QseUNBQXlDO0FBQ3pDLG1EQUFtRDtBQUNuRCwyQ0FBdUM7QUEyQ3ZDOztHQUVHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsc0JBQVM7SUF1QzFDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7O1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUMxQyxpQkFBaUIsRUFBRSxFQUFFLENBQUMsaUJBQWlCLENBQUMsU0FBUztTQUNsRCxDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ3ZFLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsTUFBTSxFQUFFLFdBQVc7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxVQUFVLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQ3BGLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO1lBQzNFLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNwRSxlQUFlLEVBQUU7Z0JBQ2YsTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDO2dCQUN6QyxvQkFBb0IsRUFBRSxVQUFVLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCO2dCQUN2RSxXQUFXLEVBQUU7b0JBQ1g7d0JBQ0UsU0FBUyxFQUFFLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxjQUFjO3dCQUN4RCxlQUFlLEVBQUUsYUFBYTtxQkFDL0I7aUJBQ0Y7Z0JBQ0QscUJBQXFCLFFBQUUsS0FBSyxDQUFDLHFCQUFxQixtQ0FBSSxJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7b0JBQ3hILHVCQUF1QixFQUFFLGFBQWEsQ0FBQyw4QkFBOEI7aUJBQ3RFLENBQUM7YUFDSDtZQUNELGlCQUFpQixFQUFFLFlBQVk7WUFDL0IsV0FBVyxFQUFFLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztZQUMvQixXQUFXO1lBQ1gsc0JBQXNCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLGFBQWE7U0FDeEUsQ0FBQyxDQUFDO1FBRUgsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbkMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQzVCLElBQUksRUFBRSxLQUFLLENBQUMsVUFBVTtZQUN0QixNQUFNLEVBQUUsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQ3hGLENBQUMsQ0FBQztRQUVILElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3pDLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixJQUFJLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDdEIsTUFBTSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksT0FBTyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUN4RixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsRUFBRTtZQUNoQyxnRUFBZ0U7WUFDOUQsSUFBSSxFQUFFLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtnQkFDMUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxjQUFjLENBQUM7b0JBQ2hELElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzt3QkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO3dCQUN6QixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxhQUFhLENBQUMsQ0FBQztxQkFDdEQsQ0FBQztpQkFDSCxDQUFDO2dCQUNGLFFBQVEsRUFBRTtvQkFDUixPQUFPLEVBQUUsSUFBSTtvQkFDYixNQUFNLEVBQUUsV0FBVztvQkFDbkIsVUFBVSxFQUFFO3dCQUNWLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVU7d0JBQzlCLEdBQUcsRUFBRSxhQUFhO3dCQUNsQixJQUFJLEVBQUUsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQzt3QkFDN0QsV0FBVyxFQUFFLGtCQUFrQjt3QkFDL0IsWUFBWSxFQUFFLGdEQUFnRDtxQkFDL0Q7b0JBQ0Qsa0JBQWtCLEVBQUUsRUFBRSxDQUFDLGtCQUFrQixDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUM7aUJBQ3ZEO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDcEQsTUFBTSxhQUFhLEdBQUcsSUFBSSxRQUFRLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7Z0JBQ3RFLFlBQVksRUFBRSxLQUFLLENBQUMsVUFBVTtnQkFDOUIsSUFBSSxFQUFFLEtBQUssQ0FBQyxVQUFVO2dCQUN0QixXQUFXLEVBQUUsS0FBSyxDQUFDLFNBQVM7YUFDN0IsQ0FBQyxDQUFDO1lBQ0gsaUNBQWlDO1lBQ2pDLE1BQU0sb0JBQW9CLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQXlDLENBQUM7WUFDN0gsTUFBTSxlQUFlLEdBQUcsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBK0IsQ0FBQztZQUMvRyxlQUFlLENBQUMsbUJBQW1CLENBQUMsNkRBQTZELEVBQUUsY0FBYyxDQUFDLENBQUM7U0FDcEg7SUFDSCxDQUFDOztBQTdISCxzQ0E4SEM7OztBQTdIQzs7RUFFRTtBQUNZLDRDQUE4QixHQUErQztJQUN6RixrQkFBa0IsRUFBRTtRQUNsQixRQUFRLEVBQUUsSUFBSTtLQUNmO0lBQ0QsWUFBWSxFQUFFO1FBQ1osV0FBVyxFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJO1FBQy9DLFFBQVEsRUFBRSxJQUFJO0tBQ2Y7SUFDRCxjQUFjLEVBQUU7UUFDZCxjQUFjLEVBQUUsVUFBVSxDQUFDLHFCQUFxQixDQUFDLCtCQUErQjtRQUNoRixRQUFRLEVBQUUsSUFBSTtLQUNmO0lBQ0QsdUJBQXVCLEVBQUU7UUFDdkIsbUJBQW1CLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQy9DLGlCQUFpQixFQUFFLElBQUk7UUFDdkIsT0FBTyxFQUFFLElBQUk7UUFDYixRQUFRLEVBQUUsSUFBSTtLQUNmO0lBQ0QsYUFBYSxFQUFFO1FBQ2IsVUFBVSxFQUFFLElBQUk7UUFDaEIsU0FBUyxFQUFFLElBQUk7UUFDZixRQUFRLEVBQUUsSUFBSTtLQUNmO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBhY20gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNlcnRpZmljYXRlbWFuYWdlcic7XG5pbXBvcnQgKiBhcyBjbG91ZGZyb250IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZGZyb250JztcbmltcG9ydCAqIGFzIG9yaWdpbnMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQtb3JpZ2lucyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyByb3V0ZTUzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzJztcbmltcG9ydCAqIGFzIHBhdHRlcm5zIGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzLXBhdHRlcm5zJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXJvdXRlNTMtdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0ICogYXMgY3IgZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIFN0YXRpY1dlYnNpdGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdGF0aWNXZWJzaXRlUHJvcHMge1xuICAvKipcbiAgICogVGhlIGRvbWFpbiBuYW1lIGZvciB0aGlzIHN0YXRpYyB3ZWJzaXRlXG4gICAqXG4gICAqIEBleGFtcGxlIHd3dy5teS1zdGF0aWMtd2Vic2l0ZS5jb21cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGhvc3RlZCB6b25lIHdoZXJlIHJlY29yZHMgc2hvdWxkIGJlIGFkZGVkXG4gICAqL1xuICByZWFkb25seSBob3N0ZWRab25lOiByb3V0ZTUzLklIb3N0ZWRab25lO1xuXG4gIC8qKlxuICAgKiBBIGJhY2tlbmQgY29uZmlndXJhdGlvbiB0aGF0IHdpbGwgYmUgc2F2ZWQgYXMgYGNvbmZpZy5qc29uYFxuICAgKiBpbiB0aGUgUzMgYnVja2V0IG9mIHRoZSBzdGF0aWMgd2Vic2l0ZS5cbiAgICpcbiAgICogVGhlIGZyb250ZW5kIGNhbiBxdWVyeSB0aGlzIGNvbmZpZyBieSBkb2luZyBgZmV0Y2goJy9jb25maWcuanNvbicpYC5cbiAgICpcbiAgICogQGV4YW1wbGUgeyB1c2VyUG9vbElkOiAnMTIzNCcsIGFwaUVuZG9pbnQ6ICdodHRwczovL3d3dy5teS1hcGkuY29tL2FwaScgfVxuICAgKi9cbiAgcmVhZG9ubHkgYmFja2VuZENvbmZpZ3VyYXRpb24/OiBhbnk7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBkb21haW4gbmFtZXMgdGhhdCBzaG91bGQgcmVkaXJlY3QgdG8gYGRvbWFpbk5hbWVgXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRvbWFpbiBuYW1lIG9mIHRoZSBob3N0ZWQgem9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVkaXJlY3RzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFJlc3BvbnNlIGhlYWRlcnMgcG9saWN5IGZvciB0aGUgZGVmYXVsdCBiZWhhdmlvclxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHBvbGljeSBpcyBjcmVhdGVkIHdpdGggYmVzdCBwcmFjdGljZSBzZWN1cml0eSBoZWFkZXJzXG4gICAqL1xuICByZWFkb25seSByZXNwb25zZUhlYWRlcnNQb2xpY3k/OiBjbG91ZGZyb250LlJlc3BvbnNlSGVhZGVyc1BvbGljeTtcbn1cblxuLyoqXG4gKiBBIENsb3VkRnJvbnQgc3RhdGljIHdlYnNpdGUgaG9zdGVkIG9uIFMzXG4gKi9cbmV4cG9ydCBjbGFzcyBTdGF0aWNXZWJzaXRlIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgLyoqXG4gICogQmVzdCBwcmFjdGljZSBzZWN1cml0eSBoZWFkZXJzIHVzZWQgYXMgZGVmYXVsdFxuICAqL1xuICBwdWJsaWMgc3RhdGljIGRlZmF1bHRTZWN1cml0eUhlYWRlcnNCZWhhdmlvcjogY2xvdWRmcm9udC5SZXNwb25zZVNlY3VyaXR5SGVhZGVyc0JlaGF2aW9yID0ge1xuICAgIGNvbnRlbnRUeXBlT3B0aW9uczoge1xuICAgICAgb3ZlcnJpZGU6IHRydWUsXG4gICAgfSxcbiAgICBmcmFtZU9wdGlvbnM6IHtcbiAgICAgIGZyYW1lT3B0aW9uOiBjbG91ZGZyb250LkhlYWRlcnNGcmFtZU9wdGlvbi5ERU5ZLFxuICAgICAgb3ZlcnJpZGU6IHRydWUsXG4gICAgfSxcbiAgICByZWZlcnJlclBvbGljeToge1xuICAgICAgcmVmZXJyZXJQb2xpY3k6IGNsb3VkZnJvbnQuSGVhZGVyc1JlZmVycmVyUG9saWN5LlNUUklDVF9PUklHSU5fV0hFTl9DUk9TU19PUklHSU4sXG4gICAgICBvdmVycmlkZTogdHJ1ZSxcbiAgICB9LFxuICAgIHN0cmljdFRyYW5zcG9ydFNlY3VyaXR5OiB7XG4gICAgICBhY2Nlc3NDb250cm9sTWF4QWdlOiBEdXJhdGlvbi5zZWNvbmRzKDMxNTM2MDAwKSxcbiAgICAgIGluY2x1ZGVTdWJkb21haW5zOiB0cnVlLFxuICAgICAgcHJlbG9hZDogdHJ1ZSxcbiAgICAgIG92ZXJyaWRlOiB0cnVlLFxuICAgIH0sXG4gICAgeHNzUHJvdGVjdGlvbjoge1xuICAgICAgcHJvdGVjdGlvbjogdHJ1ZSxcbiAgICAgIG1vZGVCbG9jazogdHJ1ZSxcbiAgICAgIG92ZXJyaWRlOiB0cnVlLFxuICAgIH0sXG4gIH07XG5cbiAgLyoqXG4gICAqIFRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBvZiB0aGlzIHN0YXRpYyB3ZWJzaXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGlzdHJpYnV0aW9uOiBjbG91ZGZyb250LkRpc3RyaWJ1dGlvbjtcblxuICAvKipcbiAgICogVGhlIFMzIGJ1Y2tldCBvZiB0aGlzIHN0YXRpYyB3ZWJzaXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0OiBzMy5CdWNrZXQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFN0YXRpY1dlYnNpdGVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGhpcywgJ0J1Y2tldCcsIHtcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBzMy5CbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgfSk7XG5cbiAgICBjb25zdCBjZXJ0aWZpY2F0ZSA9IG5ldyBhY20uRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUodGhpcywgJ0NlcnRpZmljYXRlJywge1xuICAgICAgZG9tYWluTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICAgIGhvc3RlZFpvbmU6IHByb3BzLmhvc3RlZFpvbmUsXG4gICAgICByZWdpb246ICd1cy1lYXN0LTEnLFxuICAgIH0pO1xuXG4gICAgY29uc3Qgb3JpZ2luUmVxdWVzdCA9IG5ldyBjbG91ZGZyb250LmV4cGVyaW1lbnRhbC5FZGdlRnVuY3Rpb24odGhpcywgJ09yaWdpblJlcXVlc3QnLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ29yaWdpbi1yZXF1ZXN0LWhhbmRsZXInKSksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTRfWCxcbiAgICB9KTtcblxuICAgIHRoaXMuZGlzdHJpYnV0aW9uID0gbmV3IGNsb3VkZnJvbnQuRGlzdHJpYnV0aW9uKHRoaXMsICdEaXN0cmlidXRpb24nLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgICAgb3JpZ2luOiBuZXcgb3JpZ2lucy5TM09yaWdpbih0aGlzLmJ1Y2tldCksXG4gICAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBjbG91ZGZyb250LlZpZXdlclByb3RvY29sUG9saWN5LlJFRElSRUNUX1RPX0hUVFBTLFxuICAgICAgICBlZGdlTGFtYmRhczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGV2ZW50VHlwZTogY2xvdWRmcm9udC5MYW1iZGFFZGdlRXZlbnRUeXBlLk9SSUdJTl9SRVFVRVNULFxuICAgICAgICAgICAgZnVuY3Rpb25WZXJzaW9uOiBvcmlnaW5SZXF1ZXN0LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIHJlc3BvbnNlSGVhZGVyc1BvbGljeTogcHJvcHMucmVzcG9uc2VIZWFkZXJzUG9saWN5ID8/IG5ldyBjbG91ZGZyb250LlJlc3BvbnNlSGVhZGVyc1BvbGljeSh0aGlzLCAnUmVzcG9uc2VIZWFkZXJzUG9saWN5Jywge1xuICAgICAgICAgIHNlY3VyaXR5SGVhZGVyc0JlaGF2aW9yOiBTdGF0aWNXZWJzaXRlLmRlZmF1bHRTZWN1cml0eUhlYWRlcnNCZWhhdmlvcixcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgICAgZGVmYXVsdFJvb3RPYmplY3Q6ICdpbmRleC5odG1sJyxcbiAgICAgIGRvbWFpbk5hbWVzOiBbcHJvcHMuZG9tYWluTmFtZV0sXG4gICAgICBjZXJ0aWZpY2F0ZSxcbiAgICAgIG1pbmltdW1Qcm90b2NvbFZlcnNpb246IGNsb3VkZnJvbnQuU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFNfVjFfMl8yMDIxLFxuICAgIH0pO1xuXG4gICAgbmV3IHJvdXRlNTMuQVJlY29yZCh0aGlzLCAnQVJlY29yZCcsIHtcbiAgICAgIHJlY29yZE5hbWU6IHByb3BzLmRvbWFpbk5hbWUsXG4gICAgICB6b25lOiBwcm9wcy5ob3N0ZWRab25lLFxuICAgICAgdGFyZ2V0OiByb3V0ZTUzLlJlY29yZFRhcmdldC5mcm9tQWxpYXMobmV3IHRhcmdldHMuQ2xvdWRGcm9udFRhcmdldCh0aGlzLmRpc3RyaWJ1dGlvbikpLFxuICAgIH0pO1xuXG4gICAgbmV3IHJvdXRlNTMuQWFhYVJlY29yZCh0aGlzLCAnQWFhYVJlY29yZCcsIHtcbiAgICAgIHJlY29yZE5hbWU6IHByb3BzLmRvbWFpbk5hbWUsXG4gICAgICB6b25lOiBwcm9wcy5ob3N0ZWRab25lLFxuICAgICAgdGFyZ2V0OiByb3V0ZTUzLlJlY29yZFRhcmdldC5mcm9tQWxpYXMobmV3IHRhcmdldHMuQ2xvdWRGcm9udFRhcmdldCh0aGlzLmRpc3RyaWJ1dGlvbikpLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmJhY2tlbmRDb25maWd1cmF0aW9uKSB7XG4gICAgLy8gU2F2ZSBiYWNrZW5kIGNvbmZpZyB0byBidWNrZXQsIGNhbiBiZSBxdWVyaWVkIGJ5IHRoZSBmcm9udGVuZFxuICAgICAgbmV3IGNyLkF3c0N1c3RvbVJlc291cmNlKHRoaXMsICdQdXRDb25maWcnLCB7XG4gICAgICAgIHBvbGljeTogY3IuQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3kuZnJvbVN0YXRlbWVudHMoW1xuICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGFjdGlvbnM6IFsnczM6UHV0T2JqZWN0J10sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmJ1Y2tldC5hcm5Gb3JPYmplY3RzKCdjb25maWcuanNvbicpXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSksXG4gICAgICAgIG9uVXBkYXRlOiB7XG4gICAgICAgICAgc2VydmljZTogJ1MzJyxcbiAgICAgICAgICBhY3Rpb246ICdwdXRPYmplY3QnLFxuICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgIEJ1Y2tldDogdGhpcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICAgIEtleTogJ2NvbmZpZy5qc29uJyxcbiAgICAgICAgICAgIEJvZHk6IFN0YWNrLm9mKHRoaXMpLnRvSnNvblN0cmluZyhwcm9wcy5iYWNrZW5kQ29uZmlndXJhdGlvbiksXG4gICAgICAgICAgICBDb250ZW50VHlwZTogJ2FwcGxpY2F0aW9uL2pzb24nLFxuICAgICAgICAgICAgQ2FjaGVDb250cm9sOiAnbWF4LWFnZT0wLCBuby1jYWNoZSwgbm8tc3RvcmUsIG11c3QtcmV2YWxpZGF0ZScsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwaHlzaWNhbFJlc291cmNlSWQ6IGNyLlBoeXNpY2FsUmVzb3VyY2VJZC5vZignY29uZmlnJyksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBpZiAoIXByb3BzLnJlZGlyZWN0cyB8fCBwcm9wcy5yZWRpcmVjdHMubGVuZ3RoICE9PSAwKSB7XG4gICAgICBjb25zdCBodHRwc1JlZGlyZWN0ID0gbmV3IHBhdHRlcm5zLkh0dHBzUmVkaXJlY3QodGhpcywgJ0h0dHBzUmVkaXJlY3QnLCB7XG4gICAgICAgIHRhcmdldERvbWFpbjogcHJvcHMuZG9tYWluTmFtZSxcbiAgICAgICAgem9uZTogcHJvcHMuaG9zdGVkWm9uZSxcbiAgICAgICAgcmVjb3JkTmFtZXM6IHByb3BzLnJlZGlyZWN0cyxcbiAgICAgIH0pO1xuICAgICAgLy8gRm9yY2UgbWluaW11bSBwcm90b2NvbCB2ZXJzaW9uXG4gICAgICBjb25zdCByZWRpcmVjdERpc3RyaWJ1dGlvbiA9IGh0dHBzUmVkaXJlY3Qubm9kZS50cnlGaW5kQ2hpbGQoJ1JlZGlyZWN0RGlzdHJpYnV0aW9uJykgYXMgY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uO1xuICAgICAgY29uc3QgY2ZuRGlzdHJpYnV0aW9uID0gcmVkaXJlY3REaXN0cmlidXRpb24ubm9kZS50cnlGaW5kQ2hpbGQoJ0NGRGlzdHJpYnV0aW9uJykgYXMgY2xvdWRmcm9udC5DZm5EaXN0cmlidXRpb247XG4gICAgICBjZm5EaXN0cmlidXRpb24uYWRkUHJvcGVydHlPdmVycmlkZSgnRGlzdHJpYnV0aW9uQ29uZmlnLlZpZXdlckNlcnRpZmljYXRlLk1pbmltdW1Qcm90b2NvbFZlcnNpb24nLCAnVExTdjEuMl8yMDIxJyk7XG4gICAgfVxuICB9XG59XG4iXX0=