import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import * as events from 'aws-cdk-lib/aws-events';
import { Construct } from 'constructs';
/**
 * Properties for a CodeCommitMirror
 */
export interface CodeCommitMirrorProps {
    /**
     * The source repository
     */
    readonly repository: CodeCommitMirrorSourceRepository;
    /**
     * The ECS cluster where to run the mirroring operation
     */
    readonly cluster: ecs.ICluster;
    /**
     * The schedule for the mirroring operation
     *
     * @default - everyday at midnight
     */
    readonly schedule?: events.Schedule;
    /**
     * Where to run the mirroring Fargate tasks
     *
     * @default - public subnets
     */
    readonly subnetSelection?: ec2.SubnetSelection;
}
/**
 * A source repository for AWS CodeCommit mirroring
 */
export declare abstract class CodeCommitMirrorSourceRepository {
    /**
     * Public GitHub repository
     */
    static gitHub(owner: string, name: string): CodeCommitMirrorSourceRepository;
    /**
     * Private repository with HTTPS clone URL stored in a AWS Secrets Manager secret or
     * a AWS Systems Manager secure string parameter.
     *
     * @param name the repository name
     * @param url the secret containing the HTTPS clone URL
     */
    static private(name: string, url: ecs.Secret): CodeCommitMirrorSourceRepository;
    /**
     * The name of the repository
     */
    abstract readonly name: string;
    /** The HTTPS clone URL in plain text, used for a public repository */
    abstract readonly plainTextUrl?: string;
    /**
     * The HTTPS clone URL if the repository is private.
     *
     * The secret should contain the username and/or token.
     *
     * @example
     * `https://TOKEN@github.com/owner/name`
     * `https://USERNAME:TOKEN@bitbucket.org/owner/name.git`
     */
    abstract readonly secretUrl?: ecs.Secret;
}
/**
 * Mirror a repository to AWS CodeCommit on schedule
 */
export declare class CodeCommitMirror extends Construct {
    constructor(scope: Construct, id: string, props: CodeCommitMirrorProps);
}
