"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegendPosition = exports.Color = exports.Shading = exports.SingleValueWidget = exports.GraphWidget = exports.GraphWidgetView = exports.AlarmWidget = void 0;
const cdk = require("@aws-cdk/core");
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line.
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    /**
     *
     */
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    /**
     * Return the widget JSON for use in the dashboard.
     */
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * Types of view.
 */
var GraphWidgetView;
(function (GraphWidgetView) {
    GraphWidgetView["TIME_SERIES"] = "timeSeries";
    GraphWidgetView["BAR"] = "bar";
    GraphWidgetView["PIE"] = "pie";
})(GraphWidgetView = exports.GraphWidgetView || (exports.GraphWidgetView = {}));
/**
 * A dashboard widget that displays metrics.
 */
class GraphWidget extends widget_1.ConcreteWidget {
    /**
     *
     */
    constructor(props) {
        var _a, _b;
        super(props.width || 6, props.height || 6);
        this.props = props;
        this.leftMetrics = (_a = props.left) !== null && _a !== void 0 ? _a : [];
        this.rightMetrics = (_b = props.right) !== null && _b !== void 0 ? _b : [];
    }
    /**
     * Add another metric to the left Y axis of the GraphWidget.
     *
     * @param metric the metric to add.
     */
    addLeftMetric(metric) {
        this.leftMetrics.push(metric);
    }
    /**
     * Add another metric to the right Y axis of the GraphWidget.
     *
     * @param metric the metric to add.
     */
    addRightMetric(metric) {
        this.rightMetrics.push(metric);
    }
    /**
     * Return the widget JSON for use in the dashboard.
     */
    toJson() {
        var _a;
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.leftMetrics, this.rightMetrics);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: (_a = this.props.view) !== null && _a !== void 0 ? _a : GraphWidgetView.TIME_SERIES,
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                    liveData: this.props.liveData,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric.
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    /**
     *
     */
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    /**
     * Return the widget JSON for use in the dashboard.
     */
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation.
 */
var Shading;
(function (Shading) {
    Shading["NONE"] = "none";
    Shading["ABOVE"] = "above";
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/**
 * blue - hex #1f77b4.
 */
Color.BLUE = '#1f77b4';
/**
 * brown - hex #8c564b.
 */
Color.BROWN = '#8c564b';
/**
 * green - hex #2ca02c.
 */
Color.GREEN = '#2ca02c';
/**
 * grey - hex #7f7f7f.
 */
Color.GREY = '#7f7f7f';
/**
 * orange - hex #ff7f0e.
 */
Color.ORANGE = '#ff7f0e';
/**
 * pink - hex #e377c2.
 */
Color.PINK = '#e377c2';
/**
 * purple - hex #9467bd.
 */
Color.PURPLE = '#9467bd';
/**
 * red - hex #d62728.
 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 */
var LegendPosition;
(function (LegendPosition) {
    LegendPosition["BOTTOM"] = "bottom";
    LegendPosition["RIGHT"] = "right";
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhcGguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJncmFwaC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBcUM7QUFHckMsbURBQTBEO0FBQzFELHFDQUEwQzs7OztBQXlGMUMsTUFBYSxXQUFZLFNBQVEsdUJBQWM7Ozs7SUFHN0MsWUFBWSxLQUF1QjtRQUNqQyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztJQUNyQixDQUFDOzs7O0lBRU0sTUFBTTtRQUNYLE9BQU8sQ0FBQztnQkFDTixJQUFJLEVBQUUsUUFBUTtnQkFDZCxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7Z0JBQ2pCLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDbkIsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO2dCQUNULENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztnQkFDVCxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFLFlBQVk7b0JBQ2xCLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUs7b0JBQ3ZCLE1BQU0sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU07b0JBQzNDLFdBQVcsRUFBRTt3QkFDWCxNQUFNLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUM7cUJBQ3BDO29CQUNELEtBQUssRUFBRTt3QkFDTCxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUztxQkFDNUU7aUJBQ0Y7YUFDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUE1QkQsa0NBNEJDOzs7O0FBS0QsSUFBWSxlQWFYO0FBYkQsV0FBWSxlQUFlO0lBSXpCLDZDQUEwQixDQUFBO0lBSTFCLDhCQUFXLENBQUE7SUFJWCw4QkFBVyxDQUFBO0FBQ2IsQ0FBQyxFQWJXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBYTFCOzs7O0FBaUZELE1BQWEsV0FBWSxTQUFRLHVCQUFjOzs7O0lBTzdDLFlBQVksS0FBdUI7O1FBQ2pDLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxJQUFJLENBQUMsRUFBRSxLQUFLLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLElBQUksQ0FBQyxXQUFXLFNBQUcsS0FBSyxDQUFDLElBQUksbUNBQUksRUFBRSxDQUFDO1FBQ3BDLElBQUksQ0FBQyxZQUFZLFNBQUcsS0FBSyxDQUFDLEtBQUssbUNBQUksRUFBRSxDQUFDO0lBQ3hDLENBQUM7Ozs7OztJQU9NLGFBQWEsQ0FBQyxNQUFlO1FBQ2xDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ2hDLENBQUM7Ozs7OztJQU9NLGNBQWMsQ0FBQyxNQUFlO1FBQ25DLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ2pDLENBQUM7Ozs7SUFFTSxNQUFNOztRQUNYLE1BQU0scUJBQXFCLEdBQUc7WUFDNUIsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDaEUsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLElBQUksRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUNuRSxDQUFDO1FBRUYsTUFBTSxPQUFPLEdBQUcsK0JBQW1CLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDekUsT0FBTyxDQUFDO2dCQUNOLElBQUksRUFBRSxRQUFRO2dCQUNkLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztnQkFDakIsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNuQixDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBQ1QsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO2dCQUNULFVBQVUsRUFBRTtvQkFDVixJQUFJLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLG1DQUFJLGVBQWUsQ0FBQyxXQUFXO29CQUNwRCxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLO29CQUN2QixNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNO29CQUMzQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPO29CQUMzQixPQUFPLEVBQUUsT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUztvQkFDakQsV0FBVyxFQUFFLHFCQUFxQixDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsVUFBVSxFQUFFLHFCQUFxQixFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7b0JBQ2pHLEtBQUssRUFBRTt3QkFDTCxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUzt3QkFDM0UsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFNBQVM7cUJBQy9FO29CQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7b0JBQ3JHLFFBQVEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVE7aUJBQzlCO2FBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBN0RELGtDQTZEQzs7OztBQXNCRCxNQUFhLGlCQUFrQixTQUFRLHVCQUFjOzs7O0lBR25ELFlBQVksS0FBNkI7UUFDdkMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLElBQUksQ0FBQyxFQUFFLEtBQUssQ0FBQyxNQUFNLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDckIsQ0FBQzs7OztJQUVNLE1BQU07UUFDWCxPQUFPLENBQUM7Z0JBQ04sSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLO2dCQUNqQixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07Z0JBQ25CLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztnQkFDVCxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxhQUFhO29CQUNuQixLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLO29CQUN2QixNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNO29CQUMzQyxPQUFPLEVBQUUsK0JBQW1CLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO29CQUNwRCxvQkFBb0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQjtpQkFDdEQ7YUFDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUF4QkQsOENBd0JDOzs7O0FBNENELElBQVksT0FlWDtBQWZELFdBQVksT0FBTztJQUlqQix3QkFBYSxDQUFBO0lBS2IsMEJBQWUsQ0FBQTtJQUtmLDBCQUFlLENBQUE7QUFDakIsQ0FBQyxFQWZXLE9BQU8sR0FBUCxlQUFPLEtBQVAsZUFBTyxRQWVsQjs7OztBQUtELE1BQWEsS0FBSzs7QUFBbEIsc0JBd0JDOzs7O0FBdEJ3QixVQUFJLEdBQUcsU0FBUyxDQUFDOzs7O0FBR2pCLFdBQUssR0FBRyxTQUFTLENBQUM7Ozs7QUFHbEIsV0FBSyxHQUFHLFNBQVMsQ0FBQzs7OztBQUdsQixVQUFJLEdBQUcsU0FBUyxDQUFDOzs7O0FBR2pCLFlBQU0sR0FBRyxTQUFTLENBQUM7Ozs7QUFHbkIsVUFBSSxHQUFHLFNBQVMsQ0FBQzs7OztBQUdqQixZQUFNLEdBQUcsU0FBUyxDQUFDOzs7O0FBR25CLFNBQUcsR0FBRyxTQUFTLENBQUM7Ozs7QUFNekMsSUFBWSxjQWVYO0FBZkQsV0FBWSxjQUFjO0lBSXhCLG1DQUFpQixDQUFBO0lBS2pCLGlDQUFlLENBQUE7SUFLZixtQ0FBaUIsQ0FBQTtBQUNuQixDQUFDLEVBZlcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFlekI7QUFFRCxTQUFTLGFBQWEsQ0FBQyxLQUFhO0lBQ2xDLE9BQU8sQ0FBQyxDQUF1QixFQUFFLEVBQUU7UUFDakMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDO0lBQ3pCLENBQUMsQ0FBQztBQUNKLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBJQWxhcm0gfSBmcm9tICcuL2FsYXJtLWJhc2UnO1xuaW1wb3J0IHsgSU1ldHJpYyB9IGZyb20gJy4vbWV0cmljLXR5cGVzJztcbmltcG9ydCB7IGFsbE1ldHJpY3NHcmFwaEpzb24gfSBmcm9tICcuL3ByaXZhdGUvcmVuZGVyaW5nJztcbmltcG9ydCB7IENvbmNyZXRlV2lkZ2V0IH0gZnJvbSAnLi93aWRnZXQnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTWV0cmljV2lkZ2V0UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGl0bGU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB3aWR0aD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaGVpZ2h0PzogbnVtYmVyO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFlBeGlzUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1pbj86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWF4PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsYWJlbD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2hvd1VuaXRzPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBbGFybVdpZGdldFByb3BzIGV4dGVuZHMgTWV0cmljV2lkZ2V0UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbGFybTogSUFsYXJtO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsZWZ0WUF4aXM/OiBZQXhpc1Byb3BzO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFsYXJtV2lkZ2V0IGV4dGVuZHMgQ29uY3JldGVXaWRnZXQge1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBBbGFybVdpZGdldFByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBBbGFybVdpZGdldFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMud2lkdGggfHwgNiwgcHJvcHMuaGVpZ2h0IHx8IDYpO1xuICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgfVxuXG4gIHB1YmxpYyB0b0pzb24oKTogYW55W10ge1xuICAgIHJldHVybiBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogdGhpcy53aWR0aCxcbiAgICAgIGhlaWdodDogdGhpcy5oZWlnaHQsXG4gICAgICB4OiB0aGlzLngsXG4gICAgICB5OiB0aGlzLnksXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHZpZXc6ICd0aW1lU2VyaWVzJyxcbiAgICAgICAgdGl0bGU6IHRoaXMucHJvcHMudGl0bGUsXG4gICAgICAgIHJlZ2lvbjogdGhpcy5wcm9wcy5yZWdpb24gfHwgY2RrLkF3cy5SRUdJT04sXG4gICAgICAgIGFubm90YXRpb25zOiB7XG4gICAgICAgICAgYWxhcm1zOiBbdGhpcy5wcm9wcy5hbGFybS5hbGFybUFybl0sXG4gICAgICAgIH0sXG4gICAgICAgIHlBeGlzOiB7XG4gICAgICAgICAgbGVmdDogdGhpcy5wcm9wcy5sZWZ0WUF4aXMgIT09IHVuZGVmaW5lZCA/IHRoaXMucHJvcHMubGVmdFlBeGlzIDogdW5kZWZpbmVkLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9XTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIEdyYXBoV2lkZ2V0VmlldyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBUSU1FX1NFUklFUyA9ICd0aW1lU2VyaWVzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQkFSID0gJ2JhcicsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFBJRSA9ICdwaWUnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgR3JhcGhXaWRnZXRQcm9wcyBleHRlbmRzIE1ldHJpY1dpZGdldFByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbGVmdD86IElNZXRyaWNbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmlnaHQ/OiBJTWV0cmljW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsZWZ0QW5ub3RhdGlvbnM/OiBIb3Jpem9udGFsQW5ub3RhdGlvbltdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJpZ2h0QW5ub3RhdGlvbnM/OiBIb3Jpem9udGFsQW5ub3RhdGlvbltdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdGFja2VkPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxlZnRZQXhpcz86IFlBeGlzUHJvcHM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmlnaHRZQXhpcz86IFlBeGlzUHJvcHM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbGVnZW5kUG9zaXRpb24/OiBMZWdlbmRQb3NpdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxpdmVEYXRhPzogYm9vbGVhbjtcblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmlldz86IEdyYXBoV2lkZ2V0Vmlldztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgR3JhcGhXaWRnZXQgZXh0ZW5kcyBDb25jcmV0ZVdpZGdldCB7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczogR3JhcGhXaWRnZXRQcm9wcztcblxuICBwcml2YXRlIHJlYWRvbmx5IGxlZnRNZXRyaWNzOiBJTWV0cmljW107XG4gIHByaXZhdGUgcmVhZG9ubHkgcmlnaHRNZXRyaWNzOiBJTWV0cmljW107XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IEdyYXBoV2lkZ2V0UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcy53aWR0aCB8fCA2LCBwcm9wcy5oZWlnaHQgfHwgNik7XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgIHRoaXMubGVmdE1ldHJpY3MgPSBwcm9wcy5sZWZ0ID8/IFtdO1xuICAgIHRoaXMucmlnaHRNZXRyaWNzID0gcHJvcHMucmlnaHQgPz8gW107XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkTGVmdE1ldHJpYyhtZXRyaWM6IElNZXRyaWMpIHtcbiAgICB0aGlzLmxlZnRNZXRyaWNzLnB1c2gobWV0cmljKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkUmlnaHRNZXRyaWMobWV0cmljOiBJTWV0cmljKSB7XG4gICAgdGhpcy5yaWdodE1ldHJpY3MucHVzaChtZXRyaWMpO1xuICB9XG5cbiAgcHVibGljIHRvSnNvbigpOiBhbnlbXSB7XG4gICAgY29uc3QgaG9yaXpvbnRhbEFubm90YXRpb25zID0gW1xuICAgICAgLi4uKHRoaXMucHJvcHMubGVmdEFubm90YXRpb25zIHx8IFtdKS5tYXAobWFwQW5ub3RhdGlvbignbGVmdCcpKSxcbiAgICAgIC4uLih0aGlzLnByb3BzLnJpZ2h0QW5ub3RhdGlvbnMgfHwgW10pLm1hcChtYXBBbm5vdGF0aW9uKCdyaWdodCcpKSxcbiAgICBdO1xuXG4gICAgY29uc3QgbWV0cmljcyA9IGFsbE1ldHJpY3NHcmFwaEpzb24odGhpcy5sZWZ0TWV0cmljcywgdGhpcy5yaWdodE1ldHJpY3MpO1xuICAgIHJldHVybiBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogdGhpcy53aWR0aCxcbiAgICAgIGhlaWdodDogdGhpcy5oZWlnaHQsXG4gICAgICB4OiB0aGlzLngsXG4gICAgICB5OiB0aGlzLnksXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHZpZXc6IHRoaXMucHJvcHMudmlldyA/PyBHcmFwaFdpZGdldFZpZXcuVElNRV9TRVJJRVMsXG4gICAgICAgIHRpdGxlOiB0aGlzLnByb3BzLnRpdGxlLFxuICAgICAgICByZWdpb246IHRoaXMucHJvcHMucmVnaW9uIHx8IGNkay5Bd3MuUkVHSU9OLFxuICAgICAgICBzdGFja2VkOiB0aGlzLnByb3BzLnN0YWNrZWQsXG4gICAgICAgIG1ldHJpY3M6IG1ldHJpY3MubGVuZ3RoID4gMCA/IG1ldHJpY3MgOiB1bmRlZmluZWQsXG4gICAgICAgIGFubm90YXRpb25zOiBob3Jpem9udGFsQW5ub3RhdGlvbnMubGVuZ3RoID4gMCA/IHsgaG9yaXpvbnRhbDogaG9yaXpvbnRhbEFubm90YXRpb25zIH0gOiB1bmRlZmluZWQsXG4gICAgICAgIHlBeGlzOiB7XG4gICAgICAgICAgbGVmdDogdGhpcy5wcm9wcy5sZWZ0WUF4aXMgIT09IHVuZGVmaW5lZCA/IHRoaXMucHJvcHMubGVmdFlBeGlzIDogdW5kZWZpbmVkLFxuICAgICAgICAgIHJpZ2h0OiB0aGlzLnByb3BzLnJpZ2h0WUF4aXMgIT09IHVuZGVmaW5lZCA/IHRoaXMucHJvcHMucmlnaHRZQXhpcyA6IHVuZGVmaW5lZCxcbiAgICAgICAgfSxcbiAgICAgICAgbGVnZW5kOiB0aGlzLnByb3BzLmxlZ2VuZFBvc2l0aW9uICE9PSB1bmRlZmluZWQgPyB7IHBvc2l0aW9uOiB0aGlzLnByb3BzLmxlZ2VuZFBvc2l0aW9uIH0gOiB1bmRlZmluZWQsXG4gICAgICAgIGxpdmVEYXRhOiB0aGlzLnByb3BzLmxpdmVEYXRhLFxuICAgICAgfSxcbiAgICB9XTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU2luZ2xlVmFsdWVXaWRnZXRQcm9wcyBleHRlbmRzIE1ldHJpY1dpZGdldFByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1ldHJpY3M6IElNZXRyaWNbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2V0UGVyaW9kVG9UaW1lUmFuZ2U/OiBib29sZWFuO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgU2luZ2xlVmFsdWVXaWRnZXQgZXh0ZW5kcyBDb25jcmV0ZVdpZGdldCB7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFNpbmdsZVZhbHVlV2lkZ2V0UHJvcHM7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFNpbmdsZVZhbHVlV2lkZ2V0UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcy53aWR0aCB8fCA2LCBwcm9wcy5oZWlnaHQgfHwgMyk7XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICB9XG5cbiAgcHVibGljIHRvSnNvbigpOiBhbnlbXSB7XG4gICAgcmV0dXJuIFt7XG4gICAgICB0eXBlOiAnbWV0cmljJyxcbiAgICAgIHdpZHRoOiB0aGlzLndpZHRoLFxuICAgICAgaGVpZ2h0OiB0aGlzLmhlaWdodCxcbiAgICAgIHg6IHRoaXMueCxcbiAgICAgIHk6IHRoaXMueSxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3NpbmdsZVZhbHVlJyxcbiAgICAgICAgdGl0bGU6IHRoaXMucHJvcHMudGl0bGUsXG4gICAgICAgIHJlZ2lvbjogdGhpcy5wcm9wcy5yZWdpb24gfHwgY2RrLkF3cy5SRUdJT04sXG4gICAgICAgIG1ldHJpY3M6IGFsbE1ldHJpY3NHcmFwaEpzb24odGhpcy5wcm9wcy5tZXRyaWNzLCBbXSksXG4gICAgICAgIHNldFBlcmlvZFRvVGltZVJhbmdlOiB0aGlzLnByb3BzLnNldFBlcmlvZFRvVGltZVJhbmdlLFxuICAgICAgfSxcbiAgICB9XTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEhvcml6b250YWxBbm5vdGF0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZhbHVlOiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxhYmVsPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb2xvcj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmaWxsPzogU2hhZGluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmlzaWJsZT86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBTaGFkaW5nIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgTk9ORSA9ICdub25lJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBQk9WRSA9ICdhYm92ZScsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQkVMT1cgPSAnYmVsb3cnXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ29sb3Ige1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQkxVRSA9ICcjMWY3N2I0JztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEJST1dOID0gJyM4YzU2NGInO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgR1JFRU4gPSAnIzJjYTAyYyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEdSRVkgPSAnIzdmN2Y3Zic7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgT1JBTkdFID0gJyNmZjdmMGUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBQSU5LID0gJyNlMzc3YzInO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBVUlBMRSA9ICcjOTQ2N2JkJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSRUQgPSAnI2Q2MjcyOCc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIExlZ2VuZFBvc2l0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQk9UVE9NID0gJ2JvdHRvbScsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUklHSFQgPSAncmlnaHQnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIEhJRERFTiA9ICdoaWRkZW4nXG59XG5cbmZ1bmN0aW9uIG1hcEFubm90YXRpb24oeUF4aXM6IHN0cmluZyk6ICgoeDogSG9yaXpvbnRhbEFubm90YXRpb24pID0+IGFueSkge1xuICByZXR1cm4gKGE6IEhvcml6b250YWxBbm5vdGF0aW9uKSA9PiB7XG4gICAgcmV0dXJuIHsgLi4uYSwgeUF4aXMgfTtcbiAgfTtcbn0iXX0=