"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeAlarm = void 0;
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
/**
 * A Composite Alarm based on Alarm Rule.
 */
class CompositeAlarm extends alarm_base_1.AlarmBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: (_a = props.compositeAlarmName) !== null && _a !== void 0 ? _a : core_1.Lazy.string({ produce: () => this.generateUniqueId() }),
        });
        if (props.alarmRule.renderAlarmRule().length > 10240) {
            throw new Error('Alarm Rule expression cannot be greater than 10240 characters, please reduce the conditions in the Alarm Rule');
        }
        this.alarmRule = props.alarmRule.renderAlarmRule();
        const alarm = new cloudwatch_generated_1.CfnCompositeAlarm(this, 'Resource', {
            alarmName: this.physicalName,
            alarmRule: this.alarmRule,
            alarmDescription: props.alarmDescription,
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.list({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.list({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.list({ produce: () => this.okActionArns }),
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
    }
    /**
     * Import an existing CloudWatch composite alarm provided an Name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param compositeAlarmName Composite Alarm Name.
     */
    static fromCompositeAlarmName(scope, id, compositeAlarmName) {
        const stack = core_1.Stack.of(scope);
        return this.fromCompositeAlarmArn(scope, id, stack.formatArn({
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: compositeAlarmName,
        }));
    }
    /**
     * Import an existing CloudWatch composite alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param compositeAlarmArn Composite Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm/CompositeAlarmName).
     */
    static fromCompositeAlarmArn(scope, id, compositeAlarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = compositeAlarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(compositeAlarmArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = core_1.Names.uniqueId(this);
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.CompositeAlarm = CompositeAlarm;
//# sourceMappingURL=data:application/json;base64,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